/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <unistd.h>
#include <string.h>
#include <gdk/gdkkeysyms.h>

#include "config.h"
#include "google.h"
#include "gtk-utils.h"
#include "utils.h"
#include "kz-icons.h"
#include "kz-bookmark-menu.h"
#include "kz-actions.h"
#include "kz-entry.h"
#include "intl.h"

#define GOOGLE_URI _("http://www.google.com/search?ie=UTF-8&oe=utf8&q=%s")

static gchar       *google_entry_get_uri     (Google *google);
static const gchar *google_entry_get_keyword (Google *google);

/* callbacks */
static void     cb_google_entry_activate  (GtkEntry *widget, Google *google);
static gboolean cb_google_entry_key_press (GtkWidget *widget, GdkEventKey *event,
					   Google *google);

static void
cb_google_destroy (GtkWidget *hbox, Google *google)
{
	g_free(google);
}


/* create new google search item */
GtkWidget *
google_new (KzWindow *kz)
{
	GtkWidget *hbox;
	
	Google *google =NULL;
	google = g_new0(Google, 1);
	google->kz = kz;

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	g_signal_connect(G_OBJECT(hbox), "destroy",
			 G_CALLBACK(cb_google_destroy), google);
	
	/* create the keyword entry */
	google->entry = kz_entry_new();
	gtk_widget_set_size_request(google->entry, 120, -1);
	kz_entry_set_backtext(KZ_ENTRY(google->entry), _("Google Search"));
	kz_entry_set_icon_from_stock(KZ_ENTRY(google->entry),
				     KZ_STOCK_GOOGLE,
				     GTK_ICON_SIZE_MENU);
	gtk_box_pack_start(GTK_BOX(hbox), google->entry, TRUE, TRUE, 0);
	
	g_signal_connect(G_OBJECT(GTK_ENTRY(google->entry)), 
			 "activate",
			 G_CALLBACK(cb_google_entry_activate), google);
	g_signal_connect(G_OBJECT(google->entry),
			 "key-press-event",
			 G_CALLBACK(cb_google_entry_key_press), google);
	gtk_widget_show_all(hbox);

	g_object_set_data(G_OBJECT(hbox), "Google::Google", google);
	
	return hbox;
}

Google *
google_get (GtkWidget *widget)
{
	g_return_val_if_fail(G_IS_OBJECT(widget), NULL);
	return g_object_get_data(G_OBJECT(widget), "Google::Google");
}

static void
cb_google_entry_activate (GtkEntry *widget, Google *google)
{
	gchar *uri;
	gboolean new_tab;

	uri = google_entry_get_uri(google);

	KZ_CONF_GET("Global", "entry_open_in_new_tab", new_tab, BOOL);
	if (new_tab)
		kz_window_open_new_tab(google->kz, uri);
	else	
		kz_window_load_url(google->kz, uri);

	gtk_entry_set_text(widget, "");

	g_free(uri);
}


static gboolean
cb_google_entry_key_press (GtkWidget *widget, GdkEventKey *event,
			   Google *google)
{
	gboolean new_tab;

	if ((event->keyval == GDK_Return || event->keyval == GDK_ISO_Enter)
	    && (event->state & GDK_CONTROL_MASK))
	{
		gchar *uri;
		KZ_CONF_GET("Global", "entry_open_in_new_tab", new_tab, BOOL);

		uri = google_entry_get_uri(google);
		if (new_tab)
			kz_window_load_url(google->kz, uri);
		else 
			kz_window_open_new_tab(google->kz, uri);

		gtk_entry_set_text(GTK_ENTRY(widget), "");

		g_free(uri);
		return TRUE;
	}

	return FALSE;
}


static const gchar *
google_entry_get_keyword (Google *google)
{
	GtkEntry *entry = GTK_ENTRY(google->entry);
	const gchar *text= gtk_entry_get_text(entry);
	
	return text;
}


static gchar *
google_entry_get_uri (Google *google)
{
	gchar *text_encoded;
	gchar *search_uri;

	text_encoded = url_encode(google_entry_get_keyword(google));
	search_uri = g_strdup_printf(GOOGLE_URI, text_encoded);
	g_free(text_encoded);
	return search_uri;
}
