/***************************************************************************
 *   Copyright (C) 2004-2009 by Thomas Fischer                             *
 *   fischer@unix-ag.uni-kl.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <qlayout.h>
#include <qlabel.h>
#include <qspinbox.h>
#include <qbuffer.h>
#include <qfile.h>

#include <kpushbutton.h>
#include <klocale.h>
#include <kcombobox.h>
#include <kiconloader.h>
#include <kio/jobclasses.h>
#include <kio/job.h>
#include <kio/netaccess.h>

#include <settings.h>
#include <fileimporterris.h>
#include "webquerysciencedirect.h"

namespace KBibTeX
{
    WebQueryScienceDirectWidget::WebQueryScienceDirectWidget( QWidget *parent, const char *name )
            : WebQueryWidget( parent, name )
    {
        init();

        QString allValues;
        Settings *settings = Settings::self();
        QString value = settings->getWebQueryDefault( "ScienceDirect_title" );
        value = value == QString::null ? "" : value;
        lineEditQuery->setText( value );
        allValues += value;
        value = settings->getWebQueryDefault( "ScienceDirect_author" );
        value = value == QString::null ? "" : value;
        lineEditAuthor->setText( value );
        allValues += value;
        value = settings->getWebQueryDefault( "ScienceDirect_journal" );
        value = value == QString::null ? "" : value;
        lineEditJournal->setText( value );
        allValues += value;
        value = settings->getWebQueryDefault( "ScienceDirect_volume" );
        value = value == QString::null ? "" : value;
        lineEditVolume->setText( value );
        allValues += value;
        value = settings->getWebQueryDefault( "ScienceDirect_issue" );
        value = value == QString::null ? "" : value;
        lineEditIssue->setText( value );
        allValues += value;
        value = settings->getWebQueryDefault( "ScienceDirect_page" );
        value = value == QString::null ? "" : value;
        lineEditPage->setText( value );
        allValues += value;

        slotTextChanged( allValues, true );
    }

    void WebQueryScienceDirectWidget::init()
    {
        QVBoxLayout *vLayout = new QVBoxLayout( this, 0, KDialog::spacingHint() );

        QHBoxLayout *hLayout = new QHBoxLayout( );
        vLayout->addLayout( hLayout );
        KPushButton *clearSearchText = new KPushButton( this );
        clearSearchText->setIconSet( QIconSet( SmallIcon( "locationbar_erase" ) ) );
        hLayout->addWidget( clearSearchText );
        QLabel *label = new QLabel( i18n( "Title, abstract, keywords:" ), this );
        hLayout->addWidget( label );
        lineEditQuery = new KLineEdit( this );
        KCompletion *completionQuery = lineEditQuery->completionObject();
        hLayout->addWidget( lineEditQuery );
        label->setBuddy( lineEditQuery );
        connect( clearSearchText, SIGNAL( clicked() ), lineEditQuery, SLOT( clear() ) );
        connect( lineEditQuery, SIGNAL( textChanged( const QString& ) ), this, SLOT( slotTextChangedSD( ) ) );
        hLayout->setStretchFactor( lineEditQuery, 7 );
        connect( lineEditQuery, SIGNAL( returnPressed() ), this, SIGNAL( startSearch() ) );
        connect( lineEditQuery, SIGNAL( returnPressed( const QString& ) ), completionQuery, SLOT( addItem( const QString& ) ) );


        hLayout = new QHBoxLayout( );
        vLayout->addLayout( hLayout );

        KPushButton *clearAuthor = new KPushButton( this );
        clearAuthor->setIconSet( QIconSet( SmallIcon( "locationbar_erase" ) ) );
        hLayout->addWidget( clearAuthor );
        label = new QLabel( i18n( "Author:" ), this );
        hLayout->addWidget( label );
        lineEditAuthor = new KLineEdit( this );
        completionQuery = lineEditAuthor->completionObject();
        hLayout->addWidget( lineEditAuthor );
        label->setBuddy( lineEditAuthor );
        connect( clearAuthor, SIGNAL( clicked() ), lineEditAuthor, SLOT( clear() ) );
        connect( lineEditAuthor, SIGNAL( textChanged( const QString& ) ), this, SLOT( slotTextChangedSD( ) ) );
        hLayout->setStretchFactor( lineEditAuthor, 7 );
        connect( lineEditAuthor, SIGNAL( returnPressed() ), this, SIGNAL( startSearch() ) );
        connect( lineEditAuthor, SIGNAL( returnPressed( const QString& ) ), completionQuery, SLOT( addItem( const QString& ) ) );

        hLayout->addSpacing( KDialog::spacingHint() * 2 );

        KPushButton *clearJournal = new KPushButton( this );
        clearJournal->setIconSet( QIconSet( SmallIcon( "locationbar_erase" ) ) );
        hLayout->addWidget( clearJournal );
        label = new QLabel( i18n( "Journal/book title:" ), this );
        hLayout->addWidget( label );
        lineEditJournal = new KLineEdit( this );
        completionQuery = lineEditJournal->completionObject();
        hLayout->addWidget( lineEditJournal );
        label->setBuddy( lineEditJournal );
        connect( clearJournal, SIGNAL( clicked() ), lineEditJournal, SLOT( clear() ) );
        connect( lineEditJournal, SIGNAL( textChanged( const QString& ) ), this, SLOT( slotTextChangedSD( ) ) );
        hLayout->setStretchFactor( lineEditJournal, 4 );
        connect( lineEditJournal, SIGNAL( returnPressed() ), this, SIGNAL( startSearch() ) );
        connect( lineEditJournal, SIGNAL( returnPressed( const QString& ) ), completionQuery, SLOT( addItem( const QString& ) ) );

        hLayout = new QHBoxLayout( );
        vLayout->addLayout( hLayout );

        label = new QLabel( i18n( "Volume:" ), this );
        hLayout->addWidget( label );
        lineEditVolume = new KLineEdit( this );
        hLayout->addWidget( lineEditVolume );
        hLayout->setStretchFactor( lineEditVolume, 1 );
        label->setBuddy( lineEditVolume );
        connect( lineEditVolume, SIGNAL( returnPressed() ), this, SIGNAL( startSearch() ) );

        hLayout->addSpacing( KDialog::spacingHint() * 2 );

        label = new QLabel( i18n( "Issue:" ), this );
        hLayout->addWidget( label );
        lineEditIssue = new KLineEdit( this );
        hLayout->addWidget( lineEditIssue );
        label->setBuddy( lineEditIssue );
        hLayout->setStretchFactor( lineEditIssue, 1 );
        connect( lineEditIssue, SIGNAL( returnPressed() ), this, SIGNAL( startSearch() ) );

        hLayout->addSpacing( KDialog::spacingHint() * 2 );

        label = new QLabel( i18n( "Page:" ), this );
        hLayout->addWidget( label );
        lineEditPage = new KLineEdit( this );
        hLayout->addWidget( lineEditPage );
        hLayout->setStretchFactor( lineEditPage, 1 );
        label->setBuddy( lineEditPage );
        connect( lineEditPage, SIGNAL( returnPressed() ), this, SIGNAL( startSearch() ) );

        hLayout->addSpacing( KDialog::spacingHint() * 2 );

        label = new QLabel( i18n( "&Number of results:" ), this );
        hLayout->addWidget( label );
        spinBoxMaxHits = new QSpinBox( 1, 100, 1, this );
        spinBoxMaxHits->setValue( 10 );
        hLayout->addWidget( spinBoxMaxHits );
        hLayout->setStretchFactor( spinBoxMaxHits, 1 );
        label->setBuddy( spinBoxMaxHits );
        hLayout->addStretch( 5 );
    }

    void WebQueryScienceDirectWidget::slotTextChangedSD()
    {
        QString text = lineEditQuery->text().stripWhiteSpace().append( lineEditAuthor->text().stripWhiteSpace() ).append( lineEditJournal->text().stripWhiteSpace() ).replace( '$', "" );
        emit enableSearch( !text.isEmpty() );
    }

    WebQueryScienceDirect::WebQueryScienceDirect( QWidget* parent ): WebQuery( parent )
    {
        m_importer = new BibTeX::FileImporterBibTeX( FALSE );
        m_importer->setIgnoreComments( TRUE );
        m_widget = new WebQueryScienceDirectWidget( parent );
    }

    WebQueryScienceDirect::~WebQueryScienceDirect()
    {
        delete m_widget;
        delete m_importer;
    }

    QString WebQueryScienceDirect::title()
    {
        return i18n( "ScienceDirect" );
    }

    QString WebQueryScienceDirect::disclaimer()
    {
        return i18n( "About ScienceDirect" );
    }

    QString WebQueryScienceDirect::disclaimerURL()
    {
        return "http://info.sciencedirect.com/";
    }

    WebQueryWidget *WebQueryScienceDirect::widget()
    {
        return m_widget;
    }

    void WebQueryScienceDirect::query()
    {
        WebQuery::query();

        Settings *settings = Settings::self();
        settings->setWebQueryDefault( "ScienceDirect_author", m_widget->lineEditAuthor->text() );
        settings->setWebQueryDefault( "ScienceDirect_journal", m_widget->lineEditJournal->text() );
        settings->setWebQueryDefault( "ScienceDirect_volume", m_widget->lineEditVolume->text() );
        settings->setWebQueryDefault( "ScienceDirect_issue", m_widget->lineEditIssue->text() );
        settings->setWebQueryDefault( "ScienceDirect_page", m_widget->lineEditPage->text() );

        QString tak = m_widget->lineEditQuery->text().stripWhiteSpace().replace( '$', "" ).replace( "%", "%25" ).replace( "+", "%2B" ).replace( " ", "%20" ).replace( "#", "%23" ).replace( "&", "%26" ).replace( "?", "%3F" );
        QString author = m_widget->lineEditAuthor->text().stripWhiteSpace().replace( '$', "" ).replace( "%", "%25" ).replace( "+", "%2B" ).replace( " ", "%20" ).replace( "#", "%23" ).replace( "&", "%26" ).replace( "?", "%3F" );
        QString journal = m_widget->lineEditJournal->text().stripWhiteSpace().replace( '$', "" ).replace( "%", "%25" ).replace( "+", "%2B" ).replace( " ", "%20" ).replace( "#", "%23" ).replace( "&", "%26" ).replace( "?", "%3F" );
        QString volume = m_widget->lineEditVolume->text().stripWhiteSpace().replace( '$', "" ).replace( "%", "%25" ).replace( "+", "%2B" ).replace( " ", "%20" ).replace( "#", "%23" ).replace( "&", "%26" ).replace( "?", "%3F" );
        QString issue = m_widget->lineEditIssue->text().stripWhiteSpace().replace( '$', "" ).replace( "%", "%25" ).replace( "+", "%2B" ).replace( " ", "%20" ).replace( "#", "%23" ).replace( "&", "%26" ).replace( "?", "%3F" );
        QString page = m_widget->lineEditPage->text().stripWhiteSpace().replace( '$', "" ).replace( "%", "%25" ).replace( "+", "%2B" ).replace( " ", "%20" ).replace( "#", "%23" ).replace( "&", "%26" ).replace( "?", "%3F" );

        if ( tak.isEmpty() && author.isEmpty() && journal.isEmpty() )
        {
            setEndSearch( WebQuery::statusInvalidQuery );
            return;
        }

        setNumStages( 4 );

        if ( !getStartPage() ) return;

        if ( !getResultPage( tak, author, journal, volume, issue, page ) ) return;

        if ( !getArticleListPage() ) return;

        if ( !getRISFile() ) return;
    }

    bool  WebQueryScienceDirect::getStartPage()
    {
        QString startPage = downloadHTML( KURL( "http://www.sciencedirect.com/" ) );
        if ( m_aborted )
        {
            setEndSearch( WebQuery::statusAborted );
            return false;
        }
        else if ( startPage == QString::null )
        {
            setEndSearch( WebQuery::statusError );
            return false;
        }

        int pos = startPage.find( "<input type=\"hidden\" name=\"_acct\" value=\"" );
        if ( pos < 0 )
        {
            setEndSearch( WebQuery::statusError );
            return false;
        }
        m_account = startPage.mid( pos + 41, startPage.find( "\"", pos + 43 ) - pos - 41 );
        pos = startPage.find( "<input type=\"hidden\" name=\"md5\" value=\"" );
        if ( pos < 0 )
        {
            setEndSearch( WebQuery::statusError );
            return false;
        }
        m_md5 = startPage.mid( pos + 39, startPage.find( "\"", pos + 41 ) - pos - 39 );

        return true;
    }

    bool WebQueryScienceDirect::getResultPage( const QString &tak, const QString &author, const QString &journal, const QString &volume, const QString &issue, const QString &page )
    {
        KURL url = KURL( QString( "http://www.sciencedirect.com/science?_ob=QuickSearchURL&_method=submitForm&qs_tak=" ).append( tak ).append( "&qs_author=" ).append( author ).append( "&qs_title=" ).append( journal ).append( "&qs_vol=" ).append( volume ).append( "&qs_issue=" ).append( issue ).append( "&qs_pages=" ).append( page ).append( "&_acct=" ).append( m_account ).append( "&md5=" ).append( m_md5 ).append( "&x=0&y=0&=Submit" ) );
        QString resultPage = downloadHTML( url );
        if ( m_aborted )
        {
            setEndSearch( WebQuery::statusAborted );
            return false;
        }
        else if ( resultPage == QString::null )
        {
            setEndSearch( WebQuery::statusError );
            return false;
        }

        int pos = resultPage.find( "<input type=\"hidden\" name=\"_ArticleListID\" value=" );
        if ( pos < 0 )
        {
            setEndSearch( WebQuery::statusError );
            return false;
        }
        m_articleListID = resultPage.mid( pos + 49, resultPage.find( ">", pos + 50 ) - pos - 49 );
        pos = resultPage.find( "<input type=\"hidden\" name=\"md5\" value=\"" );
        pos = resultPage.find( "<input type=\"hidden\" name=\"md5\" value=\"" , pos + 2 );
        if ( pos < 0 )
        {
            setEndSearch( WebQuery::statusError );
            return false;
        }
        m_md5 = resultPage.mid( pos + 39, 32 );
        pos = resultPage.find( "_userid=" );
        if ( pos < 0 )
        {
            setEndSearch( WebQuery::statusError );
            return false;
        }
        m_userid = resultPage.mid( pos + 8, resultPage.find( "&", pos + 9 ) - pos - 8 );

        return true;
    }

    bool WebQueryScienceDirect::getArticleListPage()
    {
        KURL url = KURL( QString( "http://www.sciencedirect.com/science?_ob=ArticleListURL&_method=tag&refSource=search&_st=13&_chunk=0&NEXT_LIST=1&view=c&md5=%1&_ArticleListID=%2&sisr_search=&sisrterm=&export=Export+Citations&count=%3" ).arg( m_md5 ).arg( m_articleListID ).arg( m_widget->spinBoxMaxHits->value() ) );
        QString articleListPage = downloadHTML( url );
        if ( m_aborted )
        {
            setEndSearch( WebQuery::statusAborted );
            return false;
        }
        else if ( articleListPage == QString::null )
        {
            setEndSearch( WebQuery::statusError );
            return false;
        }
        else if ( articleListPage.find( "subscription does not entitle" ) >= 0 )
        {
            qWarning( "Your subscription does not entitle you to access the download feature of ScienceDirect" );
            setEndSearch( WebQuery::statusInsufficientPermissions );
            return false;
        }

        int pos = articleListPage.find( "<input type=hidden name=md5 value=" );
        if ( pos < 0 )
        {
            setEndSearch( WebQuery::statusError );
            return false;
        }
        m_md5 = articleListPage.mid( pos + 34, articleListPage.find( ">", pos + 36 ) - pos - 34 );
        pos = articleListPage.find( "<input type=hidden name=_ArticleListID value=" );
        if ( pos < 0 )
        {
            setEndSearch( WebQuery::statusError );
            return false;
        }
        m_articleListID = articleListPage.mid( pos + 45, articleListPage.find( ">", pos + 46 ) - pos - 45 );

        return true;
    }

    bool WebQueryScienceDirect::getRISFile()
    {
        m_incomingData = "";
        QString data = QString( "_ob=DownloadURL&_method=finish&_acct=%1&_userid=%2&_ArticleListID=%3&count=10&md5=%4&JAVASCRIPT_ON=&format=cite&citation-type=RIS&RETURN_URL=http://www.sciencedirect.com/science/home" ).arg( m_account ).arg( m_userid ).arg( m_articleListID ).arg( m_md5 );
        KURL url = KURL( "http://www.sciencedirect.com/science" );
        KIO::TransferJob *job = KIO::http_post( url, data.utf8(), false );
        job->addMetaData( "content-type", "Content-Type: application/x-www-form-urlencoded" );
        connect( job, SIGNAL( data( KIO::Job *, const QByteArray & ) ), this, SLOT( slotData( KIO::Job *, const QByteArray & ) ) );
        connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotResult( KIO::Job * ) ) );

        return true;
    }

    void WebQueryScienceDirect::slotData( KIO::Job *, const QByteArray &data )
    {
        if ( data.size() > 0 )
            m_incomingData.append( QCString( data, data.size() + 1 ) );
    }

    void WebQueryScienceDirect::slotResult( KIO::Job *job )
    {
        if ( job->error() )
        {
            setEndSearch( WebQuery::statusError );
            return;
        }

        BibTeX::FileImporterRIS importer;
        QBuffer buffer;

        buffer.open( IO_WriteOnly );
        QTextStream ts( &buffer );
        ts.setEncoding( QTextStream::UnicodeUTF8 );
        ts << m_incomingData << endl;
        buffer.close();

        buffer.open( IO_ReadOnly );
        BibTeX::File *tmpBibFile = importer.load( &buffer );
        buffer.close();

        if ( tmpBibFile != NULL )
        {
            int count = m_widget->spinBoxMaxHits->value();
            for ( BibTeX::File::ElementList::iterator it = tmpBibFile->begin(); count > 0 && it != tmpBibFile->end(); ++it )
            {
                BibTeX::Entry *entry = dynamic_cast<BibTeX::Entry*>( *it );
                if ( entry != NULL )
                {
                    emit foundEntry( new BibTeX::Entry( entry ), false );
                    --count;
                }
            }

            delete tmpBibFile;
        }
        else
        {
            setEndSearch( WebQuery::statusError );
            return;
        }

        setEndSearch( WebQuery::statusSuccess );
    }
}
#include "webquerysciencedirect.moc"
