/***************************************************************************
 *   Copyright (C) 2004 by Roberto Virga                                   *
 *   rvirga@users.sf.net                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <kglobal.h>
#include <kiconloader.h>
#include <klibloader.h>
#include <ktrader.h>

#include <kbsboincmonitor.h>
#include <kbshostnode.h>
#include <kbspanelnode.h>
 
#include "kbsprojectnode.h"

KBSProjectNode::KBSProjectNode(const QString& project, KBSTreeNode *parent, const char *name)
              : KBSTreeNode(parent, name),
                m_suspended(false), m_extinguished(false), m_project(project)
{
  setupMonitor();
  
  addPlugins();
}

unsigned KBSProjectNode::type() const
{
  return 2;
}

QString KBSProjectNode::name() const
{
  return KBSProjectNode::name(m_project, m_monitor);
}

QStringList KBSProjectNode::icons() const
{
  return QStringList(m_suspended ? "project_suspend" : "project");
}

QString KBSProjectNode::project() const
{
  return m_project;
}

bool KBSProjectNode::isSuspended() const
{
  return m_suspended;
}

bool KBSProjectNode::isExtinguished() const
{
  return m_extinguished;
}

QString KBSProjectNode::name(const QString &project, KBSBOINCMonitor *monitor)
{
  if(NULL == monitor) return project;
  const BOINCClientState *state = monitor->state();
  if(NULL == state) return project;
  
  const QString project_name = state->project[project].project_name;
  return project_name.isEmpty() ? project : project_name;
}

KBSBOINCMonitor *KBSProjectNode::monitor()
{
  return m_monitor;
}

void KBSProjectNode::update()
{
  const BOINCClientState *state = m_monitor->state();
  if(NULL == state) return;
  
  const BOINCProject *project = &state->project[m_project];
  
  const bool suspended = project->suspended_via_gui,
             extinguished = project->dont_request_more_work;
  
  bool changed = false;
  
  if(m_suspended != suspended) {
    m_suspended = suspended;
    changed = true;
  }
  if(m_extinguished != extinguished) {
    m_extinguished = extinguished;
    changed = true;
  }
  
  if(changed) emit nodeChanged(this);
} 

void KBSProjectNode::setupMonitor()
{
  KBSHostNode *host = static_cast<KBSHostNode*>(findAncestor("KBSHostNode"));
  m_monitor = (NULL != host) ? host->monitor() : NULL;
  if(NULL == m_monitor) return;
  
  connect(m_monitor, SIGNAL(stateUpdated()), this, SLOT(update()));
  
  const BOINCClientState *state = m_monitor->state();
  if(NULL == state) return;
  
  const BOINCProject *project = &state->project[m_project];
  
  m_suspended = project->suspended_via_gui;
  m_extinguished = project->dont_request_more_work;
}

void KBSProjectNode::addPlugins()
{
  QString constraints;
  KTrader::OfferList offers;

  constraints = "([X-KDE-Target] == 'Project') and ('%1' in [X-KDE-Projects])";
  offers = KTrader::self()->query("KBSPanelNode", constraints.arg(m_project));

  for(KTrader::OfferListIterator offer = offers.begin(); offer != offers.end(); ++offer)
  {
    QStringList args = (*offer)->property("X-KDE-Arguments").toStringList();
    args.prepend(m_project);
    
    KLibFactory *factory = KLibLoader::self()->factory((*offer)->library());
    if(NULL == factory) continue;
    
    insertChild(static_cast<KBSPanelNode*>(factory->create(this, 0, "KBSPanelNode", args)));
  }
}

#include "kbsprojectnode.moc"
