/***************************************************************************
 *   Copyright (C) 2004 by Roberto Virga                                   *
 *   rvirga@users.sf.net                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qlayout.h>
#include <qpopupmenu.h>
#include <qprogressbar.h>
#include <qptrlist.h>
#include <qstringlist.h>
#include <qstyle.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kglobal.h>
#include <klocale.h>
#include <kmenubar.h>
#include <ktoolbar.h>

#include <kbsboincmonitor.h>
#include <kbsrpcmonitor.h>

#include "kbstransferview.h"

KBSTransferView::KBSTransferView(KBSBOINCMonitor *monitor, QWidget *parent, const char *name, WFlags f)
               : KMainWindow(parent, name, f),
                 m_view(new KListView(this)), m_monitor(monitor), m_selection(NULL)
{
  KBSRPCMonitor *rpcMonitor = monitor->rpcMonitor();
  
  connect(rpcMonitor, SIGNAL(transfersUpdated()), this, SLOT(updateTransfers()));
  connect(rpcMonitor, SIGNAL(updated()), this, SLOT(updateState()));
  
  const QString host = rpcMonitor->host();
  
  setCaption(i18n("%1 - Transfers").arg(host));
  
  setCentralWidget(m_view);
  setupView();
  
  setAutoSaveGeometry(QString("%1 - Transfers").arg(host));
  
  setupActions();
  
  updateTransfers();
}

KBSBOINCMonitor *KBSTransferView::monitor()
{
  return m_monitor;
}

void KBSTransferView::setAutoSaveGeometry(const QString &group)
{
  m_group = group;
  
  if(!m_group.isEmpty()) readGeometry(m_group);
}

void KBSTransferView::updateTransfers()
{
  const QMap<QString,BOINCFileTransfer> transfers = m_monitor->rpcMonitor()->fileTransfers()->file_transfer;
  
  QPtrList<QListViewItem> deleted;
  QStringList added = transfers.keys();
  for(QListViewItem *item = m_view->firstChild(); NULL != item; item = item->nextSibling())
  {
    const QString name = item->text(1);
    
    if(added.contains(name)) {
      static_cast<KBSTransferItem*>(item)->setTransfer(transfers[name]);
      added.remove(name);
    } else
      deleted.append(item);
  }
  for(QStringList::const_iterator name = added.begin(); name != added.end(); ++name)
    (new KBSTransferItem(m_view))->setTransfer(transfers[*name]);
  for(QListViewItem *item = deleted.first(); item != NULL; item = deleted.next())
    m_view->takeItem(item);

  updateState();
}

void KBSTransferView::updateState()
{
  KBSRPCMonitor *rpcMonitor = m_monitor->rpcMonitor();
  const bool suspended = (rpcMonitor->runMode() == RunNever)
                         || (rpcMonitor->networkMode() == ConnectNever);

  for(QListViewItem *item = m_view->firstChild(); NULL != item; item = item->nextSibling())
    static_cast<KBSTransferItem*>(item)->setSuspended(suspended);
}

void KBSTransferView::transferRetry()
{
  if(NULL == m_selection) return;
  
  m_monitor->rpcMonitor()->retryFileTransfer(m_selection->transfer().project_url,
                                             m_selection->transfer().name);
  m_selection = NULL;
}

void KBSTransferView::transferAbort()
{
  if(NULL == m_selection) return;
  
  m_monitor->rpcMonitor()->abortFileTransfer(m_selection->transfer().project_url,
                                             m_selection->transfer().name);
  m_selection = NULL;
}

void KBSTransferView::slotContextMenu(KListView *view, QListViewItem *selection, const QPoint &pos)
{
  m_selection = NULL;
  
  if(view != m_view) return;
  m_selection = static_cast<KBSTransferItem*>(selection);
  
  QPopupMenu *context = static_cast<QPopupMenu*>(guiFactory()->container("context", this));
  context->popup(pos);
}

void KBSTransferView::setupView()
{
  m_view->addColumn(i18n("Project"));
  m_view->setColumnAlignment(0, AlignLeft);
  m_view->addColumn(i18n("File"));  
  m_view->setColumnAlignment(1, AlignLeft);
  m_view->addColumn(i18n("Progress"));
  m_view->setColumnAlignment(2, AlignCenter);
  m_view->addColumn(i18n("Size"));
  m_view->setColumnAlignment(2, AlignCenter);
  m_view->addColumn(i18n("Time"));
  m_view->setColumnAlignment(2, AlignCenter);
  m_view->addColumn(i18n("Speed"));
  m_view->setColumnAlignment(2, AlignCenter);
  m_view->addColumn(i18n("Status"));
  m_view->setColumnAlignment(2, AlignCenter);
  
  m_view->setSelectionMode(QListView::NoSelection);
  
  m_view->setShowSortIndicator(true);
  m_view->setSorting(1);
  m_view->sort();
  
  connect(m_view, SIGNAL(contextMenu(KListView *, QListViewItem *, const QPoint &)),
          this, SLOT(slotContextMenu(KListView *, QListViewItem *, const QPoint &))); 
}

void KBSTransferView::setupActions()
{
  setStandardToolBarMenuEnabled(false);
  
  KStdAction::close(this, SLOT(close()), actionCollection())->setText(i18n("Close &Window"));
  
  transfer_retry = new KAction(i18n("&Retry File Transfer"), 0, this, SLOT(transferRetry()),
                                    actionCollection(), "transfer_retry");
  
  transfer_abort = new KAction(i18n("&Abort File Transfer"), 0, this, SLOT(transferAbort()),
                                    actionCollection(), "transfer_abort");
  
  createGUI("kbstransferui.rc", false);
  
  menuBar()->hide();
  toolBar()->hide();
}

void KBSTransferView::readGeometry(const QString &group)
{
  KConfig *config = kapp->config();
  config->setGroup(group);
  
  const int screen = kapp->desktop()->screenNumber(parentWidget());
  const QRect desk = kapp->desktop()->screenGeometry(screen);
  
  QRect geometry;
  
  geometry.setTop(config->readNumEntry(QString("Top %1 %2").arg(desk.top()).arg(desk.height()), -1));
  if(geometry.top() < 0) return;
  
  geometry.setLeft(config->readNumEntry(QString("Left %1 %2").arg(desk.left()).arg(desk.width()), -1));
  if(geometry.left() < 0) return;
  
  geometry.setHeight(config->readNumEntry(QString("Height %1").arg(desk.height()), 0));
  if(geometry.height() <= 0) return;
  
  geometry.setWidth(config->readNumEntry(QString("Width %1").arg(desk.width()), -1));
  if(geometry.width() < 0) return;
  
  setGeometry(geometry);
}

void KBSTransferView::writeGeometry(const QString &group)
{
  KConfig *config = kapp->config();
  config->setGroup(group);
  
  const int screen = kapp->desktop()->screenNumber(parentWidget());
  const QRect desk = kapp->desktop()->screenGeometry(screen);
  
  config->writeEntry(QString("Top %1 %2").arg(desk.top()).arg(desk.height()), geometry().top());
  config->writeEntry(QString("Left %1 %2").arg(desk.left()).arg(desk.width()), geometry().left());
  config->writeEntry(QString("Height %1").arg(desk.height()), geometry().height());
  config->writeEntry(QString("Width %1").arg(desk.width()), geometry().width());
}

KBSTransferView::KBSTransferItem::KBSTransferItem(QListView *parent)
                                : KListViewItem(parent), m_suspended(false)
{
}

const BOINCFileTransfer &KBSTransferView::KBSTransferItem::transfer() const
{
  return m_transfer;
}

void KBSTransferView::KBSTransferItem::setTransfer(const BOINCFileTransfer &transfer)
{
  m_transfer = transfer;
  
  if(text(0).isNull()) setText(0, transfer.project_name);
  
  if(text(1).isNull()) setText(1, transfer.name);
  
  KLocale *locale = KGlobal::locale();

  double ratio = (transfer.nbytes > 0) ? transfer.file_xfer.bytes_xferred / transfer.nbytes : 0;
  setText(2, locale->formatNumber(1e2 * ratio, 0));
  
  setText(3, i18n("%1/%2").arg(locale->formatNumber(transfer.file_xfer.bytes_xferred, 0))
                          .arg(locale->formatNumber(transfer.nbytes, 0)));
  
  setText(4, formatTime(transfer.persistent_file_xfer.time_so_far));
  
  setText(5, i18n("%1 KBps").arg(locale->formatNumber(transfer.file_xfer.xfer_speed)));

  setText(6, status());
}

bool KBSTransferView::KBSTransferItem::isSuspended() const
{
  return m_suspended;
}

void KBSTransferView::KBSTransferItem::setSuspended(bool suspended)
{
  m_suspended = suspended;

  setText(6, status());
}

QString KBSTransferView::KBSTransferItem::status() const
{
  if(m_suspended) return i18n("Suspended");
  
  switch(m_transfer.status) {
    case 0:
      break;
    case -114:
      return i18n("Download failed");
    case -115:
      return i18n("Upload failed");
    default:
      return i18n("Error");
  }
  
  if(m_transfer.generated_locally)
    return i18n("Uploading");
  else
    return i18n("Downloading");
}

void KBSTransferView::KBSTransferItem::paintCell(QPainter *p, const QColorGroup &cg,
                                                  int column, int width, int align)
{
  if(column != 2) return KListViewItem::paintCell(p, cg, column, width, align);
  
  QStyle::SFlags sflags = QStyle::Style_Default;
  if(listView()->isEnabled()) sflags |= QStyle::Style_Enabled;
  if(listView()->hasFocus()) sflags |= QStyle::Style_HasFocus;

  QRect r(1, 1, width - 2, height() - 2);
  
  QProgressBar progress;
  progress.setProgress(text(2).toInt(0, 10));
  progress.setGeometry(r);
  
  listView()->style().drawControl(QStyle::CE_ProgressBarContents, p, &progress, r, cg, sflags);
  listView()->style().drawControl(QStyle::CE_ProgressBarLabel, p, &progress, r, cg, sflags);
}

QString KBSTransferView::KBSTransferItem::key(int column, bool ascending) const
{
  switch(column) {
    case 0:
      return m_transfer.project_name;
    case 1:
      return m_transfer.name;
    case 2:
      {
        double ratio = 0.0;
        if(m_transfer.nbytes > 0)
          ratio = m_transfer.file_xfer.bytes_xferred / m_transfer.nbytes;
        
        return QString().sprintf("%05.lf", 1e5 * ratio);
      }
    case 3:
      return QString().sprintf("%012.0lf", m_transfer.file_xfer.bytes_xferred);
    case 4:
      return QString().sprintf("%09.lf", 1e2 * m_transfer.persistent_file_xfer.time_so_far);
    case 5:
      return QString().sprintf("%012.0lf", 1e2 * m_transfer.file_xfer.xfer_speed);
    default:
      return QListViewItem::key(column, ascending);
  }
}

#include "kbstransferview.moc"

