/***************************************************************************
                          cddb.cpp  -  description
                             -------------------
    begin                : Thu May 17 2001
    copyright            : (C) 2001 by Pascal 'PeP' Panneels
    email                : pepouille@skynet.be
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cddb.h"
#include "version.h"

DiscDBEntry::DiscDBEntry(QString s1, QString s2, QString s3)
{
	Category = s1;
	DiscID = s2;
	DiscTitle = s3;
}
			
DiscDBEntry::DiscDBEntry( const DiscDBEntry &p )
{
	Category = p.Category;
	DiscID = p .DiscID;
	DiscTitle = p.DiscTitle;
}

void DiscDBEntry::Clear()
{
	Category = QString("");
	DiscID = QString("");
	DiscTitle = QString("");
}

DiscContent::DiscContent()
: Head(QString(""), QString(""), QString(""))
{
}

DiscContent::DiscContent( const DiscContent &p )
{
	Head = p.Head;
	Artist = p.Artist;
	for ( unsigned int i=0; i<p.Tracks.size(); i++ )
		Tracks.push_back( p.Tracks[i] );
}
		
DiscContent::~DiscContent()
{
	Tracks.erase( Tracks.begin(), Tracks.end() );
}

void DiscContent::Clear()
{
	Head.Clear();
	Artist = QString("");
	Tracks.erase( Tracks.begin(), Tracks.end() );	
}

CDDB::CDDB( CDDB_ProtocolType proto ) : QObject()
{
	Protocol 	= proto;
	Connected 	= false;	// not yet connected to cddb server
	CDDBServer 	= QString("freedb.freedb.org");
	CDDBPort	= 888;
	Command		= new char[256];
}

CDDB::~CDDB()
{
	if ( Connected )
		quit();
	delete [] Command;
}
	
/** set the server name/port to use */
void CDDB::setServerPort(QString server, unsigned short int port )
{
	CDDBServer 	= server;
	CDDBPort 	= port;
	sprintf(msg, "Set CDDB server to %s (%d).", server.latin1(), port );
	emit statusText((char *)msg);
}	
	
/** intial client-server handshake */
CDDB_Answer CDDB::hello()
{
	char hostname[128], username[128];
	hostname[0]=0;
	username[0]=0;
	gethostname(hostname, sizeof(hostname));
    strcpy(username, getenv("LOGNAME"));
	// send the hello command
	sprintf(Command, "cddb hello %s %s KCDLabel %s", username, hostname, __KCDL_VERSION_CDDB);
	thesock.writeBlock(Command, strlen(Command));
	sprintf(msg, "Saying hello to %s", CDDBServer.latin1() );
	emit statusText((char *)msg);
	// get the answer
	char answer[128];
	// get the answer
	thesock.readBlock(answer, sizeof(answer)-1);
	sprintf(msg, "%s: %s", CDDBServer.latin1(), answer );
	emit statusText((char *)msg);	
	return getCDDBResponseCode(answer);
}

/** quit (disconnect) of the server */
void CDDB::quit()
{
	sprintf(Command, "cddb quit");
	sprintf(msg, "Quitting %s", CDDBServer.latin1() );
	emit statusText( (char *)msg );
	thesock.writeBlock( Command, strlen(Command) );
	thesock.disconnect();
	Connected = false;
}

/** connect the socket to access the cddb server */
bool CDDB::connect()
{
	char welcome[256];
	sprintf(msg, "Connecting to %s (%d)", CDDBServer.latin1(), CDDBPort );
	emit statusText( (char *)msg );	
	int res = thesock.connectToHost( CDDBServer, CDDBPort );
	Connected = (res==StateType(CONNECTED));
	if (Connected)
		{
		thesock.readBlock( welcome, sizeof(welcome) );
		sprintf(msg, "%s: %s", CDDBServer.latin1(), welcome);
		emit statusText( (char *)msg );	
		}
	return Connected;
}

/** query the server for a specific CD, with the offsets of each tracks, and the CD length in seconds*/
CDDB_QueryReturnType CDDB::query( QString DiscID, unsigned int nrtracks, std::vector<int> trackoffsets, int nsecs )
{
	QString Tracks="";
	char offset[10];
	CDDB_Answer rep;
	char line[256];
	char * ptr, *newline, *ptr2;
	QString tmp, s1, s2, s3;
	
	// are we already connected to the server ?
	if ( !Connected )
		{
		connect();
		if ( !Connected )
			{
			sprintf(msg, "Cannot connect to %s (%d)", CDDBServer.latin1(), CDDBPort );
			emit statusText(msg);
			return CDDB_QueryReturnType( CONNECT_ERROR );
			}
		rep = hello();
		if ( rep!=CDDB_Answer( C20x ) && rep!=CDDB_Answer( C40x ) )
			{
			sprintf(msg, "Error saying hello to %s (%d)", CDDBServer.latin1(), CDDBPort );
			emit statusText(msg);
			return CDDB_QueryReturnType( CONNECT_ERROR );
			}
		}
	// build the query string
	for ( unsigned int i=0;i<nrtracks;i++ )
		{
		sprintf(offset, "%d ", trackoffsets[i]);
		Tracks = Tracks+QString(offset);
		}
	emit statusText((char *)"Posting a query to the server for your CD.");
	sprintf(Command, "cddb query %s %d %s %d", DiscID.latin1(), nrtracks, Tracks.latin1(), nsecs );
	// send the query command to the cddb server
	thesock.writeBlock(Command, strlen(Command));
	// get the answer
	thesock.readBlock(line, sizeof(line));
	rep = getCDDBResponseCode(line);
	switch ( ReturnedCode )
		{
		case 200 :	// exact match --> fill in the received Disc ID + category + disc title
			emit statusText((char *)"Get an exact matching from the server!");
			newline = strchr(/*ptr+9*/line, '\r');	// remove trailing \n
			if (newline)
				*newline=0;
			ptr = strtok(line, " ");	// the code
			ptr = strtok(NULL, " ");	// the category
			CDContent.Head.Category = QString(ptr);
			ptr = strtok(NULL, " ");	// the discid
			CDContent.Head.DiscID = QString(ptr);
			ptr2 = strtok(ptr+9, "/");
			// split the artist name from the disc title
			if ( ptr2 )	// we found a separator --> Disc Tile / Artist
				{
				CDContent.Head.DiscTitle = QString(ptr2+strlen(ptr2)+1);
				CDContent.Artist = QString(ptr2);	
				}
			else			// we didn't find any separator --> Disc Title == Artist
				{
				CDContent.Head.DiscTitle = QString(ptr+9);
				CDContent.Artist = QString(ptr+9);	
				}
			
			// read the cd content from the cddb database server...
			read( CDContent.Head.Category, CDContent.Head.DiscID );
			// quit
			//quit();
			return CDDB_QueryReturnType( EXACT_MATCH );
		case 210 : 	// mutliple exact matches			
		case 211 :	// fuzzy matches
			emit statusText((char *)"Get inexact matchings from the server!");
			CloseMatches.erase( CloseMatches.begin(), CloseMatches.end() );
			do
				{
				thesock.readBlock( line, sizeof(line) );
				tmp = QString(line);
				if ( tmp[0]=='.' )		// end of transmission
					break;
				else
					{
					ptr = strtok(line, " ");	// category
					s1 = QString(ptr);
					ptr = strtok(NULL, " ");	// discid
					s2 = QString(ptr);
					//ptr = strtok(NULL, " ");	// disc title
					ptr+=s2.length()+1;
					newline = strchr( ptr, '\r');
					if ( newline )
						*newline = 0;
					s3 = QString( ptr );
					CloseMatches.push_back( DiscDBEntry(s1, s2, s3) );
					}				
				} while (true);
			//quit();
			return CDDB_QueryReturnType( INEXACT_MATCH );
			
		default :
			emit statusText((char *)"Get no matching from the server!");
			//quit();
			return CDDB_QueryReturnType( NO_MATCH );
		}
}

/** return the response code from the cddb server to a request */
CDDB_Answer CDDB::getCDDBResponseCode(char * answer)
{
	// store the code in an integer
	char ccode[4];
	strncpy(ccode, answer,3);
	ccode[3]=0;
	ReturnedCode = atoi(ccode);
	
	// decode the answer cCDDBMatchCDDlgode to get a generic form
	switch ( answer[0] )
		{
		case '1' :
			return CDDB_Answer( C1xx );
		case '2' :
			switch ( answer[1] )
				{
				case '0' :
					return CDDB_Answer( C20x );
				case '1' :
					return CDDB_Answer( C21x );
				case '2' :
					return CDDB_Answer( C22x );
				case '3' :
					return CDDB_Answer( C23x );
				default :
					return CDDB_Answer( UNKNOWN_2 );
				}
		case '3' :
			return CDDB_Answer( C3xx );
		case '4' :
			switch ( answer[1] )
				{
				case '0' :
					return CDDB_Answer( C40x );
				case '1' :
					return CDDB_Answer( C41x );
				case '3' :
					return CDDB_Answer( C43x );
				default :
					return CDDB_Answer( UNKNOWN_4 );
				}
		case '5' :
			switch ( answer[1] )
				{
				case '0' :
					return CDDB_Answer( C50x );
				case '1' :
					return CDDB_Answer( C51x );
				case '3' :
					return CDDB_Answer( C53x );
				default :
					return CDDB_Answer( UNKNOWN_5 );
				}
		default :
			return CDDB_Answer( UNKNOWN );
		}	
}

/** read a cd content	 */
CDDB_Answer CDDB::read(QString cat, QString discid)
{
	char line[256];
	QString s;
	char * ptr, *ptr2;
	
	// build the command
	emit statusText((char *)"Posting a read message to the server.");
	sprintf( Command, "cddb read %s %s", cat.latin1(), discid.latin1() );
	thesock.writeBlock( Command, strlen(Command) );
	// read and decode the answer
	thesock.readBlock( line, sizeof(line) );
	CDDB_Answer rep = getCDDBResponseCode(line);
	if ( ReturnedCode==210 )	// ok, CDDB entry will follow
		{
		do
			{
			thesock.readBlock( line, sizeof(line) );
			s = QString(line);
			if (line[0] == '.' )		// end of transmission
				break;
			else
				{
				if ( line[0]=='#' )		// this is a comment line --> skip it
					continue;
				else
					{
					char *newline = strchr(line, '\r');	// remove trailing \n
					if (newline)
						*newline=0;

					if ( strstr(line, "DISCID") )
						continue;
					if ( strstr(line, "DTITLE") )
						{
						ptr = strchr(line, '=' )+1;
						ptr2 = strtok(ptr, "/");
						// split the artist name from the disc title
						if ( ptr2 )	// we found a separator --> Disc Tile / Artist
							{
							QString tmp = QString( ptr2+strlen(ptr2)+1 ).stripWhiteSpace();
							CDContent.Head.DiscTitle = tmp;//string(ptr2+strlen(ptr2)+1);
							tmp = QString( ptr2 ).stripWhiteSpace();
							CDContent.Artist = tmp;	
							}
						else			// we didn't find any separator --> Disc Title == Artist
							{
							CDContent.Head.DiscTitle = QString(ptr);
							CDContent.Artist = QString(ptr);	
							}
						}
					else
						{
						if ( strstr(line, "TTITLE") )
							{
							ptr = strchr(line,'=')+1;
							ptr2 = strchr(ptr, '\r');
							if (ptr2)
								*ptr2=0;
							CDContent.Tracks.push_back(QString(ptr));
							}
						}
					}
				}
			} while (true);
		}
	
	return rep;
}

/** return the disc content */
DiscContent & CDDB::getDiscContent()
{
	return CDContent;
}

/** return the matched entries; */
MatchedEntries & CDDB::getMatchedEntries()
{
	return CloseMatches;
}
