/***************************************************************************
 *   Copyright (C) 2004, 2005 by Johnathan Burchill                        *
 *   jkerrb@users.sourceforge.net                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifndef _CONTROLLER_H_
#define _CONTROLLER_H_

#include <qobject.h>

#include <dar/libdar.hpp>

class kdar;
class KDarConfig;

class QListViewItem;
class QSemaphore;

/**
 * This is the abstract interface to the KDar controller class. It controls
 * the libdar operations based on user inputs, and is responsible for
 * passing results on to the main
 * view.
 *
 * @short KDar Controller Interface
 * @author Johnathan Burchill, <jkerrb@users.sourceforge.net>
 * @version VERSION
 */

class controller : public QObject
{

public:

    enum threadType
    {
        CREATE_THREAD,
        DIFF_THREAD,
        ISOLATE_THREAD,
        LIST_THREAD,
        OPEN_THREAD,
        POPULATETREE_THREAD,
        RESTORE_THREAD,
        TEST_THREAD
    };

    virtual ~controller() {};

    /**
     *  setParent sets the parent widget
     */
    virtual void setParent( void * ) = 0;
    /**
     *  closeArchive is a controller method to close an archive
     *  in anticipation of opening a new archive.
     */
    virtual void closeArchive() = 0;
    /**
     * Closes the logFile.
     */
    virtual void closeLogFile() = 0;
    /**
     *  createArchive is a controller method to create an archive
     *  from filesystem data.
     */
    virtual void createArchive( KDarConfig *, bool updateListView = true ) = 0;
    /**
     *  diffArchive is a controller method to compare an archive
     *  against the filesystem.
     */
    virtual void diffArchive( KDarConfig * ) = 0;
    /**
     *  isolateArchive is a controller method to isolate an archive
     *  catalog into its own file.
     */
    virtual void isolateArchive( KDarConfig * ) = 0;
    /**
     *  listArchive is a controller method to initiate listing the
     *  contents of an archive in the browser window (KListView).
     */
    virtual void listArchive( KDarConfig *, bool ) = 0;
    /**
     *  openArchive is a controller method to open an archive.
     */
    virtual void openArchive( KDarConfig *, bool updateListView = true ) = 0;
    /**
     * Opens the logFile for appending.
     */
    virtual void openLogFile( const QString & ) = 0;
    /**
     *  populateArchiveTree is a controller method to list the contents
     *  of a subdirectory in an archive. It needs to receive a QListViewItem
     */
    virtual void populateArchiveTree( KDarConfig *, QListViewItem * ) = 0;
    /**
     *  restoreArchive is a controller method to restore the contents
     *  of an archive to the filesystem.
     */
    virtual void restoreArchive( KDarConfig * ) = 0;
     /**
     *  testArchive is a controller method to test an archive.
     */
    virtual void testArchive( KDarConfig * ) = 0;
    /**
     *  waitForThread checks to see if the given thread type
     *  is running, and waits for it to finish.
     */
    virtual void waitForThread( threadType ) = 0;
    /**
     * Writes a message to the logfile.
     */
    virtual void writeLogMessage( QString & ) = 0;
    /**
     * Cancel a running thread. Use only with a threadsafe libdar.
     */
    virtual bool cancelOperation() = 0;
    /**
     * Opens the KDEWallet for storing and retrieving a password.
     */
    virtual bool openKDEWallet() = 0;
    /**
     * Obtain a password from a wallet or the user, and store it temporarily
     * in the controller. The stored password can be retrieved
     * with @ref password(). This password should be cleared from memory
     * with @ref clearPassword() once it is not required.
     */
    virtual bool obtainPassword() = 0;
    /**
     * Get the password that is temporarily stored in the controller.
     */
    virtual QString password() = 0;
    /**
     * Clear the password from memory.
     */
    virtual void clearPassword() = 0;
    /**
     * Change the password that is stored in the wallet
     *
     * Returns true if successful, false if not (e.g. user cancelled).
     */
    virtual bool changePassword() = 0;
    /**
     * Store a boolean response from user ( for KDarInteraction::pause() )
     */
    virtual void setUserResponse( const bool ) = 0;
    /**
     * Get the boolean response from the controller
     * ( for KDarInteraction::pause() )
     */
    virtual bool userResponse() = 0;
    /**
     * Returns the pointer to the controller's semaphore object.
     * Use this to put a thread to sleep, and to signal it to wake up.
     */
    virtual QSemaphore * semaphore() = 0;

    /**
     * Returns whether libdar supports extended attributes
     */
    virtual bool handlesEA() const = 0;
    /**
     * Returns whether libdar supports largefiles (>2GB)
     */
    virtual bool handlesLargefiles() const = 0;
    /**
     * Returns whether libdar supports ext2 nodump feature
     */
    virtual bool handlesNoDump() const = 0;
    /**
     * Returns whether libdar uses Special Memory Allocation scheme
     */
    virtual bool usesSpecialAlloc() const = 0;
    /**
     * Returns size of integer type
     */
    virtual libdar::U_I integerSizeBits() const = 0;
    /**
     * Returns whether libdar is thread-safe
     */
    virtual bool isThreadSafe() const = 0;
    /**
     * Returns whether libdar supports libz compression
     */
    virtual bool handlesZCompression() const = 0;
    /**
     * Returns whether libdar supports libbz2 compression
     */
    virtual bool handlesBZ2Compression() const = 0;
    /**
     * Returns whether libdar supports strong encryption
     */
    virtual bool handlesStrongCrypto() const = 0;

};

#endif // _CONTROLLER_H_
