/***************************************************************************
 *   Copyright (C) 2004, 2005 by Johnathan Burchill                              *
 *   jkerrb@users.sourceforge.net                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifndef _KDARCONFIG_H_
#define _KDARCONFIG_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "kdarConfigureDlg.h"
#include "kdarcompressionfilterdialog.h"
#include "kdardirectoryfilterdialog.h"
#include "kdarfilefilterdialog.h"

#include <qfont.h>
#include <qwidget.h>
#include <qcstring.h>
#include <qstring.h>
#include <qstringlist.h>
#include <string>

#include "kdarEnums.h"
#include <dar/libdar.hpp>


class KConfig;
class QListView;

/**
 * A configuration class/dialog for KDar.
 *
 * @short KDar configuration class.
 * @author Johnathan K. Burchill, jkerrb@users.sourceforge.net
 * @version 1.3.0
 */

class KDarConfig : public kdarConfigureDlg
{
    Q_OBJECT

public:

    /**
     * Constructor
     */
    KDarConfig( QWidget *parent=0, const char *name=0,
                    KDarConfig *kdarConfig=0, bool addHelpButton = false );

    /**
     * Copy constructor
     */
    KDarConfig( const KDarConfig & );
    /**
     * Destructor
     */
    virtual ~KDarConfig();
    /**
     * Assignment operator
     */
    void operator=( const KDarConfig &kdc );

    /**
     * Disables Z compression features
     * (i.e. libdar does not support Z compression)
     */
    void disableZCompression();
    /**
     * Disables BZ2 compression features
     * (i.e. libdar does not support BZ2 compression)
     */
    void disableBZ2Compression();
    /**
     * Disables the compression module
     */
    void disableCompressionModule();
    /**
     * Disables strong encryption features
     * (i.e. libdar does not support strong encryption)
     */
    void disableStrongCrypto();
    /**
     * Disables configure modules that are not needed for creation
     */
    void setupCreateConfig();
    /**
     * Disables configure modules that are not needed for restoration.
     */
    void setupRestoreConfig();
    /**
     * Hides the createArchiveWizard checkbox.
     */
    void hideCreateArchiveWizardCheckBox();
    /**
     * Hides the KDarSplash checkbox.
     */
    void hideKDarSplashCheckBox();

    /**
     * Options storing routines.
     */

    /**
     * This sets the global config object pointer. We need a custom global
     * config object when saving or restoring profiles.
     */
    void setGlobalConfig( KConfig * kconfig );

    /**
     * Set the archive name. This is the one you will open.
     */
    void setArchiveName( const QString & archiveName );
    /**
     * Set the default archive storage directory.
     */
    void setArchiveStorageDirectory( const QString & archiveStorageDirectory );
    /**
     * Set the new name for an archive about to be created.
     */
    void setNewArchiveName( const QString & newArchiveName );
    /**
     * Set the root (top-level) directory to backup.
     */
    void setDirectoryToBackup( const QString & directoryToBackup );
    /**
     * Set whether to perform a differential backup.
     */
    void setDifferentialBackup( const bool differentialBackup );
    /**
     * Set the reference archive name for the differential backup.
     */
    void setDifferentialBackupArchiveName( const QString &
                                differentialBackupArchiveName );
    /**
     * Set the input pipe string.
     */
    void setInputPipe( const QString & inputPipe );
    /**
     * Set the output pipe string.
     */
    void setOutputPipe( const QString & outputPipe );
    /**
     * Set the command to execute after creating a slice.
     */
    void setExecuteCommand( const QString & executeCommand );
    /**
     * Set the crypto block size.
     * Default is 10240 bytes.
     */
    void setCryptoBlockSize( const int cryptoBlockSize );
    /**
     * Set the cryptography algorithm
     */
    void setCryptoAlgorithm( const int algorithm );
    /**
     * Set whether to store password
     */
    void setStorePassword( bool storePassword );
    /**
     * Set the password string.
     */
    void setPasswordString( const QString & passwordString );
    /**
     * Set the command to execute before reading a slice.
     */
    void setExecuteRef( const QString & executeRef );
    /**
     *
     */
    void setPassRef( const QString &passRef );

    /**
     * Set whether to display messages during libdar operations.
     */
    void setVerbose( const bool verbose );
    /**
     * Set the log level
     */
    void setLogLevel( const int logLevel );
    /**
     * Set whether to allow file overwriting during restoration and archive
     * slice creation.
     */
    void setAllowOverwrite( const bool allowOverwrite );
    /**
     * Set whether to warn user of overwrites.
     */
    void setWarnOnOverwrite( const bool warnOnOverwrite );
    /**
     * Set whether to ignore User Extended Attributes.
     */
    void setUserEA( const bool userEA );
    /**
     * Set whether to ignore System Extended Attributes.
     */
    void setSystemEA( const bool systemEA );
    /**
     * Set whether to ignore user and group IDs.
     */
    void setIgnoreID( const bool ignoreID );
    /**
     * Set whether to not delete files during incremental restorations.
     */
    void setNoDelete( const bool noDelete );
    /**
     * Set whether to restore only files that are more recent than those
     * on disk.
     */
    void setRestoreRecent( const bool restoreRecent );
    /**
     * Set whether to restore all files to top-level restoration directory.
     */
    void setFlatRestore( const bool flatRestore );

    /**
     * Set whether to use the create archive Wizard.
     */
    void setUseCreateArchiveWizard( const bool useCreateArchiveWizard );
    /**
     * Set whether to split the archive during creation.
     */
    void setSplitArchive( const bool splitArchive );
    /**
     * Set whether to pause after a slice has been created.
     */
    void setPauseBetweenSlices( const bool pauseBetweenSlices );
    /**
     * Set whether to keep excluded directory names.
     */
    void setKeepPruned( const bool keepPruned );
    /**
     * Set whether to ignore the ext2 dump flag.
     */
    void setIgnoreDump( const bool ignoreDump );

    /**
     * Set the archive slice media type.
     */
    void setMedia( const int media );
    /**
     * Set the slice size number.
     */
    void setSliceSizeString( const QString & sliceSizeString );
    /**
     * Set the slice size suffix.
     */
    void setSliceSizeSuffix( const int sliceSizeSuffix );
    /**
     * Set the first slice size numbers as a QStringList.
     */
    void setFirstSliceSizeStrings( const QStringList & );
    /**
     * Set the first slice size number.
     */
    void setFirstSliceSizeString( int medium, const QString &
                                    firstSliceSizeString );
    /**
     * Set the first slice size suffices as a QValueList<int>.
     */
    void setFirstSliceSizeSuffices( const QValueList<int> & );
    /**
     * Set the first slice size suffix.
     */
    void setFirstSliceSizeSuffix( int medium, const int firstSliceSizeSuffix );
    /**
     * Set the compression algorithm (libdar::bzip2, libdar::gzip, libdar::none).
     */
    void setCompressionAlgorithm( const int compressionAlgorithm );
    /**
     * Set the gzip compression level.
     */
    void setCompressionLevel( const int compressionLevel );
    /**
     * Set the minimum file size for compression.
     */
    void setMinimumCompressionSizeInt( const int minimumCompressionSizeInt );
    /**
     * Set the hourshift value.
     */
    void setHourShiftInt( const int hourShiftInt );

    /**
     * Set the compression mask values.
     */
    void setCompressionMaskList( const QStringList & compressionMaskList );
    /**
     * Set the directory mask list.
     */
    void setDirectoryMaskList( const QStringList & directoryMaskList );
    /**
     * Set the file mask values.
     */
    void setFileMaskList( const QStringList & fileMaskList );
    /**
     * Set the compression type (KDar::INCLUDE, KDar::EXCLUDE).
     */
    void setCompressionMaskType( const int compressionMaskType );
    /**
     * Set the directory mask types (QStringList).
     */
    void setDirectoryMaskTypes( const QStringList & directoryMaskTypes );
    /**
     * Set the file mask type (KDar::INCLUDE, KDar::EXCLUDE).
     */
    void setFileMaskType( const int fileMaskType );

    /**
     * Set the general font.
     */
    void setGeneralFont( const QFont & generalFont );
    /**
     * Set the archive browser font.
     */
    void setFileBrowserFont( const QFont & fileBrowserFont );
    /**
     * Set the messages font.
     */
    void setMessageWindowFont( const QFont & messageWindowFont );
    /**
     * Set the statusbar font.
     */
    void setStatusBarFont( const QFont & statusBarFont );

    /**
     * Set whether to show the KDar splash screen on startup. Disabling
     * this splash screen shaves 3 seconds off the startup time.
     */
    void setShowKDarSplashScreen( const bool showKDarSplashScreen );

    //members with no corresponding GUI --- used for storage
    // for create and restore dialogs. I.e. these aren't used globally.
    /**
     * Set whether to perform a dry-run when doing a libdar operation. Currently
     * only works for create and restore.
     */
    void setDryRun( const bool dryRun );
    /**
     * Set the name of the isolation catalog.
     */
    void setIsolationArchiveName( const QString & isolationArchiveName );
    /**
     * Set the directory to compare the archive contents with.
     */
    void setDiffArchiveDirectory( const QString & diffArchiveDirectory );
    /**
     * Set the directory to extract the archive into.
     */
    void setExtractArchiveDirectory( const QString & extractArchiveDirectory );
    /**
     *
     */
    void setNewArchiveBasename( const QString & newArchiveBasename );
    /**
     * Set the log file URL
     */
    void setLogFile( const QString & logFile );

    /**
     * Returns the global KConfig *.
     */
    KConfig * globalConfig() const;
    /**
     * Returns the archiveName. This is used for opening an archive.
     */
    QString archiveName() const;
    /**
     * Returns the default archive storage directory.
     */
    QString archiveStorageDirectory() const;
    /**
     * Returns the name of an archive that is to be created.
     */
    QString newArchiveName( ) const;
    /**
     * Returns the name of the directory to backup.
     */
    QString directoryToBackup( ) const;
    /**
     * Returns whether to perform a differential backup.
     */
    bool differentialBackup( ) const;
    /**
     * Returns the reference archive name for differential backups
     */
    QString differentialBackupArchiveName( ) const;
    /**
     * Returns the input pipe string.
     */
    QString inputPipe() const;
    /**
     * Returns the output pipe string.
     */
    QString outputPipe() const;
    /**
     * Returns the "command to execute after slice creation" string.
     */
    QString executeCommand() const;
    /**
     * Returns the cryptography algorithm
     */
    int cryptoAlgorithm() const;
    /**
     * Returns the crypto block size.
     */
    int cryptoBlockSize() const;
    /**
     * Returns whether to store password
     */
    bool storePassword() const;
    /**
     * Returns the password string.
     */
    QString passwordString() const;
    /**
     * Returns the "command to execute before slice reading" string.
     */
    QString executeRef() const;
    /**
     * Returns the password string for reference archives.
     */
    QString passRef() const;
    /**
     * Returns whether to display verbose messages.
     */
    bool verbose() const;
    /**
     * Returns whether to allow file and slice overwriting.
     */
    bool allowOverwrite() const;
    /**
     * Returns whether to warn the user before overwriting a file.
     */
    bool warnOnOverwrite() const;
    /**
     * Returns whether to ignore the user extended attributes.
     */
    bool userEA() const;
    /**
     * Returns whether to ignore the system extended attributes.
     */
    bool systemEA() const;
    /**
     * Returns whether to ignore user and group IDs.
     */
    bool ignoreID() const;
    /**
     * Returns whether to delete files on disk during incremental restorations.
     */
    bool noDelete() const;
    /**
     * Returns whether to restore only files that are more recent than
     * those on the filesystem.
     */
    bool restoreRecent() const;
    /**
     * Returns whether to restore all files to a common top-level directory.
     */
    bool flatRestore() const;

    /**
     * Returns whether to use the create archive Wizard.
     */
    bool useCreateArchiveWizard() const;
    /**
     * Returns whether to split archives into slices.
     */
    bool splitArchive() const;
    /**
     * Returns whether to pause after each slice has been created.
     */
    bool pauseBetweenSlices() const;
    /**
     * Returns whether to keep excluded directory names during creation.
     */
    bool keepPruned() const;
    /**
     * Returns whether to ignore the ext2 filesystem dump flag.
     */
    bool ignoreDump() const;
    /**
     * Returns the media type (Custom, DVD, CD-R, etc.).
     */
    int media() const;
    /**
     * Returns the text of the current media type.
     */
    QString mediaText() const;
    /**
     * Returns the slice size number as a string.
     */
    QString sliceSizeString() const;
    /**
     * Returns the slice size suffix.
     */
    int sliceSizeSuffix() const;
    /**
     * Returns the first slice size numbers as a QStringList.
     */
    QStringList firstSliceSizeStrings() const;
    /**
     * Returns the first slice size number.
     */
    QString firstSliceSizeString() const;
    /**
     * Returns the first slice size suffices as a QValueList<int>.
     */
    QValueList<int> firstSliceSizeSuffices() const;
    /**
     * Returns the first slice size suffix.
     */
    int firstSliceSizeSuffix() const;
    /**
     * Returns the compression algorithm (libdar::bzip2, libdar::gzip,
     * libdar::none)
     */
    int compressionAlgorithm() const;
    /**
     * Returns the gzip compression level.
     */
    int compressionLevel() const;
    /**
     * Returns the minimum size that a file must be to be considered for
     * compression.
     */
    int minimumCompressionSizeInt() const;
    /**
     * Returns the hourshift value.
     */
    int hourShiftInt() const;

    /**
     * Returns the compression masks as a QStringList.
     */
    QStringList compressionMaskList() const;
    /**
     * Returns the directory mask list as a QStringList.
     */
    QStringList directoryMaskList() const;
    /**
     * Returns the file mask list as a QStringList.
     */
    QStringList fileMaskList() const;
    /**
     * Returns the compression mask type (KDar::EXCLUDE, KDar::INCLUDE).
     */
    int compressionMaskType() const;
    /**
     * Returns the directory mask types as a QStringList.
     */
    QStringList directoryMaskTypes() const;
    /**
     * Returns the file mask types.
     */
    int fileMaskType() const;

    /**
     * Returns the current general font.
     */
    QFont generalFont() const;
    /**
     * Returns the current archive browser font.
     */
    QFont fileBrowserFont() const;
    /**
     * Returns the current messages font.
     */
    QFont messageWindowFont() const;
    /**
     * Returns the current statusbar font.
     */
    QFont statusBarFont() const;

    /**
     * Returns whether to show the KDar splash screen on startup.
     */
    bool showKDarSplashScreen() const;

    //Non-gui members
    /**
     * Returns whether the libdar operation will be a dry-run.
     */
    bool dryRun() const;
    /**
     * Returns the name of the catalog archive that will be obtained by
     * isolating an archive.
     */
    QString isolationArchiveName() const;
    /**
     * Returns the directory for comparing with the contents of the archive.
     */
    QString diffArchiveDirectory() const;
    /**
     * Returns the directory to extract archive contents into.
     */
    QString extractArchiveDirectory() const;
    /**
     * Returns the name of a new archive that will be created.
     */
    QString newArchiveBasename() const;
    /**
     * Returns the log file URL
     */
    QString logFile() const;
    /**
     * Returns the KDar logging level.
     */
    int logLevel() const;
    /**
     * Returns the slice size as a libdar::infinint.
     */
    libdar::infinint libdarSliceSize();
    /**
     * Returns the first slice size as a libdar::infinint.
     */
    libdar::infinint libdarFirstSliceSize();
    /**
     * Returns the full creation compression mask.
     */
    libdar::et_mask libdarCreateCompressionMask();
    /**
     * Returns the full creation file mask.
     */
    libdar::et_mask libdarCreateFileMask();
    /**
     * Returns the full creation directory mask.
     */
    libdar::et_mask libdarCreateDirectoryMask();
    /**
     * Returns this KDarConfig's equivalent dar command line
     */
    QString buildDarCommandLine( KDar::libdarCommand command );
    /**
     * Call this method before this class is deleted  if you want
     * the settings you have (or the user has) chosen to become the
     * global, default settings.
     */
    bool writeGlobalSettings();

    /**
     * Call this method to restore configuration settings from the global
     * config file, or a profile file.
     */
    bool readGlobalSettings();
    /**
     * Calls the fillInDialog() method. Use this from a parent class to update
     * the contents of the dialog GUI.
     */
    void updateGUI();
    /**
     * Removes multiple slashes from the path.
     */
    static QString cleanPath( const QString & path );
    /**
     * Removes multiple slashes from each path in the list.
     */
    static QStringList cleanPath( const QStringList & paths );

protected:

    /**
     * Call this method to set the dialog's graphical elements to correspond
     * to the current state of the configuration object member variables.
     */
    void fillInDialog();
    /**
     * Fills the file or compression listview with the specified QStringList
     * of mask values.
     */
    void fillListView( QListView *, QStringList );
    /**
     * Updates the directory listview with the current directory mask values
     * and types.
     */
    void loadDirectoryFilters();
    /**
     * Returns a QStringList of values from a the contents of the listView.
     */
    QStringList makeQStringList( QListView * listView, int column = 0 );
    /**
     * Makes QStringLists for the directory filter mask values and types.
     */
    void makeDirectoryQStringLists( QListView * listView, QStringList &
directories, QStringList & types );


public slots:
    /**
     * Use this function to activate the help information for this
     * widget. The function is particulary useful if the help button is
     * not displayed as specified by the constructor. Normally you want
     * to hide the help button if this widget is embedded into a larger
     * dialog box that has its own help button. See kedit
     * (optiondialog.cpp) for an example.
     */
    void activateHelp( void );
    /**
     * Sets the media size for the specified media type ( i.e. media() ).
     */
    void setMediaSize( int );

    /**
     * Slot for setting the archive name
     */
    virtual void slotArchiveName( const QString & );
    /**
     * Slot for setting the default archive storage directory.
     */
    virtual void slotArchiveStorageDirectory( const QString & );
    /**
     * Slot for setting the input pipe name.
     */
    virtual void slotInputPipe( const QString & );
    /**
     * Slot for setting the output pipe name.
     */
    virtual void slotOutputPipe( const QString & );
    /**
     * Slot for setting the command to execute after writing an archive slice.
     */
    virtual void slotExecuteCommand( const QString & );
    /**
     * Slot for setting the encryption algorithm.
     */
    virtual void slotCryptoAlgorithm( int );
    /**
     * Slot for setting the encryption block size.
     */
    virtual void slotCryptoBlockSize( int );
    /**
     * Slot for setting the command to execute before reading a reference
     * archive slice.
     */
    virtual void slotExecuteRef( const QString & );
    /**
     * Slot for setting the password for descrambling reference archives.
     */
    virtual void slotPassRef( const QString & );
    /**
     * Slot for setting whether to store the encryption password
     */
    virtual void slotStorePassword( bool );
    /**
     * Change the password stored in the wallet
     */
    virtual void slotChangePassword();
    /**
     * Slot for setting the "verbose output" option.
     */
    virtual void slotVerbose( bool );
    /**
     * Slot for setting the "allow overwrites" option.
     */
    virtual void slotAllowOverwrite( bool );
    /**
     * Slot for setting "warn on overwrite" option.
     */
    virtual void slotWarnOnOverwrite( bool );
    /**
     * Slot for setting "user EA" option.
     */
    virtual void slotUserEA( bool );
    /**
     * Slot for setting the "system EA" option.
     */
    virtual void slotSystemEA( bool );
    /**
     * Slot for setting the "ignore user and group IDs" option.
     */
    virtual void slotIgnoreID( bool );
    /**
     * Slot for setting the "no delete" option.
     */
    virtual void slotNoDelete( bool );
    /**
     * Slot for setting the "Restore recent" option.
     */
    virtual void slotRestoreRecent( bool );
    /**
     * Slot for setting the "flat restore" option.
     */
    virtual void slotFlatRestore( bool );

    /**
     * Slot for setting the "use creation archive Wizard" option.
     */
    virtual void slotUseCreateArchiveWizard( bool );
    /**
     * Slot for setting the split-archive option.
     */
    virtual void slotSplitArchive( bool );
    /**
     * Slot for setting the pause-between-slices option.
     */
    virtual void slotPauseBetweenSlices( bool );
    /**
     * Slot for setting the keepPruned option.
     */
    virtual void slotKeepPruned( bool );
    /**
     * Slot for setting the ignoreDump option.
     */
    virtual void slotIgnoreDump( bool );
    /**
     * Slot for setting the media type.
     */
    virtual void slotMedia( int );
    /**
     * Slot for setting the slice size string. The slice size is the
     * numerical value of the slice size string in units of the
     * slice size suffix.
     */
    virtual void slotSliceSizeString( const QString & );
    /**
     * Slot for setting the slice size suffix.
     */
    virtual void slotSliceSizeSuffix( int );
    /**
     * Slot for setting the first slice size string. The first slice size is the
     * numerical value of the first slice size string in units of the
     * first slice size suffix.
     */
    virtual void slotFirstSliceSizeString( const QString & );
    /**
     * Slot for setting the first slice size suffix.
     */
    virtual void slotFirstSliceSizeSuffix( int );
    /**
     * Slot for setting the compression algorithm to "bzip2".
     */
    virtual void slotCompressionAlgorithmBZip2( bool );
    /**
     * Slot for setting the compression algorithm to "gzip".
     */
    virtual void slotCompressionAlgorithmGZip( bool );
    /**
     * Slot for setting the compression algorithm to "none".
     */
    virtual void slotCompressionAlgorithmNone( bool );
    /**
     * Slot for setting the gzip compression level.
     */
    virtual void slotCompressionLevel( int );
    /**
     * Slot for setting the minimum file size for compression.
     */
    virtual void slotMinimumCompressionSizeInt( int );
    /**
     * Slot for setting the hour shift.
     */
    virtual void slotHourShiftInt( int );

    /**
     * Slot for removing all file filters.
     */
    virtual void slotRemoveAllFileFilter();
    /**
     * Slot for removing a file filter.
     */
    virtual void slotRemoveFileFilter();
    /**
     * Slot for editing a file filter.
     */
    virtual void slotEditFileFilter();
    /**
     * Slot for adding a new file filter.
     */
    virtual void slotNewFileFilter();
    /**
     * Sets a new file filter in the file filter listview.
     */
    virtual void newFileFilter();
    /**
     * Slot for removing all directory filters.
     */
    virtual void slotRemoveAllDirectoryFilter();
    /**
     * Slot for removing a directory filter.
     */
    virtual void slotRemoveDirectoryFilter();
    /**
     * Slot for editing a directory filter.
     */
    virtual void slotEditDirectoryFilter();
    /**
     * Slot for adding a new directory filter.
     */
    virtual void slotNewDirectoryFilter();
    /**
     * Sets a new directory filter in the directory listview.
     */
    virtual void newDirectoryFilter();
    /**
     * Slot for removing all compression filters.
     */
    virtual void slotRemoveAllCompressionFilter();
    /**
     * Slot for removing a compression filter.
     */
    virtual void slotRemoveCompressionFilter();
    /**
     * Slot for editing a compression filter.
     */
    virtual void slotEditCompressionFilter();
    /**
     * Slot for adding a new compression filter to the configure dialog.
     */
    virtual void slotNewCompressionFilter();
    /**
     * Sets a new compression filter in the compression listview.
     */
    virtual void newCompressionFilter();
    /**
     * Slot for updating the compression mask list from the configuration dialog.
     */
    virtual void slotUpdateCompressionMaskList();
    /**
     * Slot for updating the directory mask list from the configuration dialog.
     */
    virtual void slotUpdateDirectoryMaskList();
    /**
     * Slot for updating the file mask list from the configuration dialog.
     */
    virtual void slotUpdateFileMaskList();

    /**
     * Slot for setting the compression mask list.
     */
    virtual void slotCompressionMaskList( const QStringList & );
    /**
     * Slot for setting the directory mask list.
     */
    virtual void slotDirectoryMaskList( const QStringList & );
    /**
     * Slot for setting the file mask list.
     */
    virtual void slotFileMaskList( const QStringList & );
    /**
     * Slot for setting the compression mask types.
     */
    virtual void slotCompressionMaskType( int );
    /**
     * Slot for setting the directory mask types.
     */
    virtual void slotDirectoryMaskTypes( const QStringList & );
    /**
     * Slot for setting the file mask type.
     */
    virtual void slotFileMaskType( int );

    /**
     * Slot for setting the general font. This doesn't work for reasons unknown.
     */
    virtual void slotGeneralFont( const QFont & );
    /**
     * Slot for setting the archive browser font.
     */
    virtual void slotFileBrowserFont( const QFont &);
    /**
     * Slot for setting the message window font.
     */
    virtual void slotMessageWindowFont( const QFont &);
    /**
     * Slot for setting the status bar font.
     */
    virtual void slotStatusBarFont( const QFont &);
    /**
     * Slot for setting whether to show KDarSplashScreen on startup.
     */
    virtual void slotKDarSplashScreen( bool );
    /**
     * Slot for setting the log file URL
     */
    virtual void slotLogFile( const QString & );
    /**
     * Slot for setting the log level to KDar::LOG_NONE
     */
    virtual void slotLogLevelNone( bool );
    /**
     * Slot for setting the log level to KDar::LOG_BASIC
     */
    virtual void slotLogLevelBasic( bool );
    /**
     * Slot for setting the log level to KDar::LOG_FULL
     */
    virtual void slotLogLevelFull( bool );
    /**
     * Slot for setting the current widget in the widget stack.
     */
    virtual void slotRaiseWidgetStack( const QString & );
    /**
     * Returns whether ZCompression is enabled
     */
    virtual bool ZCompression() const;
    /**
     * Returns whether BZ2Compression is enabled
     */
    virtual bool BZ2Compression() const;
    /**
     * Returns whether strong encryption is enabled
     */
    virtual bool strongCrypto() const;

protected slots:

    /**
     * Slot for showing the help dialog.
     */
    void slotHelp();

protected:
    // The options
    /**
     * Should we store the password in a wallet?
     */
    bool m_storePassword;
    /**
     *
     */
    bool m_nodialog;
    /**
     *
     */
    KConfig *m_kc;
    /**
     *
     */
    kdarCompressionFilterDialog *m_compressionDlg;
    /**
     *
     */
    kdarDirectoryFilterDialog *m_directoryDlg;
    /**
     *
     */
    kdarFileFilterDialog *m_fileDlg;

    /**
     *
     */
    QString m_archiveName;
    /**
     *
     */
    QString m_archiveStorageDirectory;
    /**
     *
     */
    QString m_newArchiveName;
    /**
     *
     */
    QString m_directoryToBackup;
    /**
     *
     */
    bool m_differentialBackup;
    /**
     *
     */
    QString m_differentialBackupArchiveName;
    /**
     *
     */
    QString m_inputPipe;
    /**
     *
     */
    QString m_outputPipe;
    /**
     *
     */
    QString m_executeCommand;
    /**
     *
     */
    int m_cryptoAlgorithm;
    /**
     *
     */
    int m_cryptoBlockSize;
    /**
     *
     */
    QString m_executeRef;
    /**
     *
     */
    QString m_passRef;

    /**
     *
     */
    bool m_verbose;
    /**
     *
     */
    bool m_allowOverwrite;
    /**
     *
     */
    bool m_warnOnOverwrite;
    /**
     *
     */
    bool m_userEA;
    /**
     *
     */
    bool m_systemEA;
    /**
     *
     */
    bool m_ignoreID;
    /**
     *
     */
    bool m_noDelete;
    /**
     *
     */
    bool m_restoreRecent;
    /**
     *
     */
    bool m_flatRestore;
    /**
     *
     */
    bool m_useCreateArchiveWizard;
    /**
     *
     */
    bool m_splitArchive;
    /**
     *
     */
    bool m_pauseBetweenSlices;
    /**
     *
     */
    bool m_keepPruned;
    /**
     *
     */
    bool m_ignoreDump;
    /**
     *
     */
    int m_media;
    /**
     *
     */
    QString m_sliceSizeString;
    /**
     *
     */
    int m_sliceSizeSuffix;
    /**
     *
     */
    QString m_firstSliceSizeString;
    /**
     *
     */
    int m_firstSliceSizeSuffix;
    /**
     *
     */
    int m_compressionAlgorithm;
    /**
     *
     */
    int m_compressionLevel;
    /**
     *
     */
    int m_minimumCompressionSizeInt;
    /**
     *
     */
    int m_hourShiftInt;

    /**
     *
     */
    QStringList m_compressionMaskList;
    /**
     *
     */
    QStringList m_directoryMaskList;
    /**
     *
     */
    QStringList m_fileMaskList;
    /**
     *
     */
    int m_compressionMaskType;
    /**
     *
     */
    QStringList m_directoryMaskTypes;
    /**
     *
     */
    int m_fileMaskType;
    /**
     *
     */
    QFont m_generalFont;
    /**
     *
     */
    QFont m_fileBrowserFont;
    /**
     *
     */
    QFont m_messageWindowFont;
    /**
     *
     */
    QFont m_statusBarFont;
    /**
     *
     */
    QString m_logFile;
    /**
     *
     */
    int m_logLevel;
signals:
    /**
     *
     */
    void configChanged();
    /**
     *
     */
    void compressionMaskListChanged();
    /**
     *
     */
    void directoryMaskListChanged();
    /**
     *
     */
    void fileMaskListChanged();
    /**
     * Signals that the user wants to change the password that is stored
     * in the wallet.
     */
    void changeWalletPassword();

private:
    //non-gui members stored in KDarConfig class
    /** Z compression enabled flag **/
    bool m_ZCompression;
    /** BZ2 compression enabled flag **/
    bool m_BZ2Compression;
    /** Strong encryption enabled flag **/
    bool m_strongCrypto;

    /**
     * Temporary holding string for a password. Definitely private!
     */
    QString m_passwordString;
    /**
     *
     */
    bool m_dryRun;
    /**
     *
     */
    QString m_isolationArchiveName;
    /**
     *
     */
    QString m_diffArchiveDirectory;
    /**
     *
     */
    QString m_extractArchiveDirectory;
    /**
     *
     */
    QString m_newArchiveBasename;
    /**
     *
     */
    bool m_showKDarSplashScreen;

    /**
     *
     */
    QStringList mediaNames;
    /**
     *
     */
    QStringList statusBarMediaNames;
    /**
     *
     */
    QStringList mediaSizes;
    /**
     *
     */
    QStringList mediaFirstSizes;
    /**
     *
     */
    QValueList<int> mediaSuffices;
    /**
     *
     */
    QValueList<int> mediaFirstSuffices;
};

#endif // _KDARCONFIG_H_
