/***************************************************************************
 *   Copyright (C) 2004 by Johnathan Burchill                              *
 *   jkerrb@users.sourceforge.net                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifndef _KDARLISTRECURSIVE_H_
#define _KDARLISTRECURSIVE_H_

#include <kio/job.h>

#include <dar/libdar.hpp>

namespace KIO {

    //This should go into KDE's global.h in kdelibs/kio/kio
    enum CommandNew
    {
        CMD_KDARLISTDIR = 'W' // 87
    };

    /**
     * A KDarListJob allows you to recursively get the contents of a directory
     * while filtering out subdirectories and files using libdar's masks.
     * Don't create the job directly, but use KIO::KDarListRecursive() instead.
     */
class KDarListJob : public SimpleJob {
    Q_OBJECT

public:
   /**
    * Do not create a KDarListJob directly. Use KIO::KDarListRecursive() instead.
    * @param url the url of the directory
    * @param prefix the prefix of the files, or QString::null for no prefix
    */
    KDarListJob( const KURL& url, libdar::et_mask directoryMask, libdar::et_mask fileMask, QString prefix = QString::null );
    /**
     * @internal
     * Called by the scheduler when a @p slave gets to
     * work on this job.
     * @param slave the slave that starts working on this job
     */
    virtual void start( Slave *slave );
    /**
     * Do not apply any KIOSK restrictions to this job.
     * @since 3.2
     */
    void setUnrestricted( bool unrestricted );

signals:
    /**
     * This signal emits the number of new archive entries found
     * from the listing.
     * @param job the job that emitted this signal
     */
    void entries( KIO::Job *job, const unsigned long newArchiveEntries );
    /**
     * Signals a redirection.
     * Use to update the URL shown to the user.
     * The redirection itself is handled internally.
     * @param job the job that is redirected
     * @param url the new url
     */
    void redirection( KIO::Job *job, const KURL &url );
    /**
     * Signals a permanent redirection.
     * The redirection itself is handled internally.
     * @param job the job that emitted this signal
     * @param fromUrl the original URL
     * @param toUrl the new URL
     * @since 3.1
     */
    void permanentRedirection( KIO::Job *job, const KURL &fromUrl, const KURL &toUrl );

protected slots:
    virtual void slotFinished( );
    virtual void slotMetaData( const KIO::MetaData &_metaData );
    virtual void slotResult( KIO::Job *job );
    void slotProcessEntries( const KIO::UDSEntryList& list );
    void slotRedirection( const KURL &url );
    void gotEntries( KIO::Job * subjob, const unsigned long newArchiveEntries );

private:
    bool recursive;
    bool includeHidden;
    QString prefix;
    unsigned long m_processedEntries;
    KURL m_redirectionURL;
    libdar::et_mask m_directoryMask;
    libdar::et_mask m_fileMask;

protected:
    virtual void virtual_hook( int id, void* data );

private:
    class ListJobPrivate* d;
};

    KDarListJob * KDarListRecursive( const KURL& url, libdar::et_mask directoryMask, libdar::et_mask fileMask );

}

#endif // _KDARLISTRECURSIVE_H_
