/***************************************************************************
 *   Copyright (C) 2004, 2005 by Johnathan Burchill                              *
 *   jkerrb@users.sourceforge.net                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#ifndef _KDARSTATISTICS_H_
#define _KDARSTATISTICS_H_

#include "kdarEnums.h"

#include <klocale.h>

#include <qstring.h>

#include <dar/libdar.hpp>

/**
 * This class handles reporting statistics on the libdar operations.
 *
 * @author Johnathan K. Burchill <jkerrb@users.sourceforge.net>
 * @version 1.3.0
 */

class KDarStatistics
{
public:

    /**
     * Constructor.
     *
     * @param libdarStats statistics structure returned by
     * various libdar operations.
     */
    KDarStatistics( libdar::statistics libdarStats );
    /**
     * Destructor.
     */
    virtual ~KDarStatistics();

    /**
     * Returns test statistics, formatted for output to message window
     * and log file.
     *
     * @param level one of KDar::LOG_NONE, KDar::LOG_BASIC, KDar::LOG_FULL
     */
    QString createStatistics( KDar::logLevel level );
    /**
     * Returns test statistics, formatted for output to message window
     * and log file.
     *
     * @param level one of KDar::LOG_NONE, KDar::LOG_BASIC, KDar::LOG_FULL
     */
    QString diffStatistics( KDar::logLevel level );
    /**
     * Returns test statistics, formatted for output to message window
     * and log file.
     *
     * @param level one of KDar::LOG_NONE, KDar::LOG_BASIC, KDar::LOG_FULL
     */
    QString isolateStatistics( KDar::logLevel level );
    /**
     * Returns test statistics, formatted for output to message window
     * and log file.
     *
     * @param level one of KDar::LOG_NONE, KDar::LOG_BASIC, KDar::LOG_FULL
     */
    QString restoreStatistics( KDar::logLevel level );
    /**
     * Returns test statistics, formatted for output to message window
     * and log file.
     *
     * @param level one of KDar::LOG_NONE, KDar::LOG_BASIC, KDar::LOG_FULL
     */
    QString testStatistics( KDar::logLevel level );

    //QString versions of statistics.
    /**
     * Returns the number of saved or restored inodes (files).
     */
    QString treated();
    /**
     * Returns the number of hard links stored.
     */
    QString hardLinks();
    /**
     * Returns the number of files not saved (no change since last backup),
     * or the number of files not restored (not in the backup).
     */
    QString skipped();
    /**
     * Returns the number of inodes that were excluded by filters.
     */
    QString ignored();
    /**
     * Returns the number of files ignored because they were older than
     * the filesystem versions.
     */
    QString tooOld();
    /**
     * Returns the number of inodes that could not be saved or restored
     * due to restrictive filesystem access permissions.
     */
    QString errors();
    /**
     * Returns the number inodes deleted or seen.
     */
    QString deleted();
    /**
     * Returns the number of EA saved or restored.
     */
    QString eaTreated();
    /**
     * Returns the total number of inodes considered by the operation.
     */
    QString total();

    //unsigned long versions of statistics.
    /**
     * Returns the number of saved or restored inodes (files).
     */
    unsigned long ul_treated();
    /**
     * Returns the number of hard links stored.
     */
    unsigned long ul_hardLinks();
    /**
     * Returns the number of files not saved (no change since last backup),
     * or the number of files not restored (not in the backup).
     */
    unsigned long ul_skipped();
    /**
     * Returns the number of inodes that were excluded by filters.
     */
    unsigned long ul_ignored();
    /**
     * Returns the number of files ignored because they were older than
     * the filesystem versions.
     */
    unsigned long ul_tooOld();
    /**
     * Returns the number of inodes that could not be saved or restored
     * due to restrictive filesystem access permissions.
     */
    unsigned long ul_errors();
    /**
     * Returns the number inodes deleted or seen.
     */
    unsigned long ul_deleted();
    /**
     * Returns the number of EA saved or restored.
     */
    unsigned long ul_eaTreated();
    /**
     * Returns the total number of inodes considered by the operation.
     */
    unsigned long ul_total();

private:

    QString m_treated;
    QString m_hardLinks;
    QString m_skipped;
    QString m_ignored;
    QString m_tooOld;
    QString m_errors;
    QString m_deleted;
    QString m_eaTreated;
    QString m_total;
};

#endif //_KDARSTATISTICS_H_
