/* This file is part of Strigi Desktop Search
 *
 * Copyright (C) 2007 Laurent Montel <montel@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "desktoplineanalyzer.h"


using namespace std;
using namespace Strigi;

// AnalyzerFactory
void
DesktopLineAnalyzerFactory::registerFields(FieldRegister& reg) {
    typeField = reg.registerField( "TODO_typefield", FieldRegister::stringType, 0, 0 );
    nameField = reg.registerField( "TODO_namefield", FieldRegister::stringType, 0, 0 );
    deviceField = reg.registerField( "TODO_devicefield", FieldRegister::stringType, 0, 0 );
    commentField = reg.registerField( "TODO_commentfield", FieldRegister::stringType, 0, 0 );
    mountPointField = reg.registerField( "TODO_mountpointfield", FieldRegister::stringType, 0, 0 );
    fileSystemField = reg.registerField( "TODO_filesystemfield", FieldRegister::stringType, 0, 0 );
    writableField = reg.registerField( "TODO_writablefield", FieldRegister::integerType, 0, 0 );
    fileTypeField = reg.registerField( "TODO_fileTypefield", FieldRegister::stringType, 0, 0 );
    fileSystemField = reg.registerField( "TODO_filesystemfield", FieldRegister::stringType, 0, 0 );
    preferedItemField = reg.registerField( "TODO_preferedItemField", FieldRegister::stringType, 0, 0 );
    linkToField = reg.registerField( "TODO_linkToField", FieldRegister::stringType, 0, 0 );
}

// Analyzer
void
DesktopLineAnalyzer::startAnalysis(AnalysisResult* i) {
    analysisResult = i;
    ready = false;
    writable = false;
}
void
DesktopLineAnalyzer::handleLine(const char* data, uint32_t length) {
    QString line(data);
    if ( !line.isEmpty() )
    {
        if ( line.contains( "Device" ) )
            device = line.startsWith( "Device=" );
        else if ( line.contains("MountPoint" ) )
            mountPoint= line.startsWith( "MountPoint" );
        /* s = i18n(file.readEntry("FSType").toLocal8Bit())*/
        else if ( line.startsWith( "FSType" ) )
            fileSystem=line.remove( "FSType=" );
        else if ( line.startsWith( "Writable" ) )
            writable = ( line.remove( "Writable=" ) == "true" );
        else if ( line.startsWith( "Link" ) )
            linkTo = line.remove("Link=" );
	else if ( line.startsWith("MimeType"))
            mimeTypes = line.remove("MimeType");
    }

#if 0
        s = file.readName();
    if (!s.isEmpty()) appendItem(group, "Name", s);

    s = file.readComment();
    if (!s.isEmpty()) appendItem(group, "Comment", s);

    QString type = file.readType();
    if (type == "FSDevice")
    {
        appendItem(group, "Type", i18n("Device"));

        s = file.readDevice();
        if (!s.isEmpty()) appendItem(group, "Device", s);

        s = file.readEntry("MountPoint");
        if (!s.isEmpty()) appendItem(group, "Mount Point", s);

        s = i18n(file.readEntry("FSType").toLocal8Bit());
        if (!s.isEmpty()) appendItem(group, "File System", s);

        appendItem(group, "Writable",
                    QVariant(!file.readEntry("ReadOnly", true)));

    }
    else if (type == "Service")
    {
        appendItem(group, "Type", i18n("Service"));

        s = file.readEntry("MimeType");
        if (!s.isEmpty())
        {
            KMimeType::Ptr mt = KMimeType::mimeType(s);
            appendItem(group, "File Type", mt->comment());
        }

        QString sType = file.readEntry("ServiceTypes");
        appendItem(group, "Service Type", sType);

        if (sType == "KFilePlugin")
        {
            QStringList preferred = file.readEntry("PreferredItems",QStringList());
            appendItem(group, "Preferred Items", preferred);
        }
    }
    else if (type == "Link")
    {
        QString url = file.readPathEntry("URL");
        appendItem(group, "Link To", url);
    }
#endif
}
void
DesktopLineAnalyzer::endAnalysis() {
    analysisResult->addValue( factory->deviceField, (const char)device.toUtf8());

    ready = true;
}
bool
DesktopLineAnalyzer::isReadyWithStream() {
    return ready;
}

//Factory
class Factory : public AnalyzerFactoryFactory {
public:
    list<StreamLineAnalyzerFactory*>
    streamLineAnalyzerFactories() const {
        list<StreamLineAnalyzerFactory*> af;
        af.push_back(new DesktopLineAnalyzerFactory());
        return af;
    }
};

STRIGI_ANALYZER_FACTORY(Factory)
