/*
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


// QT includes
#include <QLabel>
#include <QLineEdit>
#include <QCheckBox>
#include <QComboBox>
#include <QLayout>
//Added by qt3to4:
#include <QVBoxLayout>
#include <QHBoxLayout>

// KDE includes
#include <klocale.h>
#include <knuminput.h>
#include <kconfig.h>
#include <knumvalidator.h>
#include <kglobal.h>

// own includes
#include "defines.h"
#include "newgamedlg.h"


NewGameDlg::NewGameDlg(QWidget* parent)
  : KDialog(parent)
{
  setCaption(i18n("New Game"));
  setButtons(Ok|Cancel);
  setDefaultButton(Ok);
  setModal(true);
  showButtonSeparator(true);
  QFrame *frame = new QFrame(this);
  QVBoxLayout  *topLayout = new QVBoxLayout(frame);
  topLayout->setMargin( spacingHint() );
  QHBoxLayout  *l         = new QHBoxLayout;
  topLayout->addLayout( l );

  KSharedConfig::Ptr conf = KGlobal::config();
  conf->setGroup("NewGameDlg");
  bool showNewGameDlg = conf->readEntry("showNewGameDlgOnStartup",
					    SHOWNEWGAME_DEFAULT);
  bool readConfig     = conf->readEntry("readFromConfig",
					    LOADGAME_DEFAULT);
  int playerNr        = conf->readEntry("players", DEFAULT_PLAYERS);
  int money           = conf->readEntry("startMoney", START_MONEY);

  readFromConfig = new QCheckBox(i18n("Try loading a game"), frame);
  readFromConfig->adjustSize();
  readFromConfig->setChecked(readConfig);
  l->addWidget(readFromConfig);

  readFromConfigLabel = new QLabel(i18n("The following values are used if loading from config fails"), frame);
  if (!readFromConfig->isChecked())
    readFromConfigLabel->hide();
  readFromConfigLabel->adjustSize();
  l->addWidget(readFromConfigLabel);
  connect(readFromConfig, SIGNAL(toggled(bool)),
	  this,           SLOT(changeReadFromConfig(bool)));

  players = new KIntNumInput(playerNr, frame);
  players->setRange(1, MAX_PLAYERS);
  players->setLabel(i18n("How many players do you want?"));
  topLayout->addWidget(players);

  l = new QHBoxLayout;
  topLayout->addLayout( l );
  l->addWidget(new QLabel(i18n("Your name:"), frame));
  player1Name = new QLineEdit(frame);
  l->addWidget(player1Name);

  l = new QHBoxLayout;
  topLayout->addLayout( l );
  l->addWidget(new QLabel(i18n("Players' starting money:"), frame));
  moneyOfPlayers = new QLineEdit(QString("%1").arg(money), frame);
  moneyOfPlayers->setValidator( new KIntValidator( 0,999999,moneyOfPlayers ) );

  l->addWidget(moneyOfPlayers);

  l = new QHBoxLayout;
  topLayout->addLayout( l );
  l->addWidget(new QLabel(i18n("The names of your opponents:"), frame));
  computerNames = new QComboBox(frame);
  computerNames->setEditable( true );
  computerNames->setInsertPolicy(QComboBox::InsertAtCurrent);
  l->addWidget(computerNames);

  l = new QHBoxLayout;
  topLayout->addLayout( l );
  l->addWidget(new QLabel(i18n("Show this dialog every time on startup"),
			  frame));
  showDialogOnStartup = new QCheckBox(frame);
  showDialogOnStartup->setChecked(showNewGameDlg);
  l->addWidget(showDialogOnStartup);

  setPlayerNames();
}


NewGameDlg::~NewGameDlg()
{
  if (result() == Accepted) {
    KSharedConfig::Ptr conf = KGlobal::config();
    conf->setGroup("NewGameDlg"); // defaults for the newGameDlg only
    conf->writeEntry("showNewGameDlgOnStartup", showOnStartup());
    conf->writeEntry("readFromConfig", readFromConfigFile()); // just a default!
    conf->writeEntry("players", getPlayers());
    conf->writeEntry("startMoney", money());
  }

  //delete the visible elements:
  delete readFromConfigLabel;
  delete readFromConfig;
  delete players;
  delete moneyOfPlayers;
  delete showDialogOnStartup;
  delete player1Name;
  delete computerNames;

}


void NewGameDlg::setPlayerNames(int no, QString playerName)
{
  if (no < 0) {
    KGlobal::config()->setGroup("Save");
    player1Name->setText(KGlobal::config()->readEntry("Name_0", i18n("You")));
    computerNames->clear();
    for (int i = 1; i < MAX_PLAYERS; i++) {
      computerNames->addItem(KGlobal::config()->readEntry(QString("Name_%1").arg(i), i18n("Computer %1", i)));
    }
  } else if (no == 0) {
    player1Name->setText(playerName);
  } else {
    if (computerNames->count() > no)
      computerNames->insertItem(no-1, playerName);
    else
      computerNames->setItemText(no-1, playerName);
  }
}


void NewGameDlg::changeReadFromConfig(bool show)
{
  if (show)
    readFromConfigLabel->show();
  else
    readFromConfigLabel->hide();
}


bool NewGameDlg::showOnStartup()
{
  return showDialogOnStartup->isChecked();
}


int NewGameDlg::getPlayers()
{
  return players->value();
}


bool NewGameDlg::readFromConfigFile()
{
  return readFromConfig->isChecked();
}


int NewGameDlg::money()
{
  bool  ok    = true;
  int   money = moneyOfPlayers->text().toInt(&ok);
  if (ok)
    return money;
  else
    return START_MONEY;
}


QString NewGameDlg::name(int nr)
{
  if (computerNames->currentText() != computerNames->itemText(computerNames->currentIndex()))
    computerNames->setItemText(computerNames->currentIndex(), computerNames->currentText());

  if (nr == 0)
    return player1Name->text();

  if (nr <= computerNames->count())
    return computerNames->itemText(nr-1);

  return i18n("Player");
}


void NewGameDlg::hideReadingFromConfig()
{
  readFromConfig->hide();
  readFromConfigLabel->hide();
  readFromConfig->setChecked(false);
}


#include "newgamedlg.moc"
