/**
 * Copyright Michel Filippi <mfilippi@sade.rhein-main.de>
 *           Robert Williams
 *           Andrew Chant <andrew.chant@utoronto.ca>
 *           André Luiz dos Santos <andre@netvision.com.br>
 * Copyright (C) 2004-2005 Benjamin C. Meyer <ben at meyerhome dot net>
 *
 * This file is part of the ksnake package
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "pixServer.h"

#include <QPainter>
#include <QBitmap>

#include <kstandarddirs.h>
#include <klocale.h>
#include <kdebug.h>

#include "board.h"
#include "settings.h"

PixServer::PixServer( Board *b)
{
    board = b;
    initPixmaps();
    initBrickPixmap();
    initbackPixmaps();
    initRoomPixmap();
}

void PixServer::erase(int pos)
{
    if (!board->isEmpty(pos))
	return;

    QRect rect = board->rect(pos);
    QPainter p(&cachePix);
    p.drawPixmap(rect.topLeft(), backPix, rect);
}

void PixServer::restore(int pos)
{
    QRect rect = board->rect(pos);
    QPainter p(&cachePix);
    p.drawPixmap(rect.topLeft(), roomPix, rect);
}

void PixServer::draw(int pos, PixMap pix, int i)
{
    QPixmap p = QPixmap( BRICKSIZE, BRICKSIZE );

    QRect rect = board->rect(pos);

    QPainter paint(&p);
    if (! plainColor)
	paint.drawPixmap( QPoint(0, 0), backPix, rect );
    else
	p.fill(backgroundColor);

    switch (pix) {
    case SamyPix:
	paint.drawPixmap(0,0, samyPix[i]);
	break;
    case CompuSnakePix:
	paint.drawPixmap(0,0, compuSnakePix[i]);
	break;
    case ApplePix:
	paint.drawPixmap(0,0, applePix[i]);
	break;
    case BallPix:
	paint.drawPixmap(0,0, ballPix[i]);
	break;
    default:
	break;
    }

    paint.end();
    paint.begin(&cachePix);
    paint.drawPixmap(rect.topLeft(), p);
}

void PixServer::initPixmaps()
{

    QPixmap pm = QPixmap(KStandardDirs::locate("appdata", "pics/snake1.png"));
    QImage qi = pm.toImage();
    qi=qi.scaled(BRICKSIZE*18,BRICKSIZE, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
    pm = QPixmap::fromImage(qi,Qt::AvoidDither);
    QPainter paint;
    for (int x = 0 ; x < 18; x++){
		compuSnakePix[x] = QPixmap(BRICKSIZE, BRICKSIZE);
                paint.begin(&compuSnakePix[x]);
		paint.drawPixmap(0,0, pm, x*BRICKSIZE, 0, BRICKSIZE, BRICKSIZE);
                paint.end();
		compuSnakePix[x].setMask(compuSnakePix[x].createHeuristicMask());
    }

    pm = QPixmap(KStandardDirs::locate("appdata", "pics/snake2.png"));
    qi = pm.toImage();
    qi=qi.scaled(BRICKSIZE*18,BRICKSIZE, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
    pm = QPixmap::fromImage(qi,Qt::AvoidDither);
    for (int x = 0 ; x < 18; x++){
		samyPix[x] = QPixmap(BRICKSIZE, BRICKSIZE);
                paint.begin(&samyPix[x]);
		paint.drawPixmap(0,0, pm, x*BRICKSIZE, 0, BRICKSIZE, BRICKSIZE);
                paint.end();
		samyPix[x].setMask(samyPix[x].createHeuristicMask());
    }

    pm = QPixmap(KStandardDirs::locate("appdata", "pics/ball.png"));
    qi = pm.toImage();
    qi=qi.scaled(BRICKSIZE*4,BRICKSIZE, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
    pm = QPixmap::fromImage(qi,Qt::AvoidDither);
    for (int x = 0 ; x < 4; x++){
	ballPix[x] = QPixmap(BRICKSIZE, BRICKSIZE);
        paint.begin(&ballPix[x]);
	paint.drawPixmap(0,0, pm, x*BRICKSIZE, 0, BRICKSIZE, BRICKSIZE);
        paint.end();
	ballPix[x].setMask(ballPix[x].createHeuristicMask());
    }

    pm = QPixmap(KStandardDirs::locate("appdata", "pics/apples.png"));
    qi = pm.toImage();
    qi=qi.scaled(BRICKSIZE*2,BRICKSIZE, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
    pm = QPixmap::fromImage(qi,Qt::AvoidDither);
    for (int x = 0 ; x < 2; x++){
	applePix[x] = QPixmap(BRICKSIZE, BRICKSIZE);
        paint.begin(&applePix[x]);
	paint.drawPixmap(0,0, pm, x*BRICKSIZE, 0, BRICKSIZE, BRICKSIZE);
        paint.end();
	applePix[x].setMask(applePix[x].createHeuristicMask());
    }
}

void PixServer::initbackPixmaps()
{
	QString path;
	plainColor = false;

	if(Settings::bgcolor_enabled()){
	    backgroundColor = Settings::bgcolor();
	    plainColor = true;
	} else if(Settings::bgimage_enabled()) {
		// A bit of a hack.
		QStringList backgroundPixmaps =
		KGlobal::dirs()->findAllResources("appdata", "backgrounds/*.png");
	path = backgroundPixmaps.at(Settings::bgimage());
	}

    QPixmap PIXMAP;
    int pw, ph;

    backPix = QPixmap(MAPWIDTH, MAPHEIGHT);

    if (! plainColor) {

	PIXMAP = QPixmap(path);

	if (!PIXMAP.isNull()) {
	    pw = PIXMAP.width();
	    ph = PIXMAP.height();

            QPainter p(&backPix);
	    for (int x = 0; x <= MAPWIDTH; x+=pw)     //Tile BG Pixmap onto backPix
		for (int y = 0; y <= MAPHEIGHT; y+=ph)
		    p.drawPixmap(x, y, PIXMAP);
	}
	else  {
	    kDebug() << "error loading background image :" << path << endl;
	    backgroundColor = (QColor("black"));
	    plainColor = true;
	}
    }
    if ( plainColor)
	backPix.fill(backgroundColor);
}

void PixServer::initBrickPixmap()
{
    QPixmap pm = QPixmap(KStandardDirs::locate("appdata", "pics/brick.png"));
    if (pm.isNull()) {
	kFatal() << i18n("error loading %1, aborting\n", QString("brick.png"));
    }
    int pw = pm.width();
    int ph = pm.height();

    offPix = QPixmap(MAPWIDTH, MAPHEIGHT);
    QPainter p(&offPix);
    for (int x = 0; x <= MAPWIDTH; x+=pw)
	for (int y = 0; y <= MAPHEIGHT; y+=ph)
	    p.drawPixmap(x, y, pm);
}

void PixServer::initRoomPixmap()
{
    QPainter paint;

    roomPix = QPixmap(MAPWIDTH, MAPHEIGHT);
    paint.begin(&roomPix);
    paint.drawPixmap(0, 0, backPix);

    for (int x = 0; x < board->count(); x++) {
	if (board->isBrick(x))
	    drawBrick(&paint, x);
    }
    paint.end();

    cachePix = QPixmap(MAPWIDTH, MAPHEIGHT);
    paint.begin(&cachePix);
    paint.drawPixmap(0, 0, roomPix);
    paint.end();
}

void PixServer::drawBrick(QPainter *p ,int i)
{
    //Note, ROOMPIC IS OUR 'TARGET'
    QColor light(180,180,180);
    QColor dark(100,100,100);

    int topSq   = board->getNext(N, i);  //find 'address' of neighbouring squares
    int botSq   = board->getNext(S, i);
    int rightSq = board->getNext(E ,i);
    int leftSq  = board->getNext(W, i);

    QRect rect = board->rect(i); //get our square's rect

    int x = rect.x();
    int y = rect.y();      //Get x,y location of square???

    int width, height;

    int highlight = 2;    //Highlighting Distance (pixels)?

    width = height = rect.width();

    p->fillRect(x, y, width, height, light); //By default, fill square w/ Light? no.  use dark!!!!

    p->drawPixmap(x, y, offPix, x, y, width, height ); //Copy the brick pattern onto the brick

    if (!board->isBrick(rightSq)) p->fillRect(x + width - highlight, y, highlight, height, dark);  //highlight if its an end-brick.
    if (!board->isBrick(leftSq)) p->fillRect(x, y, highlight, height, light);
    if (!board->isBrick(botSq)) p->fillRect(x, y + height - highlight, width, highlight, dark);
    if (!board->isBrick(topSq)) p->fillRect(x, y, width, highlight, light);

}

