/* Yo Emacs, this -*- C++ -*-

  Copyright (C) 1999-2001 Jens Hoefkens
  jens@hoefkens.com

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  $Id: kbgoffline.cpp 661443 2007-05-05 15:24:28Z milliams $

*/

#include "kbgoffline.h"
#include "kbgoffline.moc"
#include "kbgofflinecalc.h"

#include <QLayout>
#include <QGroupBox>
#include <QCheckBox>
#include <QTimer>
#include <QSpinBox>
#include <kvbox.h>
#include <QLineEdit>
#include <QTabWidget>
#include <QLabel>
#include <QGridLayout>
#include <QMenu>
#include <ktoggleaction.h>
#include <kmessagebox.h>
#include <kinputdialog.h>
#include <kstandardaction.h>
#include <kconfig.h>
#include <klocale.h>
#include <kxmlguiwindow.h>
#include <kaction.h>
#include <kactioncollection.h>
#include <krandomsequence.h>
#include <kglobal.h>
#include <kicon.h>

#include "version.h"

using namespace std;

class KBgEngineOfflinePrivate
{
public:

    /*
     * Various flags, representing the current status of the game
     */
    bool mRollFlag, mUndoFlag, mDoneFlag, mCubeFlag, mGameFlag, mRedoFlag;

    /*
     * Store two copies of the game: one backup and a working copy
     */
    KBgStatus mGame[2];

    /*
     * Use the standard method of obtaining random numbers
     */
    KRandomSequence *mRandom;

    /*
     * Game actions
     */
    QAction *mNew, *mSwap;
    KToggleAction *mEdit;

    /*
     * Player's names
     */
    QString mName[2];

    /*
    * is the player the computer?
    */
    bool mComputer[2];

    /*
     * Who did the last roll
     */
    int mRoll;

    /*
     * How many checkers to move
     */
    int mMove;

    /*
     * Count the number of available undos
     */
    int mUndo;

    /*
     * Entry fields for the names
     */
    QLineEdit *mLe[2];

    /*
     * Checkbox to indicate whether player is computer
     */
    QCheckBox *cb[2];
    QSpinBox *sb;
    int mMatchTo;// play a match to this many points. FIXME should this be in KBgStatus?


};


// == constructor, destructor and other ========================================

/*
 * Constructor
 */
KBgEngineOffline::KBgEngineOffline(KXmlGuiWindow *parent, QMenu *pmenu)
    : KBgEngine(parent, pmenu)
{
    d = new KBgEngineOfflinePrivate();

    CompEng = new KBgEngineOfflineCalc();
    gamenumber = 0;

    /*
     * get some entropy for the dice
     */
    d->mRandom = new KRandomSequence;
    d->mRandom->setSeed(0);

    /*
     * Create engine specific actions
     */
    d->mNew = parent->actionCollection()->addAction("offline_new_game");
    d->mNew->setText(i18n("&New Game..."));
    connect(d->mNew, SIGNAL(triggered(bool) ), SLOT(newGame()));
    connect(this, SIGNAL( nextGame() ), SLOT( newGame() ) );
    //optimizing computer:
    connect(this, SIGNAL( nextMatch() ), SLOT( newMatch() ) );
    d->mSwap = parent->actionCollection()->addAction("swap_colors");
    d->mSwap->setText(i18n("&Swap Colors"));
    connect(d->mSwap, SIGNAL(triggered(bool) ), SLOT(swapColors()));

    d->mEdit = new KToggleAction(i18n("&Edit Mode"), this);
    parent->actionCollection()->addAction(QString(), d->mEdit);
    connect(d->mEdit, SIGNAL(triggered(bool) ), SLOT(toggleEditMode()));
    d->mEdit->setChecked(false);

    /*
     * create & initialize the menu
     */
    menu->addAction( d->mNew );
    menu->addAction( d->mEdit );
    menu->addAction( d->mSwap );

    /*
     * get standard board and set it
     */
    initGame();
    emit newState(d->mGame[0]);

    /*
     * initialize the commit timeout
     */
    ct = new QTimer(this);
    connect(ct, SIGNAL(timeout()), this, SLOT(done()));

    /*
     * internal statue variables
     */
    d->mRollFlag = d->mUndoFlag = d->mGameFlag = d->mDoneFlag = false;
    connect(this, SIGNAL(allowCommand(int, bool)), this, SLOT(setAllowed(int, bool)));

    /*
     * connection for computer engine
     */
    connect(this, SIGNAL(currentMove(QString *)), this, SLOT(handleMove(QString *)));


    /*
     * Restore last stored settings
     */
    readConfig();
}

/*
 * Destructor. The only child is the popup menu.
 */
KBgEngineOffline::~KBgEngineOffline()
{
    saveConfig();
    delete d->mRandom;
    delete d;
}


// == configuration handling ===================================================

/*
 * Put the engine specific details in the setup dialog
 */
void KBgEngineOffline::getSetupPages(KPageDialog *nb)
{
    /*
     * Main Widget
     */
    KVBox *vbp = new KVBox();
    KPageWidgetItem *page = new KPageWidgetItem(vbp,i18n("KBackgammon engine"));
    page->setHeader(i18n("KBackgammon Engine"));
    page->setIcon( KIcon( PROG_NAME "_engine" ));
    nb->addPage(page);

    /*
     * Get a multi page work space
     */
    QTabWidget *tc = new QTabWidget(vbp );
    tc->setObjectName( "offline tabs");

    /*
     * Player names
     */
    QWidget *w = new QWidget(tc);
    QGridLayout *gl = new QGridLayout(w);
    gl->setMargin(nb->spacingHint());

    /*
     * Group boxes
     */
    QGroupBox *gbn = new QGroupBox(i18n("Names"), w);

    gl->addWidget(gbn, 0, 0);

    gl = new QGridLayout(gbn);
    gl->setMargin(20);

    d->mLe[0] = new QLineEdit(d->mName[0], gbn);
    d->mLe[1] = new QLineEdit(d->mName[1], gbn);

    QLabel *lb[2];
    lb[0] = new QLabel(i18n("First player:"), gbn);
    lb[1] = new QLabel(i18n("Second player:"), gbn);

    d->cb[0] = new QCheckBox(i18n("Computer"), gbn);
    d->cb[0]->setChecked(d->mComputer[0]);
    d->cb[1] = new QCheckBox(i18n("Computer"), gbn);
    d->cb[1]->setChecked(d->mComputer[1]);

    d->cb[0]->setWhatsThis( i18n( "Check for the computer to play" ) );
    d->cb[1]->setWhatsThis( i18n( "Check for the computer to play" ) );

    QLabel *lb2;
    lb2 = new QLabel(i18n("Match to:"), gbn);
    d->sb = new QSpinBox;
    d->sb->setSingleStep(1);
    d->sb->setValue(d->mMatchTo);

    for (int i = 0; i < 2; i++) {
        gl->addWidget(lb[i], i, 0);
        gl->addWidget(d->mLe[i], i, 1);
	gl->addWidget(d->cb[i], i, 2);
    }
    gl->addWidget(lb2,3,0);
    gl->addWidget(d->sb, 3,1);

    d->mLe[0]->setWhatsThis( i18n("Enter the name of the first player."));
    d->mLe[1]->setWhatsThis( i18n("Enter the name of the second player."));

    /*
     * Done with the page, put it in
     */
    gl->activate();
    tc->addTab(w, i18n("&Player Names"));
}

/*
 * Called when the setup dialog is positively closed
 */
void KBgEngineOffline::setupOk()
{
    d->mName[0] = d->mLe[0]->text();
    d->mName[1] = d->mLe[1]->text();

    d->mComputer[0] = d->cb[0]->isChecked();
    d->mComputer[1] = d->cb[1]->isChecked();

    d->mMatchTo = d->sb->value();

}
void KBgEngineOffline::setupDefault()
{
    d->mName[0] = i18n("South");
    d->mName[1] = i18n("North");
}
void KBgEngineOffline::setupCancel()
{
    // do nothing
}

/*
 * Restore settings
 */
void KBgEngineOffline::readConfig()
{
    KConfigGroup config(KGlobal::config(), "offline engine");

    d->mName[0] = config.readEntry("player-one", i18n("South")); // same as above
    d->mName[1] = config.readEntry("player-two", i18n("North")); // same as above
    d->mComputer[0] = config.readEntry("player-one-computer", false);
    d->mComputer[1] = config.readEntry("player-two-computer", true );
    d->mMatchTo = (config.readEntry("match-to") ).toInt();

    cl = config.readEntry("timer", 2500);
}

/*
 * Save the engine specific settings
 */
void KBgEngineOffline::saveConfig()
{
    KConfigGroup config(KGlobal::config(), "offline engine");

    config.writeEntry("player-one", d->mName[0]);
    config.writeEntry("player-two", d->mName[1]);
    config.writeEntry("player-one-computer", d->mComputer[0]);
    config.writeEntry("player-two-computer", d->mComputer[1]);
    config.writeEntry("match-to", QString::number( d->mMatchTo ));
    config.writeEntry("timer",      cl);
}


// == start and init games =====================================================

/*
 * Start a new game.
 */
void KBgEngineOffline::newGame()
{
    int u = 0;
    int t = 0;

    /*
     * If there is a game running we warn the user first
     */
    if (d->mGameFlag && (KMessageBox::warningYesNo((QWidget *)parent(),
                                                   i18n("A game is currently in progress."
                                                        "Starting a new one will terminate it."),
                                                   QString(), KGuiItem(i18n("Start New Game")),
                                                   KGuiItem(i18n("Continue Old Game")))
                         == KMessageBox::No))
        return;

    /*
     * Separate from the previous game
     */
    emit infoText("<br/><br/><br/>");

    /*
     * Get player's names - user can still cancel
     */
    if ( d->mGame[0].points(US) == 0 && d->mGame[0].points(THEM) == 0 ) {
   	if (!queryPlayerName(US) || !queryPlayerName(THEM))
       	return;
    }



    /*
     * let the games begin
     */
    d->mGameFlag = true;

    /*
     * Initialize the board
     */
    initGame();

    /*
     * Figure out who starts by rolling
     */
    while (u == t) {
        u = getRandom();
        t = getRandom();
        emit infoText(i18n("%1 rolls %2, %3 rolls %4.",
                      d->mName[0], u, d->mName[1], t));
    }

    if (u > t) {
        emit infoText(i18n("%1 makes the first move.", d->mName[0]));
        d->mRoll = US;
    } else {
        emit infoText(i18n("%1 makes the first move.", d->mName[1]));
        d->mRoll = THEM;
        int n = u; u = t; t = n;
    }

    /*
     * set the dice and tell the board
     */
    rollDiceBackend(d->mRoll, u, t);

    /*
     * tell the user
     */
    emit statText(i18n("%1 vs. %2", d->mName[0], d->mName[1]));
}

//optimizing computer
void KBgEngineOffline::newMatch()
{
	d->mGame[0].setPoints(US,   0);
	d->mGame[0].setPoints(THEM, 0);

	//emit nextGame();
}


/*
 * Initialize the state descriptors mGame[0|1]
 */
void KBgEngineOffline::initGame()
{
    /*
     * nobody rolled yet
     */
    d->mRoll = -1;

    /*
     * set up a standard game
     */
    d->mGame[0].setCube(1, true, true);
    d->mGame[0].setDirection(+1);
    d->mGame[0].setColor(+1);
    for (int i = 1; i < 25; i++)
        d->mGame[0].setBoard(i, US, 0);
    d->mGame[0].setBoard( 1, US,   2); d->mGame[0].setBoard( 6, THEM, 5);
    d->mGame[0].setBoard( 8, THEM, 3); d->mGame[0].setBoard(12, US,   5);
    d->mGame[0].setBoard(13, THEM, 5); d->mGame[0].setBoard(17, US,   3);
    d->mGame[0].setBoard(19, US,   5); d->mGame[0].setBoard(24, THEM, 2);
    d->mGame[0].setHome(US, 0); d->mGame[0].setHome(THEM, 0);
    d->mGame[0].setBar(US, 0); d->mGame[0].setBar(THEM, 0);
    d->mGame[0].setDice(US  , 0, 0); d->mGame[0].setDice(US  , 1, 0);
    d->mGame[0].setDice(THEM, 0, 0); d->mGame[0].setDice(THEM, 1, 0);

    /*
     * save backup of the game state
     */
    d->mGame[1] = d->mGame[0];

    emit allowCommand(Load, true);


}

/*
 * Open a dialog to query for the name of player w. Return true unless
 * the dialog was canceled.
 */
bool KBgEngineOffline::queryPlayerName(int w)
{
    bool ret = false;
    QString *name;
    QString text;

    if (w == US) {
        name = &d->mName[0];
        text = i18n("Please enter the nickname of the player whose home\n"
                    "is in the lower half of the board:");
    } else {
        name = &d->mName[1];
        text = i18n("Please enter the nickname of the player whose home\n"
                    "is in the upper half of the board:");
    }

    do {
    *name = KInputDialog::getText(i18n("Player name"), text, *name, &ret, (QWidget *)parent());
    if (!ret) break;

    } while (name->isEmpty());

    return ret;
}


// == moving ===================================================================

/*
 * Finish the last move - called by the timer and directly by the used
 */
void KBgEngineOffline::done()
{
    ct->stop();

    emit allowMoving(false);
    emit allowCommand(Done, false);
    emit allowCommand(Undo, false);

    if (abs(d->mGame[0].home(d->mRoll)) == 15) {

        emit infoText(i18n("%1 wins the game. Congratulations!",
                      (d->mRoll == US) ? d->mName[0] : d->mName[1]));
	d->mGame[0].setPoints( (d->mRoll == US) ? US : THEM , d->mGame[0].points( (d->mRoll == US) ? US : THEM ) + 1 );
	emit infoText(i18n("Score: %1 to %2 of %3", d->mGame[0].points(US), d->mGame[0].points(THEM), d->mMatchTo ) );
        d->mGameFlag = false;
        emit allowCommand(Roll, false);
        emit allowCommand(Cube, false);
	if ( d->mGame[0].points( d->mRoll ) < d->mMatchTo )
		emit nextGame();
	else {
		emit infoText(i18n("%1 wins the match!", (d->mRoll == US) ? d->mName[0] : d->mName[1]));
		emit nextMatch();
		//emit nextGame();
	}

    } else {

        emit allowCommand(Roll, true);
        if (d->mGame[0].cube((d->mRoll == US ? THEM : US)) > 0) {

            d->mGame[0].setDice(US  , 0, 0); d->mGame[0].setDice(US  , 1, 0);
            d->mGame[0].setDice(THEM, 0, 0); d->mGame[0].setDice(THEM, 1, 0);

            emit newState(d->mGame[0]);
            emit getState(&d->mGame[0]);

            d->mGame[1] = d->mGame[0];

	    if ( !d->mComputer[ ( 1- d->mRoll ) ] ) {
	    	emit infoText(i18n("%1, please roll or double.",
        		(d->mRoll == THEM) ? d->mName[0] : d->mName[1]));
		emit allowCommand(Cube, true);

	    } else {
		roll();
            	emit allowCommand(Cube, false);
	    }
        } else {

            roll();//FIXME computer rolling here too
            emit allowCommand(Cube, false);
        }
    }
}

/*
 * Undo the last move
 */
void KBgEngineOffline::undo()
{
    ct->stop();

    d->mRedoFlag = true;
    ++d->mUndo;

    emit allowMoving(true);
    emit allowCommand(Done, false);
    emit allowCommand(Redo, true);
    emit undoMove();
}

/*
 * Redo the last move
 */
void KBgEngineOffline::redo()
{
    --d->mUndo;
    emit redoMove();
}

/*
 * Take the move string and make the changes on the working copy
 * of the state.
 */
void KBgEngineOffline::handleMove(QString *s)
{
    int index = 0;
    QString t = s->mid(index, s->indexOf(' ', index));
    index += 1 + t.length();
    int moves = t.toInt();

    /*
     * Allow undo and possibly start the commit timer
     */
    d->mRedoFlag &= ((moves < d->mMove) && (d->mUndo > 0));
    emit allowCommand(Undo, moves > 0);
    emit allowCommand(Redo, d->mRedoFlag);
    emit allowCommand(Done, moves == d->mMove);
    if (moves == d->mMove && cl) {
        emit allowMoving(false);
        ct->setSingleShot(true);
        ct->start(cl);
    }

    /*
     * Apply moves to d->mGame[1] and store results in d->mGame[0]
     */
    d->mGame[0] = d->mGame[1];

    /*
     * process each individual move
     */
    for (int i = 0; i < moves; i++) {
        bool kick = false;
        t = s->mid(index, s->indexOf(' ', index) - index);
        index += 1 + t.length();
        char c = '-';
        if (t.contains('+')) {
            c = '+';
            kick = true;
        }
        QString r = t.left(t.indexOf(c));
        if (r.contains("bar")) {
            d->mGame[0].setBar(d->mRoll, abs(d->mGame[0].bar(d->mRoll)) - 1);
        } else {
            int from = r.toInt();
            d->mGame[0].setBoard(from, d->mRoll, abs(d->mGame[0].board(from)) - 1);
        }
        t.remove(0, 1 + r.length());
        if (t.contains("off")) {
            d->mGame[0].setHome(d->mRoll, abs(d->mGame[0].home(d->mRoll)) + 1);
        } else {
            int to = t.toInt();
            if (kick) {
                d->mGame[0].setBoard(to, d->mRoll, 0);
                int el = ((d->mRoll == US) ? THEM : US);
                d->mGame[0].setBar(el, abs(d->mGame[0].bar(el)) + 1);
            }
            d->mGame[0].setBoard(to, d->mRoll, abs(d->mGame[0].board(to)) + 1);
        }
    }
}


// == dice & rolling ===========================================================

/*
 * Roll random dice for the player whose turn it is
 */
void KBgEngineOffline::roll()
{
    rollDice((d->mRoll == US) ? THEM : US);
}

/*
 * If possible, roll random dice for player w
 */
void KBgEngineOffline::rollDice(const int w)
{
    if ((d->mRoll != w) && d->mRollFlag) {
        rollDiceBackend(w, getRandom(), getRandom());
        return;
    }
    emit infoText(i18n("It is not your turn to roll!"));
}

/*
 * Return a random integer between 1 and 6. According to the man
 * page of rand(), this is the way to go...
 */
int KBgEngineOffline::getRandom()
{
    return 1+d->mRandom->getLong(6);
}

/*
 * Set the dice for player w to a and b. Reload the board and determine the
 * maximum number of moves
 */
void KBgEngineOffline::rollDiceBackend(const int w, const int a, const int b)
{
    /*
     * This is a special case that stems from leaving the edit
     * mode.
     */
    if (a == 0)
        return;

    /*
     * Set the dice and tel the board about the new state
     */
    d->mGame[0].setDice(w, 0, a);
    d->mGame[0].setDice(w, 1, b);
    d->mGame[0].setDice((w == US) ? THEM : US, 0, 0);
    d->mGame[0].setDice((w == US) ? THEM : US, 1, 0);
    d->mGame[0].setTurn(w);

    d->mGame[1] = d->mGame[0];

    d->mRoll = w;
    emit newState(d->mGame[0]);

    /*
     * No more roling until Done and no Undo yet
     */
    emit allowCommand(Undo, false);
    emit allowCommand(Roll, false);
    d->mRedoFlag = false;
    d->mUndo = 0;

    /*
     * Tell the players how many checkers to move
     */
    switch (d->mMove = d->mGame[0].moves()) {
    case -1:
        emit infoText(i18n("Game over!"));
        d->mGameFlag = false;
        emit allowCommand(Roll, false);
        emit allowCommand(Cube, false);
        emit allowMoving(false);
        break;
    case  0:
        emit infoText(i18n("%1, you cannot move.",
                      (w == US) ? d->mName[0] : d->mName[1]));
        if (cl)
        {
            ct->setSingleShot(true);
            ct->start(cl);
        }
        emit allowMoving(false);
        break;
//    case  1:
    default:
	emit allowMoving(true);

	if ( !d->mComputer[ d->mRoll ] )
		emit infoText(QString((w == US) ? d->mName[0] : d->mName[1]) +
		i18np(", please move 1 piece.",", please move %1 pieces.",d->mMove));
	else {
		QString s;
		s = CompEng->getComputerMove( d->mGame[0] );
		emit currentMove( &s );
		s.remove(0, 2);
		emit infoText( i18n( "%1 rolls %2, %3, and moves %4.",(d->mRoll == US) ? d->mName[0] : d->mName[1] , d->mGame[0].dice(d->mRoll,0), d->mGame[0].dice(d->mRoll,1), s ) );
		//emit infoText( i18n( "%1 %2", d->mGame[0].points(US), d->mGame[0].points(THEM) ) );
	}
        break;
    }
}


// == cube =====================================================================

/*
 * Double the cube for the player that can double  - asks player
 */
void KBgEngineOffline::cube()
{
    int w = ((d->mRoll == US) ? THEM : US);

    if (d->mRollFlag && d->mGame[0].cube(w) > 0) {
        emit allowCommand(Cube, false);

	if ( !d->mComputer[ d->mRoll ] ) {

		if (KMessageBox::questionYesNo((QWidget *)parent(),
					i18n("%1 has doubled. %2, do you accept the double?",
					(w == THEM) ? d->mName[1] : d->mName[0],
					(w == US) ? d->mName[1] : d->mName[0]),
					i18n("Doubling"), KGuiItem(i18n("Accept")), KGuiItem(i18n("Reject"))) != KMessageBox::Yes) {
		d->mGameFlag = false;
		emit allowCommand(Roll, false);
		emit allowCommand(Cube, false);
		emit infoText(i18n("%1 wins the game. Congratulations!",
				(w == US) ? d->mName[0] : d->mName[1]));
		d->mGame[0].setPoints( (w == US) ? US : THEM , d->mGame[0].points( (w == US) ? US : THEM ) + 1 );
		emit infoText(i18n("Score: %1 to %2", d->mGame[0].points(US), d->mGame[0].points(THEM) ) );
		return;
		}
	} else {
		if ( CompEng->getDouble( d->mGame[0] ) == 0 ) { // ask the computer engine whether to accept the double
			d->mGameFlag = false;
			emit allowCommand(Roll, false);
			emit allowCommand(Cube, false);
			emit infoText(i18n("%1 wins the game. Congratulations!",
				      (w == US) ? d->mName[0] : d->mName[1]));
			d->mGame[0].setPoints( (w == US) ? US : THEM , d->mGame[0].points( (w == US) ? US : THEM ) + 1 );
			emit infoText(i18n("Score: %1 to %2", d->mGame[0].points(US), d->mGame[0].points(THEM) ) );
			return;
		}
	}
        emit infoText(i18n("%1 has accepted the double. The game continues.",
                      (w == THEM) ? d->mName[0] : d->mName[1]));

        if (d->mGame[0].cube(US)*d->mGame[0].cube(THEM) > 0)
            d->mGame[0].setCube(2, w == THEM, w == US);
        else
            d->mGame[0].setCube(2*d->mGame[0].cube(w), w == THEM, w == US);

        emit newState(d->mGame[0]);
        emit getState(&d->mGame[0]);

        d->mGame[1] = d->mGame[0];

        roll();
    }
}

/*
 * Double the cube for player w
 */
void KBgEngineOffline::doubleCube(const int)
{
    cube();
}


// == various slots & functions ================================================

/*
 * Check with the user if we should really quit in the middle of a
 * game.
 */
bool KBgEngineOffline::queryClose()
{
    if (!d->mGameFlag)
        return true;

    switch (KMessageBox::warningContinueCancel((QWidget *)parent(),
                                      i18n("In the middle of a game. "
                                           "Really quit?"), QString(), KStandardGuiItem::quit())) {
    case KMessageBox::Continue :
        return true;
    case KMessageBox::Cancel :
        return false;
    default: // cancel
        return false;
    }
    return true;
}

/*
 * Quitting is fine at any time
 */
bool KBgEngineOffline::queryExit()
{
    return true;
}

/*
 * Handle textual commands. Right now, all commands are ignored
 */
void  KBgEngineOffline::handleCommand(const QString& cmd)
{
    emit infoText(i18n("Text commands are not yet working. "
                       "The command '%1' has been ignored.", cmd));
}

/*
 * Load the last known sane state of the board
 */
void KBgEngineOffline::load()
{
    if (d->mEdit->isChecked())
        emit newState(d->mGame[1]);
    else {
        // undo up to four moves
        undo();
        undo();
        undo();
        undo();
    }
}

/*
 * Store if cmd is allowed or not
 */
void KBgEngineOffline::setAllowed(int cmd, bool f)
{
    switch (cmd) {
    case Roll:
        d->mRollFlag = f;
        return;
    case Undo:
        d->mUndoFlag = f;
        return;
    case Cube:
        d->mCubeFlag = f;
        return;
    case Done:
        d->mDoneFlag = f;
        return;
    }
}

/*
 * Swaps the used colors on the board
 */
void KBgEngineOffline::swapColors()
{
    d->mGame[1].setDice(US,   0, d->mGame[0].dice(US,   0));
    d->mGame[1].setDice(US,   1, d->mGame[0].dice(US,   1));
    d->mGame[1].setDice(THEM, 0, d->mGame[0].dice(THEM, 0));
    d->mGame[1].setDice(THEM, 1, d->mGame[0].dice(THEM, 1));
    d->mGame[1].setColor(d->mGame[1].color(THEM), US);
    emit newState(d->mGame[1]);
    emit getState(&d->mGame[1]);
    d->mGame[0] = d->mGame[1];
}

/*
 * Switch back and forth between edit and play mode
 */
void KBgEngineOffline::toggleEditMode()
{
    emit setEditMode(d->mEdit->isChecked());
    if (d->mEdit->isChecked()) {
        ct->stop();
        d->mNew->setEnabled(false);
        d->mSwap->setEnabled(false);
        emit allowCommand(Undo, false);
        emit allowCommand(Roll, false);
        emit allowCommand(Done, false);
        emit allowCommand(Cube, false);
        emit statText(i18n("%1 vs. %2 - Edit Mode", d->mName[0], d->mName[1]));
    } else {
        d->mNew->setEnabled(true);
        d->mSwap->setEnabled(true);
        emit statText(i18n("%1 vs. %2", d->mName[0], d->mName[1]));
        emit getState(&d->mGame[1]);
        d->mGame[0] = d->mGame[1];
        emit allowCommand(Done, d->mDoneFlag);
        emit allowCommand(Cube, d->mCubeFlag);
        emit allowCommand(Undo, d->mUndoFlag);
        emit allowCommand(Roll, d->mRollFlag);
        int w =((d->mGame[0].dice(US, 0) && d->mGame[0].dice(US, 1)) ? US : THEM);
        rollDiceBackend(w, d->mGame[0].dice(w, 0), d->mGame[0].dice(w, 1));
    }
}
