/* Yo Emacs, this -*- C++ -*-

  Copyright (C) 1999-2001 Jens Hoefkens <jens@hoefkens.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  $Id: kbg.cpp 712975 2007-09-15 21:54:19Z aacid $

*/

#include "kbg.h"
#include "kbg.moc"

#include <QPainter>
#include <QLayout>
#include <QGroupBox>
#include <QRadioButton>
#include <QPixmap>


#include <QStringList>
#include <QIcon>
#include <QHideEvent>
#include <QLabel>
#include <QShowEvent>
#include <QGridLayout>
#include <QMenu>
#include <QSplitter>
#include <QApplication>

#include <kmenubar.h>
#include <ktoolbar.h>
#include <kedittoolbar.h>
#include <klocale.h>
#include <kaction.h>
#include <kactioncollection.h>
#include <kstandardaction.h>
#include <kstandardgameaction.h>
#include <kaboutdata.h>
#include <kmessagebox.h>
#include <kconfig.h>
#include <kcompletion.h>
#include <kcompletionbox.h>
#include <kmenu.h>
#include <kurllabel.h>
#include <krun.h>
#include <kstatusbar.h>
#include <klineedit.h>
#include <knuminput.h>
#include <kprinter.h>
#include <ktoolinvocation.h>
#include <kxmlguifactory.h>
#include <kvbox.h>
#include <kglobal.h>
#include <kactionmenu.h>
#include <kpagedialog.h>
#include <kicon.h>
#include <kselectaction.h>
#include "kbgtextview.h"
#include "offline/kbgoffline.h"
#include "fibs/kbgfibs.h"
#include "gnubg/kbggnubg.h"
#include "nextgen/kbgng.h"
#include "version.h"

// == setup ====================================================================

/*
 * Constructor creates user interface, actions and first engine.
 */
KBg::KBg()
    : KXmlGuiWindow()
{
	/*
	 * Initialize menu strings
	 */
	engineString[Offline] = i18n("KBackgammon");
	engineString[FIBS   ] = i18n("FIBS");
	engineString[GNUbg  ] = i18n("GNU Backgammon (Experimental)");
//	engineString[NextGen] = i18n("Next Generation (Experimental)");

	helpTopic[FIBSHome][0] = i18n("FIBS Home");
	helpTopic[FIBSHome][1] = "http://www.fibs.com/";

	helpTopic[RuleHome][0] = i18n("Backgammon Rules");
	helpTopic[RuleHome][1] = "http://www.bkgm.com/rules.html";

	/*
	 * The main view is shared between the board and a small text window
	 */
	panner = new QSplitter(Qt::Vertical, this);
	panner->setObjectName("panner");
	board  = new KBgBoardSetup(panner);
	board->setObjectName("board");
	status = new KBgTextView(panner);
	status->setObjectName("status");

	statusLabel= new QLabel(i18n("KBackgammon"), statusBar());
	statusBar()->addWidget(statusLabel, 1);
	setCentralWidget(panner);

	QAction *action;

	/*
	 * Create all actions needed by the application
	 */
	
	// Game
	newAction = KStandardGameAction::gameNew(this, SLOT(openNew()), actionCollection());
	KStandardGameAction::print(this, SLOT(print()), actionCollection());
	KStandardGameAction::quit(this, SLOT(close()), actionCollection());
	
	// Move
	undoAction = KStandardGameAction::undo(this, SLOT(undo()), actionCollection());
	undoAction->setEnabled(false);
	redoAction = KStandardGameAction::redo(this, SLOT(redo()), actionCollection());
	redoAction->setEnabled(false);
	rollAction = KStandardGameAction::roll(this, SLOT(roll()), actionCollection());
	rollAction->setEnabled(false);
	endAction = KStandardGameAction::endTurn(this, SLOT(done()), actionCollection());
	endAction->setEnabled(false);

  
	// AB: what the heck has this to do with redisplay? perhaps use reload instead?
	//loadAction = KStandardAction::redisplay(this, SLOT(load()), actionCollection(), "move_load");
	//loadAction->setEnabled(false);


	cubeAction = actionCollection()->addAction("move_cube");
	cubeAction->setIcon( KIcon( PROG_NAME "-double.xpm" ) );
	cubeAction->setText( i18n("Double Cube") );
	connect( cubeAction, SIGNAL(triggered(bool)), SLOT( cube() ) );
	cubeAction->setEnabled(false);

	action = KStandardAction::showMenubar(this, SLOT(toggleMenubar()), this);
	actionCollection()->addAction(action->objectName(), action);
	action = KStandardAction::preferences(this, SLOT(setupDlg()), this);
	actionCollection()->addAction(action->objectName(), action);
	action = KStandardAction::saveOptions(this, SLOT(saveConfig()), this);
	actionCollection()->addAction(action->objectName(), action);

	KActionMenu *actMenu = new KActionMenu(i18n("&Backgammon on the Web"), this);
	actionCollection()->addAction("help_www", actMenu);
	KMenu *p = actMenu->menu();

	action = actionCollection()->addAction("help_www_fibs");
	action->setText(helpTopic[FIBSHome][0]);
	connect(action, SIGNAL(triggered(bool) ), SLOT(wwwFIBS()));
	p->addAction(action);
	action = actionCollection()->addAction("help_www_rule");
	action->setText(helpTopic[RuleHome][0]);
	connect(action, SIGNAL(triggered(bool) ), SLOT(wwwRule()));
	p->addAction(action);

	/*
	 * Set up the command line - using actions, otherwise recreating the GUI will delete them
	 * (e.g. using KEditToolBar)
	 */
	cmdLabel = new QLabel(i18n("Command: "), this);
	action = actionCollection()->addAction("command_label");
	action->setText(cmdLabel->text());
	qobject_cast<KAction*>( action )->setDefaultWidget( cmdLabel );

	cmdLine  = new KLineEdit( this );
	cmdLine->setObjectName( "commandline" );
	action = actionCollection()->addAction("command_lineedit");
	action->setText(QString());
	qobject_cast<KAction*>( action )->setDefaultWidget( cmdLine );
#ifdef __GNUC__
#warning "kde4 : port it"
#endif
//	actionCmdLine->setAutoSized(true);

//	cmdLine->completionObject()->setOrder(KCompletion::Weighted);
	connect(cmdLine, SIGNAL(returnPressed(const QString &)), this, SLOT(handleCmd(const QString &)));
	/*
	 * Done with the actions, create the XML-defined parts of the
	 * user interface
	 */
	statusBar();
	setupGUI();

//	cmdLine->setFocus();

	/*
	 * Initialize the engine to the default (offline). If the user
	 * prefers a different engine, it will be started later
	 */
	for (int i = 0; i < MaxEngine; i++)
		engine[i] = 0;
	currEngine = None;
	setupEngine(Offline);

	/*
	 * Set up configuration handling.
	 * FIXME: support session management
	 */
	connect(this, SIGNAL(readSettings()), board, SLOT(readConfig()));
	connect(this, SIGNAL(saveSettings()), board, SLOT(saveConfig()));

	/*
	 * Set up some whatis messages for the online help
	 */
	status->setWhatsThis( i18n("This area contains the status messages for the game. "
			     "Most of these messages are sent to you from the current "
			     "engine."));
	toolBar("cmdToolBar")->setWhatsThis(
			i18n("This is the command line. You can type special "
			     "commands related to the current engine in here. "
			     "Most relevant commands are also available "
			     "through the menus."));
	toolBar("mainToolBar")->setWhatsThis(
			i18n("This is the button bar tool bar. It gives "
			     "you easy access to game related commands. "
			     "You can drag the bar to a different location "
			     "within the window."));
	statusBar()->setWhatsThis(
			i18n("This is the status bar. It shows you the currently "
			     "selected engine in the left corner."));

	/*
	 * Create and initialize the context menu
	 */
	QMenu* menu = (QMenu*)factory()->container("popup", this);
	board->setContextMenu(menu);
}

#ifdef __GNUC__
#warning "add action to configure knotify"
#endif

/*
 * Destructor is empty
 */
KBg::~KBg() {}


// == engine handling ==========================================================

/*
 * Set the engine according to newEngine.
 * Additional engines have to be added to the switchstatement (and only there).
 */
void KBg::setupEngine(int newEngine)
{
	/*
	 * Engine doesn't need to be changed?
	 */
	if (newEngine == currEngine)
		return;

	/*
	 * Check with the engine if it can be terminated
	 */
	if (currEngine != None && engine[currEngine] && !engine[currEngine]->queryClose()) {
		return;
	}

	//KMessageBox::information(this, i18n("setting engine type"),
	//				 i18n("Information"), "conf_menubar_information");

	/*
	 * Remove the old engine, create a new one, and hook up menu and slots/signals
	 */
	QMenu *commandMenu = (QMenu *)factory()->container("command_menu", this);
	commandMenu->clear();

	if (currEngine != None) {
		delete engine[currEngine];
		engine[currEngine] = 0;
	}

	currEngine = newEngine;
	switch (currEngine) {
	case Offline:
		kDebug()<<"Setting engine type: Offline";
		engine[currEngine] = new KBgEngineOffline(this, commandMenu);
		break;
	case FIBS:
		kDebug()<<"Setting engine type: FIBS";
		engine[currEngine] = new KBgEngineFIBS(this, commandMenu);
		break;
	case GNUbg:
		kDebug()<<"Setting engine type: GNUbg";
		engine[currEngine] = new KBgEngineGNU(this, commandMenu);
		break;
//	case NextGen:
//   kDebug()<<"Setting engine type: NextGen";
//	    engine[currEngine] = new KBgEngineNg(this, commandMenu);
//	    break;
	default: // FIXME: we need some kind of catch here...
		kDebug()<<"Setting engine type: default";
		currEngine = Offline;
		engine[currEngine] = new KBgEngineOffline(this, commandMenu);
		break;
	}

	statusLabel->setText(QString(engineString[currEngine]));

	KConfigGroup config( KGlobal::config(), "global settings");
	if (config.readEntry("enable timeout", true))
		engine[currEngine]->setCommit(config.readEntry("timeout", 2.5));
	newAction->setEnabled(engine[currEngine]->haveNewGame());

	// engine -> this
	connect(engine[currEngine], SIGNAL(statText(const QString &)), this, SLOT(updateCaption(const QString &)));
	connect(engine[currEngine], SIGNAL(infoText(const QString &)), status, SLOT(write(const QString &)));
	connect(engine[currEngine], SIGNAL(allowCommand(int, bool)),   this, SLOT(allowCommand(int, bool)));

	// this -> engine
	connect(this, SIGNAL(readSettings()), engine[currEngine], SLOT(readConfig()));
	connect(this, SIGNAL(saveSettings()), engine[currEngine], SLOT(saveConfig()));

	// board -> engine
	connect(board, SIGNAL(rollDice(const int)),    engine[currEngine], SLOT(rollDice(const int)));
	connect(board, SIGNAL(doubleCube(const int)),  engine[currEngine], SLOT(doubleCube(const int)));
	connect(board, SIGNAL(currentMove(QString *)), engine[currEngine], SLOT(handleMove(QString *)));

	// engine -> board
	connect(engine[currEngine], SIGNAL(undoMove()), board, SLOT(undoMove()));
	connect(engine[currEngine], SIGNAL(redoMove()), board, SLOT(redoMove()));
	connect(engine[currEngine], SIGNAL(setEditMode(const bool)), board, SLOT(setEditMode(const bool)));
	connect(engine[currEngine], SIGNAL(allowMoving(const bool)), board, SLOT(allowMoving(const bool)));
	connect(engine[currEngine], SIGNAL(getState(KBgStatus *)), board, SLOT(getState(KBgStatus *)));
	connect(engine[currEngine], SIGNAL(newState(const KBgStatus &)), board, SLOT(setState(const KBgStatus &)));

	// now that all signals are connected, start the engine
	engine[currEngine]->start();
}


// == configuration handing ====================================================

/*
 * Save all settings that should be saved for the next start.
 */
void KBg::saveConfig()
{
	KConfigGroup config( KGlobal::config(), "global settings");

	/*
	 * Save the main window options unless the user has asked not
	 * to do so.
	 */
	if (config.readEntry("autosave on exit", true)) {

		config.changeGroup("main window");

		config.writeEntry("origin", pos());

		config.writeEntry("font",   status->font());
		config.writeEntry("panner", (double)board->height()/(double)panner->height());

		saveMainWindowSettings(config);
	}

	/*
	 * Save the history
	 */
	config.changeGroup("command line");
	config.writeEntry("history", cmdLine->completionObject()->items());

	/*
	 * Save current engine
	 */
	config.changeGroup("engine settings");
	config.writeEntry("last engine", currEngine);

	/*
	 * Tell other objects to save their settings, too.
	 */
	emit saveSettings();

	config.sync();
}

/*
 * Read the stored configuration and apply it
 */
void KBg::readConfig()
{
	KConfigGroup config( KGlobal::config(), "global settings");

	/*
	 * Restore the main window settings unless the user has asked
	 * not to do so.
	 */
	if (config.readEntry("autosave on exit", true)) {

		config.changeGroup("main window");

		QPoint pos, defpos(10, 10);
		QFont kappFont = QApplication::font();

		pos = config.readEntry("origin", defpos);

		status->setFont(config.readEntry("font", kappFont));

		QList<int> l;
		l.append(qRound(   config.readEntry("panner", 0.75) *panner->height()));
		l.append(qRound((1-config.readEntry("panner", 0.75))*panner->height()));
		panner->setSizes(l);

		applyMainWindowSettings( config );
	}

	/*
	 * Restore the history
	 */
	config.changeGroup("command line");
	cmdLine->completionObject()->setItems(config.readEntry("history",QStringList()));

	/*
	 * Tell other objects to read their configurations
	 */
	emit readSettings();

	/*
	 * Restore last engine
	 */
	config.changeGroup("engine settings");
	setupEngine((Engines)config.readEntry("last engine",int(Offline)));
}


// == configuration ============================================================

/*
 * Connected to the setup dialog applyClicked signal. Make sure
 * that all changes are saved.
 */
void KBg::setupOk()
{
	// game type
	int gameType;
	for (int i = 0; i < 3; ++i) {
 		if ( rbEngine[i]->isChecked() )
			gameType = i;
 	}

	// global settings
	KConfigGroup config( KGlobal::config(), "global settings");

	config.writeEntry("enable timeout",   cbt->isChecked());
	config.writeEntry("timeout",          sbt->value());
	config.writeEntry("autosave on exit", cbs->isChecked());

	config.changeGroup("engine");
	config.writeEntry("last engine",      gameType);


	// tell engine about commit timer
	engine[currEngine]->setCommit(cbt->isChecked() ? sbt->value() : -1);

	// one time requests
	if (cbm->isChecked())
		KMessageBox::enableAllMessages();

	// tell children to read their changes
	board->setupOk();

	// engines
	for (int i = 0; i < MaxEngine; i++)
		engine[i]->setupOk();
	// will reset engine if it has changed
	setupEngine(gameType);

	// save it all
	saveConfig();
}

/*
 * Load default values for the user settings
 */
void KBg::setupDefault()
{
	// timeout
	cbt->setChecked(true);
	sbt->setValue(2.5);

	// messages
	cbm->setChecked(false);

	// auto save
	cbs->setChecked(true);

	// board
	board->setupDefault();

	// engines
	for (int i = 0; i < MaxEngine; i++)
		engine[i]->setupDefault();
}

/*
 * Connected to the setup dialog cancelButtonPressed signal. There
 * isn't much to do. We tell the board to undo the changes.
 */
void KBg::setupCancel()
{
	// board
	board->setupCancel();

	// engines
	for (int i = 0; i < MaxEngine; i++)
		engine[i]->setupCancel();
}

/*
 * Setup dialog is ready to be deleted. Do it later...
 */
void KBg::setupDone()
{
	nb->delayedDestruct();
	for (int i = 0; i < MaxEngine; i++)
		if (i != currEngine) engine[i] = 0;
}

// FIXME make more general...

void KBg::startKCM(const QString &url)
{
	KRun::runCommand(url, this);
}

/*
 * Initialize and display the setup dialog
 */
void KBg::setupDlg()
{
	/*
	 * Get a new notebook in which all other members can put their
	 * config pages
	 */
	nb = new KPageDialog(this);
	nb->setButtons(KDialog::Ok|KDialog::Cancel|KDialog::Default|KDialog::Apply|KDialog::Help);
	nb->setDefaultButton(KDialog::Ok);
	nb->setFaceType(KPageDialog::List);
	nb->setModal(true);
	nb->showButtonSeparator(true);
	KConfigGroup config( KGlobal::config(), "global settings");

	/*
	 * Main Widget
	 */
	KVBox *w = new KVBox();
	KPageWidgetItem *pageItem = new KPageWidgetItem( w, i18n("General settings"));
	pageItem->setHeader(i18n("General"));
	pageItem->setIcon( KIcon( "go" ));
	nb->addPage(pageItem);

	/*
	 * Group boxes
	 */
	QGroupBox *gbg = new QGroupBox(i18n("Game Type"), w);
	QGroupBox *gbm = new QGroupBox(i18n("Messages"), w);
	QGroupBox *gbt = new QGroupBox(i18n("Timer"), w);
	QGroupBox *gbs = new QGroupBox(i18n("Autosave"), w);
	QGroupBox *gbe = new QGroupBox(i18n("Events"), w);

	/*
	 * Game type
	 */
	//FIXME this should be made more general: automatically determine game types
	rbEngine[Offline] = new QRadioButton(engineString[Offline]);
	rbEngine[FIBS]    = new QRadioButton(engineString[FIBS]);
	rbEngine[GNUbg]   = new QRadioButton(engineString[GNUbg]);
	config.changeGroup("engine");
	for ( int i = 0; i < MaxEngine; ++i )
		rbEngine[i]->setChecked( config.readEntry( "last engine" ).toInt() == i );

	QGridLayout *glg = new QGridLayout(gbg);
	glg->setMargin(10);
	glg->addWidget(rbEngine[0], 0, 0);
	glg->addWidget(rbEngine[1], 0, 1);
	glg->addWidget(rbEngine[2], 0, 2);

	/*
	 * Timer box
	 */
	gbt->setWhatsThis( i18n("After you finished your moves, they have to be sent to the engine. "
				  "You can either do that manually (in which case you should not enable "
				  "this feature), or you can specify an amount of time that has to pass "
				  "before the move is committed. If you undo a move during the timeout, the "
				  "timeout will be reset and restarted once you finish the move. This is "
				  "very useful if you would like to review the result of your move."));

	cbt = new QCheckBox(i18n("Enable timeout"), gbt);
	cbt->setChecked(config.readEntry("enable timeout", true));

	sbt = new KDoubleNumInput(gbt);
	sbt->setRange(0.0, 60.0, 0.5);
	sbt->setLabel(i18n("Move timeout in seconds:"));
	sbt->setValue(config.readEntry("timeout", 2.5));

	connect(cbt, SIGNAL(toggled(bool)), sbt, SLOT(setEnabled(bool)));
	sbt->setEnabled(cbt->isChecked());

	QGridLayout *gl = new QGridLayout(gbt);
	gl->setMargin(10);
	gl->addWidget(cbt, 0, 0);
	gl->addWidget(sbt, 1, 0);

	/*
	 * Enable messages
	 */
	gbm->setWhatsThis( i18n("Check the box to enable all the messages that you have previously "
				  "disabled by choosing the \"Do not show this message again\" option."));

	QGridLayout *glm = new QGridLayout(gbm);
	glm->setMargin(nb->spacingHint());
	cbm = new QCheckBox(i18n("Reenable all messages"), gbm);
	glm->addWidget(cbm, 0, 0);

	/*
	 * Save options on exit ?
	 */
	gbm->setWhatsThis( i18n("Check the box to automatically save all window positions on program "
				  "exit. They will be restored at next start."));

	QGridLayout *gls = new QGridLayout(gbs);
	gls->setMargin(nb->spacingHint());
	cbs = new QCheckBox(i18n("Save settings on exit"), gbs);
	cbs->setChecked(config.readEntry("autosave on exit", true));
	gls->addWidget(cbs, 0, 0);

	/*
	 * Event configuration
	 */
	gbe->setWhatsThis( i18n("Event notification of %1 is configured as part of the "
				  "system-wide notification process. Click here, and you "
				  "will be able to configure system sounds, etc.",
			KGlobal::mainComponent().aboutData()->programName()));

	QGridLayout *gle = new QGridLayout(gbe);
	gle->setMargin(nb->spacingHint());
	KUrlLabel *lab = new KUrlLabel("kcmshell kcmnotify",
				       i18n("Click here to configure the event notification"), gbe);
	lab->setMaximumSize(lab->sizeHint());

	gle->addWidget(lab, 0, 0);
	connect(lab, SIGNAL(leftClickedUrl(const QString &)), SLOT(startKCM(const QString &)));

	/*
	 * Board settings
	 */
	board->getSetupPages(nb);

	/*
	 * Hack alert: this little trick makes sure that ALL engines
	 * have their settings available in the dialog.
	 */
	QMenu *dummyPopup = new QMenu(nb);
#ifdef __GNUC__
#warning possibly the parent is wrong here. It was nb before but we need a XMLGui in there
#endif
	for (int i = 0; i < MaxEngine; i++) {
		if (currEngine != i) {
			switch (i) {
			case Offline:
				engine[i] = new KBgEngineOffline(this, dummyPopup);
				break;
			case FIBS:
				engine[i] = new KBgEngineFIBS(this, dummyPopup);
				break;
			case GNUbg:
				engine[i] = new KBgEngineGNU(this, dummyPopup);
				break;
//			case NextGen:
//				engine[i] = new KBgEngineNg( this, dummyPopup);
//				break;
			}
			connect(this, SIGNAL(saveSettings()), engine[i], SLOT(saveConfig()));
		}
		engine[i]->getSetupPages(nb);
	}

	/*
	 * Connect the signals of nb
	 */
	connect(nb, SIGNAL(okClicked()),     this, SLOT(setupOk()));
	connect(nb, SIGNAL(applyClicked()),  this, SLOT(setupOk()));
	connect(nb, SIGNAL(cancelClicked()), this, SLOT(setupCancel()));
	connect(nb, SIGNAL(defaultClicked()),this, SLOT(setupDefault()));

	connect(nb, SIGNAL(finished()), this, SLOT(setupDone()));

	nb->resize(nb->minimumSize());
	nb->show();
}


// == action slots =============================================================

/*
 * Tell the board to print itself - restore and save user settings for
 * the print dialog.
 */
void KBg::print()
{
	KPrinter *prt = new KPrinter();

	KConfigGroup config( KGlobal::config(), "printing");

	prt->setNumCopies(config.readEntry("numcopies", 1));
	prt->setOutputFileName(config.readPathEntry("outputfile"));
	prt->setOutputToFile(config.readEntry("tofile", false));
	prt->setPageSize((KPrinter::PageSize) config.readEntry("pagesize", int(KPrinter::A4)));
	prt->setOrientation((KPrinter::Orientation)config.readEntry("orientation", int(KPrinter::Landscape)));

	if (prt->setup(this, i18n("Print %1", baseCaption))) {
		QPainter p;
		p.begin(prt);
		board->print(&p);
		p.end();
		config.writeEntry("tofile",      prt->outputToFile());
		config.writePathEntry("outputfile",  prt->outputFileName());
		config.writeEntry("pagesize",    (int)prt->pageSize());
		config.writeEntry("orientation", (int)prt->orientation());
		config.writeEntry("numcopies",   prt->numCopies());
	}
	delete prt;
}

/*
 * Toggle visibility of the menubar - be careful that the menu doesn't
 * get lost
 */
void KBg::toggleMenubar()
{
	if (menuBar()->isVisible()) {

		KMessageBox::information(this, i18n("You can enable the menubar again with the "
						    "right mouse button menu of the board."),
					 i18n("Information"), "conf_menubar_information");
		menuBar()->hide();

	} else {

		menuBar()->show();
	}
}

/*
 * Display a standard dialog for the toolbar content
 */
void KBg::configureToolbars()
{
	KConfigGroup cg( KGlobal::config(), "kedittoolbar settings"); // temp group
	saveMainWindowSettings( cg );
	KEditToolBar dlg(actionCollection(), this);
	dlg.setResourceFile(xmlFile());
	connect(&dlg,SIGNAL(newToolBarConfig()),this,SLOT(newToolbarConfig()));
	dlg.exec();
	cg.deleteGroup(); // delete temp group
}

/*
 * Called when clicking OK or Apply in the toolbar editor
 */
void KBg::newToolbarConfig()
{
	createGUI();
	applyMainWindowSettings(KGlobal::config()->group( "kedittoolbar settings") );
}

/*
 * Help slots
 */
void KBg::wwwFIBS() {showWWW(FIBSHome);}
void KBg::wwwRule() {showWWW(RuleHome);}

void KBg::showWWW(int t)
{
	KToolInvocation::invokeBrowser(helpTopic[t][1]);
}

/*
 * Edit slots
 */
void KBg::undo() {engine[currEngine]->undo();}
void KBg::redo() {engine[currEngine]->redo();}
void KBg::roll() {engine[currEngine]->roll();}
void KBg::cube() {engine[currEngine]->cube();}
void KBg::done() {engine[currEngine]->done();}
void KBg::load() {engine[currEngine]->load();}

/*
 * Start a new game with the current engine
 */
void KBg::openNew()
{
	engine[currEngine]->newGame();
}


// == various slots - not for actions ==========================================

/*
 * Check with the engine if it is okay to close the window.
 * If so, save settings.
 */
bool KBg::queryClose()
{
	bool ret = engine[currEngine]->queryClose();
	if ( ret )
	    saveConfig();
	return ret;
}

/*
 * Set the caption of the main window. If the user has requested pip
 * counts, they are appended, too.
 */
void KBg::updateCaption(const QString &s)
{
	baseCaption = s;
	QString msg;
	if (!s.isEmpty()) {
		msg = s;
		if (board->getPipCount(US) >= 0) {
			QString tmp;
			tmp.setNum(board->getPipCount(US  ));
			msg += " - " + tmp;
			tmp.setNum(board->getPipCount(THEM));
			msg += '-'  + tmp;
		}
	}
	setCaption(msg, false);
}

/*
 * Take the string from the commandline, give it to the engine, append
 * to the history and clear the buffer.
 */
void KBg::handleCmd(const QString &s)
{
	if (!s.trimmed().isEmpty()) {
		engine[currEngine]->handleCommand(s);
		cmdLine->completionObject()->addItem(s);
	}
	cmdLine->clear();
	cmdLine->completionBox()->close();
}

/*
 * Reflect the availability of commands in the button bar.
 */
void KBg::allowCommand(int cmd, bool f)
{
	switch (cmd) {
	case KBgEngine::Undo:
        undoAction->setEnabled(f);
		break;
	case KBgEngine::Redo:
		redoAction->setEnabled(f);
		break;
	case KBgEngine::Roll:
		rollAction->setEnabled(f);
		break;
	case KBgEngine::Cube:
		cubeAction->setEnabled(f);
		break;
	case KBgEngine::Done:
		endAction->setEnabled(f);
		break;
	//case KBgEngine::Load:
	//	loadAction->setEnabled(f);
	//	break;
	}
}

/*
 * Catch the hide envents. That way, the current engine can close its
 * child windows.
 */
void KBg::hideEvent(QHideEvent *e)
{
	KXmlGuiWindow::hideEvent(e);
	engine[currEngine]->hideEvent();
}

/*
 * Catch the show envents. That way, the current engine can open any
 * previously hidden windows.
 */
void KBg::showEvent(QShowEvent *e)
{
	KXmlGuiWindow::showEvent(e);
	engine[currEngine]->showEvent();
}

// EOF

