// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
/***************************************************************************
 *   Copyright (C) 2006 by Wilfried Huss                                   *
 *   Wilfried.Huss@gmx.at                                                  *
 *                                                                         *
 *   Copyright (C) 2006 by Stefan Kebekus                                  *
 *   kebekus@kde.org                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.         *
 ***************************************************************************/

#ifndef DATAMODEL_H
#define DATAMODEL_H

#include "ligature_export.h"

#include "anchor.h"
#include "pageNumber.h"
#include "history.h"
#include "selection.h"
#include "pageSize.h"

#include <QColor>
#include <QMap>
#include <QObject>
#include <QString>
#include <QVector>

class DataView;
class KVSPrefs;

class LIGATURECORE_EXPORT DataModel: public QObject
{
  Q_OBJECT

public:
  /**
     The default constructor reads the preferences, and sets the
     defaultPageSize accordingly.
  */
  DataModel(QObject* parent = 0);

  virtual ~DataModel();

  /** @brief get the current page number
   */
  PageNumber currentPageNumber();

  /** @brief sets the current page number

     This method sets the current page number. The signal
     currentPageNumberChanged() will be emitted. The signal
     GUIGotoAnchor() be emitted.

     @param anchor an anchor with the new page number; this number
     must not be greater than numberOfPages(). If this page number
     here equals the currentPageNumber(), the signal gotoCurrentPage()
     might be emitted, but nothing further happens.

     @param weak This parameter decides if the signal
     GUIGotoAnchor() is emitted.
  */
  void setCurrentPageNumber(const Anchor&, bool weak = false);

  unsigned int numberOfPages();
  void setNumberOfPages(unsigned int);

  /** @brief get the resolution currently used for drawing.

     @returns the resolution of the display device. In
     principle. In fact, ligature implements zooming by calling the
     setResolution()-method with values that are not exactly the
     resolution of the display, but multiplied with the zoom
     factor. Bottom line: the documentRenderer should act as if this
     field indeed contains resolution of the display device. When a
     documentRenderer is constructed, this field is set to the actual
     resolution to give a reasonable default value.
   */
  double resolution() const { return resolutionInDPI; }
  void setResolution(double resolution);

  /** @brief delete pages from the document

    This method should be called whenever pages have been removed from
    the document. This method will adjust the bookmarks accordingly
    and emit the signal numberOfPagesChanged().

    @note This method will call setNumberOfPages() to adjust the
    number of pages in the document. The method setNumberOfPages()
    need not be called again.

    @param from first page of the document that was removed, must
    satisfy 1 <= from <= numberOfPages() and from <= to

    @param to last page of the document that was removed, must satisfy
    to <= numberOfPages()
   */
  void deletePages(const PageNumber& from, const PageNumber& to);

  /** @brief insert pages into the document

    This method should be called whenever pages were inserted into the
    document. This method will adjust the bookmarks accordingly and
    emit the signal numberOfPagesChanged().

    @note This method will call setNumberOfPages() to adjust the
    number of pages in the document. The method setNumberOfPages()
    need not be called again.

    @param before this argument specifies that where the pages were
    inserted: before page "before". For instance, if pages were
    inserted at the beginning of the document, this argument would be
    "1". If pages have been appended, this argument is
    numberOfPages()+1. The argument must satisfy 1 <= before <=
    numberOfPages()+1.

    @param number_of_pages_inserted the number of pages that were
    added to the document. This argument must satisfy 1 <=
    number_of_pages_inserted.
   */
  void insertPages(const PageNumber& before, quint16 number_of_pages_inserted);

  /** @brief Checks if a page is bookmarked. */
  bool isPageBookmarked(const PageNumber& page);

  /** @brief Return a list of all bookmarked pages. */
  QList<PageNumber> bookmarks();

  /** @brief Return the label of a bookmark.

    @returns QString::null if @arg page is not bookmarked. */
  QString bookmarkLabel(const PageNumber& page);

  /** @brief Load page metadata (bookmarks, comments, ...) from its XML representation. */
  void loadDocumentInfo(const QString& filename);

  /** @brief Save page metadata (bookmarks, comments, ...) as XML. */
  void saveDocumentInfo(const QString& filename) const;

  /** @brief Add a bookmark. */
  void addBookmark(const PageNumber& page, const QString& text);

  /** @brief Rename a bookmark. */
  void renameBookmark(const PageNumber& page, const QString& text);

  /** @brief Remove a bookmark. */
  void removeBookmark(const PageNumber& page);

  /** @brief Remove all bookmarks. */
  void removeAllBookmarks();

  QList<PageNumber> selectedPages();
  
  /** @brief Creates a string representation from a list of selected pages.

      The string is in the format expected by the "kde-range" option of KPrinter.
      For example: "1-5, 7, 10-15, 19, 21" */
  QString selectedPageRange();

  void selectPage(const PageNumber& page);
  void deselectPage(const PageNumber& page);
  void deselectAllPages();

  bool isSelected(const PageNumber& page) const;

  History* history() { return &_history; }

  KVSPrefs* preferences() { return prefs; }

  const TextSelection& selectedText() const { return selection; }
  void deselectText();
  void selectText(const TextSelection& selection);

  /** Copy the selected text into the clipboard. */
  void copyText() const { selection.copyText(); }

  /** @returns the paper size that is used when useCustomPageSize() is true. */
  const pageSize &defaultPageSize() const { return _defaultPageSize; }
  void setUserPreferredPageSize(SimplePageSize s);

  /** @returns the background color of an empty page, based on the current accessibility settings.

  @warning This is the color used as the background for empty page widgets. It has nothing to do
  with the real background color of the opended file.
  */
  QColor paperColor() const;


public slots:
  void setViewMode(int viewmode);


signals:
  void currentPageNumberChanged();

  /** This signal might emitted when the method setCurrentPageNumber()
      is called. It indicates that the current page number has changed
      and that the display should be updated accordingly.
  */
  void numberOfPagesChanged();
  void GUIGotoAnchor(Anchor anc);

  void layoutChanged();

  void viewModeChanged();

  /** This signal is emitted whenever the selected text changes.
      The argument is false if no text is selected, true otherwise.

      @ref deselectText
      @ref selectText */
  void selectionChanged(bool);

  /** @brief Add a bookmark.

      Connect to this signal to add a bookmark to your view of the data.
      This signals is emitted whenever @ref addBookmark() gets executed. */
  void bookmarkAdded(const PageNumber& page, const QString& label);

  /** @brief Rename a bookmark.

      Connect to this signal to rename a bookmark in your view of the data.
      This signal is emitted whenever @ref renameBookmark() gets executed. */
  void bookmarkRenamed(const PageNumber& page, const QString& label);

  /** @brief Remove a bookmark.

      Connect to this signal to remove a bookmark from your view of the data.
      This signal is emitted whenever @ref removeBookmark() gets executed. */
  void bookmarkRemoved(const PageNumber& page);

  /** @brief Remove all bookmarks.

      Connect to this signal to remove all bookmark from your view of the data.
      This signal is emitted whenever @ref allBookmark() gets executed. */
  void allBookmarksRemoved();

  void pageSelected(const PageNumber&);
  void pageDeselected(const PageNumber&);

private:
  PageNumber _currentPageNumber;
  unsigned int _numberOfPages;

  double resolutionInDPI;

  QMap<PageNumber, QString> userBookmarks;

  // A sorted list of the currently selected pages.
  QList<PageNumber> _selectedPages;

  History _history;
  
  // This holds the currently selected text.
  TextSelection selection;
  
  // This entry stores the "default paper size" that the user has
  // requested in the preferences dialog. If that paper size is
  // actually used or not, depends on if the document specifies a
  // paper size of its own.
  pageSize _defaultPageSize;
  
  KVSPrefs* prefs;
};


#endif
