// -*- Mode: C++; c-basic-offset: 2; indent-tabs-mode: nil; c-brace-offset: 0; -*-
#include <config.h>

#include "ligature.h"
#include "kvs_debug.h"

#include <kaboutdata.h>
#include <kapplication.h>
#include <kcmdlineargs.h>
#include <klocale.h>
#include <kmimetype.h>
#include <kurl.h>
#include <QtDBus/QtDBus>
#include <QDir>

#include <cstdlib>


static KCmdLineOptions options[] =
{
  { "u", 0, 0 },
  { "unique", I18N_NOOP("Check if the file is loaded in another ligature.\nIf it is, bring up the other ligature. Otherwise, load the file."), 0 },
  { "m", 0, 0 },
  { "mimetype <mimetype>", I18N_NOOP("Loads a plugin which supports files of type <mimetype>,\nif one is installed."), 0 },
  { "g", 0, 0 },
  { "goto <pagenumber>", I18N_NOOP("Navigate to this page"), 0 },
  { "+file(s)", I18N_NOOP("Files to load"), 0 },
  KCmdLineLastOption
};


static const char description[] = I18N_NOOP("Generic framework for viewer applications");


int main(int argc, char** argv)
{
  KAboutData about ("ligature", I18N_NOOP("Ligature"), "0.9",
                    description, KAboutData::License_GPL,
                    "(C) 2000, Matthias Hoelzer-Kluepfel\n"
                    "(C) 2004-2006, Wilfried Huss",
                    I18N_NOOP("Displays various document formats. "
                    "Based on original code from KGhostView."));
  about.addAuthor ("Wilfried Huss", I18N_NOOP("Current Maintainer"),
                   "Wilfried.Huss@gmx.at");
  about.addAuthor ("Matthias Hoelzer-Kluepfel", I18N_NOOP("Framework"),
                   "mhk@caldera.de");
  about.addAuthor ("David Sweet",
                   I18N_NOOP("KGhostView Maintainer"),
                   "dsweet@kde.org",
                   "http://www.chaos.umd.edu/~dsweet");
  about.addAuthor ("Mark Donohoe",
                   I18N_NOOP("KGhostView Author"));
  about.addAuthor ("Markku Hihnala",
                   I18N_NOOP("Navigation widgets"));
  about.addAuthor ("David Faure",
                   I18N_NOOP("Basis for shell"));
  about.addAuthor ("Daniel Duley",
                   I18N_NOOP("Port to KParts"));
  about.addAuthor ("Espen Sand",
                   I18N_NOOP("Dialog boxes"));
  about.addAuthor ("Stefan Kebekus",
                   I18N_NOOP("DCOP-Interface, major improvements"),
                   "kebekus@kde.org");

  KCmdLineArgs::init(argc, argv, &about);
  KCmdLineArgs::addCmdLineOptions(options);
  KApplication app;

  // see if we are starting with session management
  if (app.isSessionRestored())
  {
    RESTORE(Ligature);
  }
  else
  {
    KCmdLineArgs* args = KCmdLineArgs::parsedArgs();

    if (args->isSet("unique"))
    {
      // With --unique, we need one argument.
      if (args->count() < 1)
      {
        args->usage();
        exit(-1);
      }

      // Find the fully qualified file name of the file we are
      // loading. Complain, if we are given a URL which does not point
      // to a local file.
      KUrl url(args->url(0));

      if (!args->url(0).isValid())
      {
        kError(kvs::shell) << QString(I18N_NOOP("The URL %1 is not well-formed.")).arg(args->arg(0)) << endl;
        return -1;
      }

      if (!args->url(0).isLocalFile())
      {
        kError(kvs::shell) << QString(I18N_NOOP("The URL %1 does not point to a local file. You can only specify local "
                   "files if you are using the '--unique' option.")).arg(args->arg(0)) << endl;
        return -1;
      }

      if (QString(args->arg(0)) == "-")
      {
        kError(kvs::shell) << QString(I18N_NOOP("You can not read from standard input if you are using the '--unique' option.")) << endl;
        return -1;
      }

      QString qualPath = QFileInfo(args->url(0).path()).absoluteFilePath();
#ifdef __GNUC__
#warning "kde4 port to dbus"
#endif      
#if 0
      app.dcopClient()->attach();
      DCOPCString id = app.dcopClient()->registerAs("unique-ligature");
      if (id.isNull())
        kError(kvs::shell) << "There was an error using dcopClient()->registerAs()." << endl;
      DCOPCStringList apps = app.dcopClient()->registeredApplications();
      for ( DCOPCStringList::Iterator it = apps.begin(); it != apps.end(); ++it )
      {
        if ((*it).indexOf("ligature") == 0)
        {
          QByteArray data, replyData;
          DCOPCString replyType;
          QDataStream arg( &data,QIODevice::WriteOnly);
          arg.setVersion(QDataStream::Qt_3_1);
          bool result;
          arg << qualPath.trimmed();
          if (!app.dcopClient()->call( *it, "ligaturePluginGUI", "is_file_loaded(QString)", data, replyType, replyData))
            kError(kvs::shell) << "There was an error using DCOP." << endl;
          else
          {
            QDataStream reply( &replyData,QIODevice::ReadOnly);
            reply.setVersion(QDataStream::Qt_3_1);
            if (replyType == "bool")
            {
              reply >> result;
              if (result == true)
              {
                if (app.dcopClient()->send(*it, "ligaturePluginGUI", "jumpToReference(QString)", args->url(0).ref()) == true)
                {
                  app.dcopClient()->detach();
                  return 0;
                }
              }
            }
            else
            {
              kError(kvs::shell) << "The DCOP function 'doIt' returned an unexpected type of reply!";
            }
          }
        }
      }
#endif
    }
    Ligature* shell;

    if (args->isSet("mimetype"))
    {
      shell = new Ligature(args->getOption("mimetype"));
    }
    else if (args->count() > 0)
    {
      // If a url is given, we try to load a matching Ligature plugin,
      // so we don't have to load the empty plugin first.
      KMimeType::Ptr mimetype = KMimeType::findByUrl(args->url(0));
      shell = new Ligature(mimetype->name());
    }
    else
    {
      // Load the empty plugin
      shell = new Ligature();
    }
    QDBusConnection::sessionBus().registerObject("/ligature", shell);

    // Show the main window before a file is loaded. This gives visual
    // feedback to the user and (hopefully) reduces the perceived
    // startup time.
    shell->show();
    app.processEvents();

    if ( args->count() > 0 )
    {
      if (QString(args->arg(0)) == "-" )
      {
        shell->openStdin();
      }
      else
      {
        KUrl url = args->url(0);
        if (!url.hasRef() && args->isSet("goto"))
        {
          // If the url doesn't already has a reference part, add the
          // argument of --goto to the url as reference, to make the
          // Ligature jump to this page.
          QString reference = args->getOption("goto");
          url.setHTMLRef(reference);
        }
        shell->openUrl(url);
      }
    }
  }

  return app.exec();
}
