/* This file is part of the KDE libraries
 *  Copyright (C) 2000 Yves Arrouye <yves@realnames.com>
 *  Copyright (C) 2000 Dawit Alemayehu <adawit at kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 **/

#include <config.h>

#include <kdebug.h>
#include <kiconloader.h>
#include <kservicetypetrader.h>
#include <kmimetype.h>
#include <klibloader.h>
#include <kstaticdeleter.h>

#include "kurifilter.h"

template class Q3PtrList<KURIFilterPlugin>;

#ifdef KDE3_SUPPORT
KURIFilterPlugin::KURIFilterPlugin( QObject *parent, const char *name, double pri )
#else
KURIFilterPlugin::KURIFilterPlugin( const QString & name, QObject *parent, double pri )
#endif
                 :QObject( parent ), m_strName( name )
{
    m_dblPriority = pri;
}

void KURIFilterPlugin::setFilteredURI( KURIFilterData& data, const KUrl& uri ) const
{
    if ( data.uri() != uri )
    {
        data.m_pURI = uri;
        data.m_bChanged = true;
    }
}

class KURIFilterDataPrivate
{
public:
    KURIFilterDataPrivate() {};
    QString abs_path;
    QString args;
    QString typedString;
};

KURIFilterData::KURIFilterData( const KURIFilterData& data )
{
    m_iType = data.m_iType;
    m_pURI = data.m_pURI;
    m_strErrMsg = data.m_strErrMsg;
    m_strIconName = data.m_strIconName;
    m_bChanged = data.m_bChanged;
    m_bCheckForExecutables = data.m_bCheckForExecutables;
    d = new KURIFilterDataPrivate;
    d->abs_path = data.absolutePath();
    d->typedString = data.typedString();
    d->args = data.argsAndOptions();
}

KURIFilterData::~KURIFilterData()
{
    delete d;
    d = 0;
}

void KURIFilterData::init( const KUrl& url )
{
    m_iType = KURIFilterData::UNKNOWN;
    m_pURI = url;
    m_strErrMsg.clear();
    m_strIconName.clear();
    m_bCheckForExecutables = true;
    m_bChanged = true;
    d = new KURIFilterDataPrivate;
    d->typedString = url.url();
}

void KURIFilterData::init( const QString& url )
{
    m_iType = KURIFilterData::UNKNOWN;
    m_pURI = url; // QString -> KUrl
    m_strErrMsg.clear();
    m_strIconName.clear();
    m_bCheckForExecutables = true;
    m_bChanged = true;
    d = new KURIFilterDataPrivate;
    d->typedString = url;
}

QString KURIFilterData::typedString() const
{
    return d->typedString;
}

void KURIFilterData::setCheckForExecutables( bool check )
{
    m_bCheckForExecutables = check;
}

bool KURIFilterData::hasArgsAndOptions() const
{
    return !d->args.isEmpty();
}

bool KURIFilterData::hasAbsolutePath() const
{
    return !d->abs_path.isEmpty();
}

bool KURIFilterData::setAbsolutePath( const QString& absPath )
{
    // Since a malformed URL could possibly be a relative
    // URL we tag it as a possible local resource...
    if( (m_pURI.protocol().isEmpty() || m_pURI.isLocalFile()) )
    {
        d->abs_path = absPath;
        return true;
    }
    return false;
}

QString KURIFilterData::absolutePath() const
{
    return d->abs_path;
}

QString KURIFilterData::argsAndOptions() const
{
    return d->args;
}

QString KURIFilterData::iconName()
{
    if( m_bChanged )
    {
        switch ( m_iType )
        {
            case KURIFilterData::LOCAL_FILE:
            case KURIFilterData::LOCAL_DIR:
            case KURIFilterData::NET_PROTOCOL:
            {
                m_strIconName = KMimeType::iconNameForURL( m_pURI );
                break;
            }
            case KURIFilterData::EXECUTABLE:
            {
                QString exeName = m_pURI.url();
                exeName = exeName.mid( exeName.lastIndexOf( '/' ) + 1 ); // strip path if given
                KService::Ptr service = KService::serviceByDesktopName( exeName );
                if (service && service->icon() != QLatin1String( "unknown" ))
                    m_strIconName = service->icon();
                // Try to find an icon with the same name as the binary (useful for non-kde apps)
                else if ( !KGlobal::iconLoader()->loadIcon( exeName, K3Icon::NoGroup, 16, K3Icon::DefaultState, 0, true ).isNull() )
                    m_strIconName = exeName;
                else
                    // not found, use default
                    m_strIconName = QLatin1String("exec");
                break;
            }
            case KURIFilterData::HELP:
            {
                m_strIconName = QLatin1String("khelpcenter");
                break;
            }
            case KURIFilterData::SHELL:
            {
                m_strIconName = QLatin1String("konsole");
                break;
            }
            case KURIFilterData::ERROR:
            case KURIFilterData::BLOCKED:
            {
                m_strIconName = QLatin1String("error");
                break;
            }
            default:
                m_strIconName.clear();
                break;
        }
        m_bChanged = false;
    }
    return m_strIconName;
}

//********************************************  KURIFilterPlugin **********************************************
void KURIFilterPlugin::setArguments( KURIFilterData& data, const QString& args ) const
{
    data.d->args = args;
}

//********************************************  KURIFilter **********************************************
KURIFilter *KURIFilter::m_self;
static KStaticDeleter<KURIFilter> kurifiltersd;

KURIFilter *KURIFilter::self()
{
    if (!m_self)
        m_self = kurifiltersd.setObject(m_self, new KURIFilter);
    return m_self;
}

KURIFilter::KURIFilter()
{
    m_lstPlugins.setAutoDelete(true);
    loadPlugins();
}

KURIFilter::~KURIFilter()
{
    m_self = 0;
}

bool KURIFilter::filterURI( KURIFilterData& data, const QStringList& filters )
{
    bool filtered = false;
    KURIFilterPluginList use_plugins;

    // If we have a filter list, only include the once
    // explicitly specified by it. Otherwise, use all available filters...
    if( filters.isEmpty() )
        use_plugins = m_lstPlugins;  // Use everything that is loaded...
    else
    {
        //kDebug() << "Named plugins requested..."  << endl;
        for( QStringList::ConstIterator lst = filters.begin(); lst != filters.end(); ++lst )
        {
            Q3PtrListIterator<KURIFilterPlugin> it( m_lstPlugins );
            for( ; it.current() ; ++it )
            {
                if( (*lst) == it.current()->name() )
                {
                    //kDebug() << "Will use filter plugin named: " << it.current()->name() << endl;
                    use_plugins.append( it.current() );
                    break;  // We already found it ; so lets test the next named filter...
                }
            }
        }
    }

    Q3PtrListIterator<KURIFilterPlugin> it( use_plugins );
    //kDebug() << "Using " << use_plugins.count() << " out of the "
    //          << m_lstPlugins.count() << " available plugins" << endl;
    for (; it.current() && !filtered; ++it)
    {
        //kDebug() << "Using a filter plugin named: " << it.current()->name() << endl;
        filtered |= it.current()->filterURI( data );
    }
    return filtered;
}

bool KURIFilter::filterURI( KUrl& uri, const QStringList& filters )
{
    KURIFilterData data = uri;
    bool filtered = filterURI( data, filters );
    if( filtered ) uri = data.uri();
    return filtered;
}

bool KURIFilter::filterURI( QString& uri, const QStringList& filters )
{
    KURIFilterData data = uri;
    bool filtered = filterURI( data, filters );
    if( filtered )  uri = data.uri().url();
    return filtered;

}

KUrl KURIFilter::filteredURI( const KUrl &uri, const QStringList& filters )
{
    KURIFilterData data = uri;
    filterURI( data, filters );
    return data.uri();
}

QString KURIFilter::filteredURI( const QString &uri, const QStringList& filters )
{
    KURIFilterData data = uri;
    filterURI( data, filters );
    return data.uri().url();
}

Q3PtrListIterator<KURIFilterPlugin> KURIFilter::pluginsIterator() const
{
    return Q3PtrListIterator<KURIFilterPlugin>(m_lstPlugins);
}

QStringList KURIFilter::pluginNames() const
{
    QStringList list;
    for(Q3PtrListIterator<KURIFilterPlugin> i = pluginsIterator(); *i; ++i)
        list.append((*i)->name());
    return list;
}

void KURIFilter::loadPlugins()
{
    const KService::List offers = KServiceTypeTrader::self()->query( "KURIFilter/Plugin" );

    KService::List::ConstIterator it = offers.begin();
    const KService::List::ConstIterator end = offers.end();

    for (; it != end; ++it )
    {
      KURIFilterPlugin *plugin = KService::createInstance<KURIFilterPlugin>( *it );

      if ( plugin ) {
        plugin->setObjectName( (*it)->desktopEntryName() );
        m_lstPlugins.append( plugin );
      }
    }

    // NOTE: Plugin priority is now determined by
    // the entry in the .desktop files...
    // TODO: Config dialog to differentiate "system"
    // plugins from "user-defined" ones...
    // m_lstPlugins.sort();
}

void KURIFilterPlugin::virtual_hook( int, void* )
{ /*BASE::virtual_hook( id, data );*/ }

#include "kurifilter.moc"
