/*
 *  This file is part of the KDE libraries
 *  Copyright (c) 2001 Michael Goffioul <kdeprint@swing.be>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License version 2 as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 **/

#include "cupsdcomment.h"

#include <QtCore/QFile>
#include <QtCore/QRegExp>
#include <klocale.h>
#include <kstandarddirs.h>

QString Comment::comment() const
{
        QString str = comment_;
        str.replace(QRegExp("<[^>]*>"), "");
        str += ("#\n" + example_);
        return str;
}

QString Comment::toolTip() const
{
        QString str = comment_;
        str.replace(QRegExp("^#[\\s]*"), "").replace(QRegExp("\n#[\\s]*"), "\n");
        return i18nc("Do not translate the keyword between brackets (e.g. ServerName, ServerAdmin, etc.)", str.toUtf8());
}

QString Comment::key() const
{
	return key_;
}

bool Comment::load(QFile *f)
{
        comment_ = "";
        example_ = "";
	key_ = "";
        QByteArray line;
	line.resize(1024);
	QString *current = &comment_;
        while (!f->atEnd())
        {
                f->readLine(line.data(), 1024);
                if (line.startsWith("$$"))
                {
                        current = &example_;
                }
		else if (line.startsWith("%%"))
		{
			key_ = line.mid(2).trimmed();
		}
                else if (line.startsWith("@@"))
                {
                        return true;
                }
                else if (line.trimmed().isEmpty())
                {
                        ; // do nothing
                }
                else
                {
                        if (line[0] != '#') break;
                        else
                        {
                                current->append(QString::fromLatin1(line)); // encoding ok?
                        }
                }
        }
        return false;
}

//------------------------------------------------------------------------------------------------------------

CupsdComment::~CupsdComment()
{
	qDeleteAll(comments_);
}

QString CupsdComment::operator[] (const QString& key)
{
        return comment(key);
}

QString CupsdComment::comment(const QString& key)
{
        if (!comments_.isEmpty() || loadComments())
	{
		Comment *comm = comments_.value(key);
		if (comm)
			return comm->comment();
	}
        return QString();
}

QString CupsdComment::toolTip(const QString& key)
{
        if (!comments_.isEmpty() || loadComments())
	{
		Comment *comm = comments_.value(key);
		if (comm)
			return comm->toolTip();
	}
        return QString();
}

bool CupsdComment::loadComments()
{
        qDeleteAll(comments_);
        comments_.clear();
        QFile f(KStandardDirs::locate("data", "kdeprint/cupsd.conf.template"));
	if (f.exists() && f.open(QIODevice::ReadOnly))
	{
                Comment         *comm;
                while (!f.atEnd())
                {
                        comm = new Comment();
                        if (!comm->load(&f))
                                break;
                        else
                        {
				if (comm->key().isEmpty())
					delete comm;
				else
					comments_.insert(comm->key(), comm);
                        }
                }
	}
        return true;
}
