/*
    Copyright (C) 2001, S.R.Haque <srhaque@iee.org>.
    Copyright (C) 2002, David Faure <david@mandrakesoft.com>
    This file is part of the KDE project

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2, as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef KFINDDIALOG_H
#define KFINDDIALOG_H

#include <kdialog.h>

class KHistoryComboBox;
class QPushButton;
class QMenu;
class QGridLayout;
class QCheckBox;
class QGroupBox;

/**
 * @brief A generic "find" dialog.
 *
 * @author S.R.Haque <srhaque@iee.org>
 *
 * \b Detail:
 *
 * This widget inherits from KDialog and implements
 * the following additional functionalities:  a find string
 * object and an area for a user-defined widget to extend the dialog.
 *
 * \b Example:
 *
 * To use the basic modal find dialog, and then run the search:
 *
 * \code
 *  KFindDialog dlg(....)
 *  if ( dlg.exec() != QDialog::Accepted )
 *      return;
 *
 *  // proceed with KFind from here
 * \endcode
 *
 * To create a non-modal find dialog:
 * \code
 *   if ( m_findDia )
 *     KWindowSystem::activateWindow( m_findDia->winId() );
 *   else
 *   {
 *     m_findDia = new KFindDialog(false,...);
 *     connect( m_findDia, SIGNAL(okClicked()), this, SLOT(findTextNext()) );
 *   }
 * \endcode
 * Don't forget to delete and reset m_findDia when closed.
 * (But do NOT delete your KFind object at that point, it's needed for "Find Next")
 *
 * To use your own extensions: see findExtension().
 *
 */
class KDEUI_EXPORT KFindDialog:
    public KDialog
{
    Q_OBJECT

public:


    /**
     * Construct a modal find dialog
     *
     * @param parent The parent object of this widget.
     * @param options A bitfield of the Options to be checked.
     * @param findStrings The find history, see findHistory()
     * @param hasSelection Whether a selection exists
     */
    explicit KFindDialog( QWidget *parent = 0, long options = 0,
                 const QStringList &findStrings = QStringList(), bool hasSelection = false );

    /**
     * Destructor.
     */
    virtual ~KFindDialog();

    /**
     * Provide the list of @p strings to be displayed as the history
     * of find strings. @p strings might get truncated if it is
     * too long.
     *
     * @param history The find history.
     * @see findHistory
     */
    void setFindHistory( const QStringList &history );

    /**
     * Returns the list of history items.
     *
     * @see setFindHistory
     */
    QStringList findHistory() const;

    /**
     * Enable/disable the 'search in selection' option, depending
     * on whether there actually is a selection.
     *
     * @param hasSelection true if a selection exists
     */
    void setHasSelection( bool hasSelection );

    /**
     * Hide/show the 'from cursor' option, depending
     * on whether the application implements a cursor.
     *
     * @param hasCursor true if the application features a cursor
     * This is assumed to be the case by default.
     */
    void setHasCursor( bool hasCursor );

     /**
     * Enable/disable the 'Find backwards' option, depending
     * on whether the application supports it.
     *
     * @param supports true if the application supports backwards find
     * This is assumed to be the case by default.
     */
    void setSupportsBackwardsFind( bool supports );

     /**
     * Enable/disable the 'Case sensitive' option, depending
     * on whether the application supports it.
     *
     * @param supports true if the application supports case sensitive find
     * This is assumed to be the case by default.
     */
    void setSupportsCaseSensitiveFind( bool supports );

     /**
     * Enable/disable the 'Whole words only' option, depending
     * on whether the application supports it.
     *
     * @param supports true if the application supports whole words only find
     * This is assumed to be the case by default.
     */
    void setSupportsWholeWordsFind( bool supports );

     /**
     * Enable/disable the 'Regular expression' option, depending
     * on whether the application supports it.
     *
     * @param supports true if the application supports regular expression find
     * This is assumed to be the case by default.
     */
    void setSupportsRegularExpressionFind( bool supports );

    /**
     * Set the options which are checked.
     *
     * @param options The setting of the Options.
     *
     * @see options()
     * @see KFind::Options
     */
    void setOptions( long options );

    /**
     * Returns the state of the options. Disabled options may be returned in
     * an indeterminate state.
     *
     * @see setOptions()
     * @see KFind::Options
     */
    long options() const;

    /**
     * Returns the pattern to find.
     */
    QString pattern() const;

    /**
     * Sets the pattern to find
     */
    void setPattern ( const QString &pattern );

    /**
     * Returns an empty widget which the user may fill with additional UI
     * elements as required. The widget occupies the width of the dialog,
     * and is positioned immediately below the regular expression support
     * widgets for the pattern string.
     */
    QWidget *findExtension() const;

Q_SIGNALS:
    /**
     * This signal is sent whenever one of the option checkboxes is toggled.
     * Call options() to get the new state of the checkboxes.
     */
     void optionsChanged();

protected Q_SLOTS:

    void slotOk();
    void slotSelectedTextToggled(bool);
    void showPatterns();
    void showPlaceholders();
    void textSearchChanged( const QString &);

protected:
    virtual void showEvent ( QShowEvent * );

private Q_SLOTS:
    /**
     * connected to the aboutToShow of the placeholders menu,
     * updates it according to the text in the pattern.
     */
    void slotPlaceholdersAboutToShow();

private:

    // Our dirty little secret is that we also implement the "replace" dialog. But we
    // keep that fact hidden from all but our friends.

    friend class KReplaceDialog;


    void init( bool forReplace, const QStringList &findStrings, bool hasSelection );

    class KFindDialogPrivate;
    KFindDialogPrivate* const d;
};

#endif // KFINDDIALOG_H
