/* This file is part of Noatun

  Copyright 2000-2006 by Charles Samuels <charles@kde.org>
  Copyright 2001-2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "noatun/player.h"
#include "noatun/global.h"
#include "noatun/pluginloader.h"
#include "noatun/playlist.h"
#include "noatun/plugininterfaces.h"
#include "noatun/plugin.h"
#include "noatun/config.h"

#include "hardwarevc.h"

#include <qregexp.h>
#include <qtimer.h>

#include <kdebug.h>
#include <kurl.h>

namespace Noatun
{

struct Player::Private
{
	static const int UPDATE_RATE = 100;

	EngineInterface     *engine;
	ItemSourceInterface *itemSource; // TODO: remove
	Global              *nInstance;
	QTimer              *filePos;
	int                  position;
	PlaylistItem         currentItem;
	HardwareVC          *hwVolumeControl;
	unsigned int         lastKnownVolume;
};


// -----------------------------------------------------------------------------


Player::Player(Global *nInstance) : QObject(nInstance), d(new Player::Private())
{
	setObjectName("Player");

	d->engine          = 0;
	d->itemSource      = 0;
	d->nInstance       = nInstance;
	d->filePos         = new QTimer(this);
	d->position        = 0;
	//d->currentItem     = PlaylistItem();
	d->hwVolumeControl = 0;
	d->lastKnownVolume = 100; // TODO: read from config

	connect(d->filePos, SIGNAL(timeout()), SLOT(updatePosition()));

	connect(d->nInstance->pluginHandler(), SIGNAL(pluginLoaded(Plugin *))  , SLOT(pluginLoaded(Plugin *)));
	connect(d->nInstance->pluginHandler(), SIGNAL(pluginUnloaded(Plugin *)), SLOT(pluginUnloaded(Plugin *)));
}


Player::~Player()
{
	kDebug(66666) << k_funcinfo << endl;
	delete d;
}


PlaylistItem Player::currentItem() const
{
	return d->currentItem;
}


Player::State Player::state() const
{
	if (!d->engine)
		return StoppedState;
	return d->engine->state();
}


bool Player::isPlaying() const
{
	return state() == Player::PlayingState;
}


bool Player::isPaused() const
{
	return state() == Player::PausedState;
}


bool Player::isStopped() const
{
	return state() == Player::StoppedState;
}


bool Player::isActive() const
{
	Player::State s = state();
	return s == Player::PausedState || s == Player::PlayingState;
}


void Player::stop()
{
	if (isStopped())
		return;
	d->engine->stop();
}


void Player::pause()
{
	if (!isPlaying())
		return;
	d->engine->pause();
}


void Player::play()
{
	if (isPlaying())
		return;
	play(d->currentItem);
}


void Player::play(const PlaylistItem &item)
{
	kDebug(66666) << k_funcinfo << "Attempting to play..." << endl;
	if (item.isNull() || !d->itemSource || !d->engine)
		return;

	if (isPaused() && d->currentItem == item)
	{
		kDebug(66666) << k_funcinfo << "Unpausing current track" << endl;
		d->engine->play(KUrl()); // this actually unpauses, FIXME: ugly code
	}
	else
	{
		//stop(); //TODO: check if this is needed?
		kDebug(66666) << k_funcinfo << "Starting to play new current track" << endl;
		if (d->engine->play(item.url()))
		{
			//TODO: why do we have a "current" item in two places?
			d->currentItem = item;
			d->itemSource->setCurrent(item);
			emit currentItemChanged(d->currentItem);
		}
		else
		{
			// TODO: this situation is probably similar to engineErrorOccurred()
		}
	}
}


void Player::playpause()
{
	if (isPlaying())
		pause();
	else
		play();
}


void Player::next()
{
	if (d->itemSource)
	{
		PlaylistItem item = d->itemSource->forward();
		play(item);
	}
	else
		kWarning(66666) << "Missing an itemsource for switching to next item" << endl;
}


void Player::previous()
{
	if (d->itemSource)
	{
		PlaylistItem item = d->itemSource->backward();
		play(item);
	}
	else
		kWarning(66666) << "Missing an itemsource for switching to previous item" << endl;
}


void Player::setPosition(int msec)
{
	if (d->engine && isActive() && msec >= 0)
	{
		//TODO: is it a good idea to add some milliseconds in here to honor
		//      missing ui-accuracy?
		d->engine->setPosition(msec + (Private::UPDATE_RATE / 2));
		updatePosition(); // update position and emit signal
	}
}

int Player::position() const
{
	return d->position;
}

int Player::currentLength() const
{
	///TODO decide if we should take length from engine or playlist
	/// playlist: + available even if the engine didn't load the file (yet),
	///           - can differ from length that gets played by engine
	/// engine  : + engine usually knows best how long a file will play
	///           - not sure if length can be determined right after playing
	///             started or if length is correct paused/stopped state

#if 0
	if (d->currentItem.isNull())
		return -1;
	return d->currentItem.length(); // return track-length in milliseconds
#else
	if (!d->engine)
		return -1;
	return d->engine->length();
#endif
}

QString Player::lengthString() const
{
	if (d->currentItem.isNull())
		return QString();
	return d->currentItem.lengthString();
}


QString Player::positionString(int pos) const
{
	QString posString;

	if (pos == -1)
		pos = position();

	if (pos < 0)
	{
		posString = "--:--";
	}
	else
	{
		// remaining time is only possible if
		// - display mode is "remaining"
		// - we have something playing
		// - it has a valid length
		int len = currentLength();
		bool remain = false; //napp->displayRemaining() && current() && (len >= 0);
		if (remain)
			pos = len - pos;

		pos /= 1000; // convert from milliseconds to seconds
		int seconds = pos % 60;
		int minutes = (pos - seconds) / 60;

		// the string has to look like '00:00/00:00'
		// TODO: How to make use of KLocale?
		if (remain)
			posString.sprintf("-%.2d:%.2d", minutes, seconds);
		else
			posString.sprintf("%.2d:%.2d", minutes, seconds);
	}
	return posString;
}


QString Player::timeString(int pos) const
{
	// TODO: i18n?
	return lengthString() + '/' + positionString(pos);
}


void Player::updateVolumeHandler() const
{
	if (d->nInstance->config()->useHardwareMixer())
	{
		if (!d->hwVolumeControl) // create needed hardware-mixer
		{
			d->hwVolumeControl = new HardwareVC();
			if (d->engine) // make sure software-volume is reset
				d->engine->setSoftwareVolume(100);
		}
	}
	else // use software mixing
	{
		if (d->hwVolumeControl) // get rid of unneeded hardware-mixer
		{
			delete d->hwVolumeControl;
			d->hwVolumeControl = 0;
		}
	}
}

unsigned int Player::volume() const
{
	updateVolumeHandler(); //TODO do not call that from here

#ifdef HARDWARE_VOLUME
	if (d->nInstance->config()->useHardwareMixer() && d->hwVolumeControl)
		return d->hwVolumeControl->volume();
#endif
	if (d->engine)
		return d->engine->softwareVolume();
	return d->lastKnownVolume;
}


void Player::setVolume(unsigned int percent)
{
	percent = qBound(percent, 0u, 100u);

	d->lastKnownVolume = percent;

#ifdef HARDWARE_VOLUME
	if (d->nInstance->config()->useHardwareMixer())
	{
		d->hwVolumeControl->setVolume(percent);
	}
	else
#endif
	{
		if (d->engine)
			d->engine->setSoftwareVolume(percent);
	}

	emit volumeChanged(percent);
}


QStringList Player::mimeTypes() const
{
	if (d->engine)
		return d->engine->mimeTypes();
	return QStringList();
}


void Player::updatePosition()
{
	if (d->engine)
	{
		int newPos = d->engine->position(); // use qMax() to avoid the < 0 case?
		if (newPos != d->position && newPos >= 0)
		{
			//kDebug(66666) << k_funcinfo << "newPos " << newPos << endl;
			d->position = newPos;
			emit positionChanged(d->position);
		}
	}
	else
		kWarning(66666) << k_funcinfo << "called with no engine loaded" << endl;
}


void Player::engineStateChanged(Player::State newState)
{
	kDebug(66666) << k_funcinfo << "new state is " << newState << endl;

	switch (newState)
	{
	case Player::PlayingState:
		d->position = d->engine->position(); // ensure correct position
		d->filePos->start(Private::UPDATE_RATE);
		emit playing();
		break;
	case Player::StoppedState:
		d->position = 0;
		d->filePos->stop();
		emit stopped();
		break;
	case Player::PausedState:
		d->position = d->engine->position(); // ensure correct position
		d->filePos->stop();
		emit paused();
		break;
	}

	emit stateChanged(newState);
}


void Player::engineErrorOccurred(Player::ErrorType errorType, const QString &errorDescription)
{
	kDebug(66666) << k_funcinfo <<
		(errorType == Player::NormalError ? "Normal" : "FATAL") << " error: " <<
		errorDescription << endl;
	kDebug(66666) << k_funcinfo << "player state is now " << state() << endl;

	//TODO decide if we should advance, stop or maybe warn
	//     the user about this error?
	//next();
}


void Player::enginePlaybackFinished()
{
	kDebug(66666) << k_funcinfo << endl;
	stop(); // just in case the engine stays in playing state
	QTimer::singleShot(0, this, SLOT(next())); // delaying this is helpful for phonon-xine
}


void Player::pluginLoaded(Plugin *plug)
{
	if (ItemSourceInterface *f = plug->interface<ItemSourceInterface>())
	{
		kDebug(66666) << k_funcinfo << "New plugin with ItemSourceInterface" << endl;
		d->itemSource = f;
	}
	else if (EngineInterface *f = plug->interface<EngineInterface>())
	{
		kDebug(66666) << k_funcinfo << "New plugin with EngineInterface" << endl;
		d->engine = f;
		d->engine->setSoftwareVolume(d->lastKnownVolume);
	}
}


void Player::pluginUnloaded(Plugin *plug)
{
	if (ItemSourceInterface *f = plug->interface<ItemSourceInterface>())
	{
		if (d->itemSource == f)
		{
			kDebug(66666) << k_funcinfo << "Plugin with ItemSourceInterface gone" << endl;
			stop();
			d->currentItem = PlaylistItem();
			emit currentItemChanged(d->currentItem);
			d->itemSource = 0;
		}
	}
	else if (EngineInterface *f = plug->interface<EngineInterface>())
	{
		if (d->engine == f)
		{
			kDebug(66666) << k_funcinfo << "Plugin with EngineInterface gone" << endl;
			d->engine = 0;
		}
	}
}


QString Player::currentTitle() const
{
	return title(d->currentItem);
}


QString Player::title(PlaylistItem item) const
{
	if (item.isNull())
		return QString();

	if (item.hasProperty("realtitle"))
		return item.property("realtitle");

	// "$(property)"
	QString format = d->nInstance->config()->titleFormat();

	QRegExp find("(?:(?:\\\\\\\\))*\\$\\((.*)");

	int start = 0;
	while (start != -1)
	{
		start = find.indexIn(format, start);
		if (start == -1)
			break;

		// test if there's an odd amount of backslashes
		if (start > 0 && format[start-1] == '\\')
		{
			// yes, so half the amount of backslashes

			// count how many there are first
			QRegExp counter("([\\\\]+)");
			counter.indexIn(format, start - 1);
			uint len = counter.cap(1).length() - 1;

			// and half them, and remove one more
			format.replace(start-1, len/2+1, "");
			start = start - 1 + len / 2 + find.cap(1).length() + 3;
			continue;
		}

		// now replace the backslashes with half as many
		if (format[start] == '\\')
		{
			// count how many there are first
			QRegExp counter("([\\\\]+)");
			counter.indexIn(format, start);
			uint len = counter.cap(1).length();

			// and half them
			format.replace(start, len / 2, "");
			start = start + len / 2;
		}

		// "sth"foo"sth"
		QString cont(find.cap(1));
		QString prefix, suffix, propname;
		int i = 0;
		if (cont[i] == '"')
		{
			i++;
			for (; i < cont.length(); i++)
			{
				if (cont[i] != '"')
					prefix += cont[i];
				else
					break;
			}
			i++;
		}


		for (; i < cont.length(); ++i)
		{
			if (cont[i]!='"' && cont[i]!=')')
				propname += cont[i];
			else
				break;
		}

		if (cont[i] == '"')
		{
			i++;
			for (; i < cont.length(); i++)
			{
				if (cont[i] != '"')
					suffix += cont[i];
				else
					break;
			}
			i++;
		}
		i++;


		QString propval = item.property(propname);
		if (propname == "title" && propval.isEmpty())
		{
			propval = item.url().fileName();
		}
		else if (propname == "comment")
		{
			// comments can contain newlines
			// these are not wanted in a formatted title
			propval.replace('\n', ' ');
		}

		if (propval.length() > 0)
		{
			propval = prefix + propval + suffix;
			format.replace(start, i+2, propval);
			start += propval.length();
		}
		else
		{
			format.replace(start, i+2, "");
		}
	} // while()
	return format;
} // QString title(PlaylistItem item) const

} // namespace Noatun

#include "player.moc"
