/* This file is part of Noatun

  Copyright 2005-2006 by Charles Samuels <charles@kde.org>
  Copyright 2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include <noatun/playlistitem.h>
#include <noatun/playlist.h>
#include <noatun/player.h>
#include <noatun/global.h>
#include <noatun/plugininterfaces.h>

#include <kdebug.h>

//#include <iostream>

namespace Noatun
{

struct PlaylistItem::PlaylistItemShared
{
	int refs;
	void *obj;
	PlaylistInterfaceGeneric *playlist;
};

void *PlaylistItem::obj() const
{
	if (shared) return shared->obj;
	return 0;
}

PlaylistItem PlaylistItem::fromGeneric(PlaylistInterfaceGeneric *playlist, void *source)
{
	PlaylistItem item;
	if (source)
	{
		item.shared = new PlaylistItemShared;
		item.shared->refs = 1;
		item.shared->obj = source;
		item.shared->playlist = playlist;
	}

	return item;
}

PlaylistItem::PlaylistItem(const PlaylistItem &source)
{
	shared = source.shared;
	addRef();
}

PlaylistItem::PlaylistItem()
{
	shared = 0;
}

PlaylistItem::~PlaylistItem()
{
	removeRef();
}

PlaylistItem &PlaylistItem::operator =(const PlaylistItem &source)
{
	if (source)
		source.addRef();
	removeRef();
	shared = source.shared;
	return *this;
}

const PlaylistItem &PlaylistItem::operator =(const PlaylistItem &source) const
{
	if (source)
		source.addRef();
	removeRef();
	shared = source.shared;
	return *this;
}

PlaylistItem::operator bool() const
{
	return shared != 0;
}

bool PlaylistItem::isNull() const
{
	return (shared == 0);
}

bool PlaylistItem::operator ==(const PlaylistItem &i) const
{
	if (shared==i.shared)
		return true;
	if (!shared || !i.shared)
		return false;

	return shared->playlist->compare_g(i.shared->obj, shared->obj);
}

bool PlaylistItem::operator !=(const PlaylistItem &i) const
{
	return ! (*this==i);
}

QString PlaylistItem::property(const QString &key, const QString &def) const
{
	return shared->playlist->property_g(shared->obj, key, def);
}

void PlaylistItem::setProperty(const QString &key, const QString &property)
{
	shared->playlist->setProperty_g(shared->obj, key, property);
}

void PlaylistItem::setProperties(const PropertyMap &properties)
{
	shared->playlist->setProperties_g(shared->obj, properties);
}

void PlaylistItem::clearProperty(const QString &key)
{
	shared->playlist->clearProperty_g(shared->obj, key);
}

QStringList PlaylistItem::properties() const
{
	return shared->playlist->properties_g(shared->obj);
}

bool PlaylistItem::hasProperty(const QString &key) const
{
	return shared->playlist->hasProperty_g(shared->obj, key);
}

KUrl PlaylistItem::url() const
{
	return shared->playlist->url_g(shared->obj);
}

void PlaylistItem::setUrl(const KUrl &url)
{
	shared->playlist->setUrl_g(shared->obj, url);
}

QString PlaylistItem::mimetype() const
{
	return shared->playlist->mimetype_g(shared->obj);
}

int PlaylistItem::length() const
{
	return shared->playlist->length_g(shared->obj);
}

void PlaylistItem::setLength(int ms) const
{
	shared->playlist->setLength_g(shared->obj, ms);
}

QString PlaylistItem::lengthString() const
{
	return shared->playlist->global()->player()->positionString(length());
}

void PlaylistItem::remove()
{
	shared->playlist->remove_g(shared->obj);
}

bool PlaylistItem::streamable() const
{
	return shared->playlist->streamable_g(shared->obj);
}

void PlaylistItem::added(Global *global)
{
	QList<PlaylistNotifier *> *list = global->playlist()->notifiers();
	QList<PlaylistNotifier *>::Iterator it;
	for (it = list->begin(); it != list->end(); ++it)
		(*it)->added(*this);
}

void PlaylistItem::removed(Global *global)
{
	QList<PlaylistNotifier *> *list = global->playlist()->notifiers();
	QList<PlaylistNotifier *>::Iterator it;
	for (it = list->begin(); it != list->end(); ++it)
		(*it)->removed(*this);
}

void PlaylistItem::modified(Global *global)
{
	QList<PlaylistNotifier *> *list = global->playlist()->notifiers();
	QList<PlaylistNotifier *>::Iterator it;
	for (it = list->begin(); it != list->end(); ++it)
		(*it)->modified(*this);
}


void PlaylistItem::removeRef() const
{
	if (shared)
	{
		shared->refs--;
		if (shared->refs == 0)
		{
			shared->playlist->deleteItem_g(shared->obj);
			delete shared;
			shared = 0;
		}
	}

}

void PlaylistItem::addRef() const
{
	if (shared)
		shared->refs++;
}

} // namespace Noatun
