// userinterface.cpp
//
// Copyright 2001 Neil Stevens <neil@qualityassistant.com>
// Copyright 1999 Charles Samuels <charles@kde.org>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
// AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name(s) of the author(s) shall not be
// used in advertising or otherwise to promote the sale, use or other dealings
// in this Software without prior written authorization from the author(s).

//#ifdef HAVE_CONFIG_H
//#include <config.h>
//#endif

#include <noatun/effects.h>
#include <noatun/app.h>
#include <noatun/controls.h>
#include <noatun/pref.h>
#include <noatun/player.h>
#include <noatun/stdaction.h>
#include <noatun/global.h>
#include <noatun/nmainwindow.h>

#include "userinterface.h"

#include <kconfig.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <kpopupmenu.h>
#include <kstatusbar.h>
#include <kstandardaction.h>
#include <kwin.h>
#include <kurldrag.h>

#include <qdragobject.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qobjectlist.h>
#include <qvbox.h>

#include <qdom.h>
#include <kgenericfactory.h>

using namespace Noatun;

K_EXPORT_COMPONENT_FACTORY(noatun_excellent, PluginFactory<ExcellentPlugin>("noatun_excellent"))


ExcellentPlugin::ExcellentPlugin(const KComponentData &instance, Global *parent, const char* name)
	: Plugin(instance, parent, name)
{
	kDebug(66666) << k_funcinfo << endl;
	mWin = new ExcellentMainWindow(this);
}

ExcellentPlugin::~ExcellentPlugin()
{
	kDebug(66666) << k_funcinfo << endl;
	delete mWin;
}

Interface *ExcellentPlugin::getInterface(const QString &interface)
{
	if (interface == "FrontendInterface")
		return this;
	return 0;
}

void ExcellentPlugin::init()
{
	kDebug(66666) << k_funcinfo << endl;
	mWin->initGUI();
}

KMainWindow *ExcellentPlugin::mainWindow()
{
	return mWin;
}

// ----------------------------------------------------------------------------


ExcellentMainWindow::ExcellentMainWindow(Plugin *plug)
	: NMainWindow(plug, false, 0, "NoatunExcellentMainWindow"), mPlugin(plug)
{
	kDebug(66666) << k_funcinfo << endl;

	setCaption(i18n("Noatun"));
	setIcon(SmallIcon("noatun"));

	// Who needs Qt Designer?
	mainFrame = new QHBox(this);
	mainFrame->setSpacing(KDialog::spacingHint());
	mainFrame->setMargin(0);
	slider = new L33tSlider(0, 1000, 10, 0, L33tSlider::Horizontal, mainFrame);
	slider->setTickmarks(QSlider::NoMarks);

	elapsed = new QLabel(mainFrame);
	QFont labelFont = elapsed->font();
	labelFont.setPointSize(24);
	labelFont.setBold(true);
	QFontMetrics labelFontMetrics = labelFont;
	elapsed->setFont(labelFont);
	elapsed->setAlignment(AlignCenter | AlignVCenter | ExpandTabs);
	elapsed->setText("--:--");
	elapsed->setFixedHeight(labelFontMetrics.height());
	elapsed->setMinimumWidth(elapsed->sizeHint().width());
	setCentralWidget(mainFrame);

	total = new QLabel(statusBar());
	labelFont = total->font();
	labelFont.setBold(true);
	total->setFont(labelFont);
	total->setAlignment(AlignCenter | AlignVCenter | ExpandTabs);
	total->setText("--:--");
	total->setMinimumWidth(total->sizeHint().width());
	total->setText("");

	statusBar()->addWidget(total, 0, true);

	connect(mPlugin->global()->player(), SIGNAL(playing()),
		this, SLOT(slotPlaying()));
	connect(mPlugin->global()->player(), SIGNAL(stopped()),
		this, SLOT(slotStopped()));
	connect(mPlugin->global()->player(), SIGNAL(paused()),
		this, SLOT(slotPaused()));

	connect(mPlugin->global()->player(), SIGNAL(timeout()),
		this, SLOT(slotTimeout()));
	connect(mPlugin->global()->player(), SIGNAL(loopTypeChange(int)),
		this, SLOT(slotLoopTypeChanged(int)));

	/* This skipToWrapper is needed to pass milliseconds to Player() as everybody
	 * below the GUI is based on milliseconds instead of some unprecise thingy
	 * like seconds or mille */
	connect(slider, SIGNAL(userChanged(int)), this, SLOT(skipToWrapper(int)));
	//connect(this, SIGNAL(skipTo(int)), mPlugin->global()->player(), SLOT(skipTo(int)));

	connect(slider, SIGNAL(sliderMoved(int)), SLOT(slotSliderMoved(int)));

	mPlugin->global()->player()->handleButtons();

	for (QPtrListIterator<QObject> i(*children()); i.current(); ++i)
		(*i)->installEventFilter(this);
}


void ExcellentMainWindow::initGUI()
{
	kDebug(66666) << k_funcinfo << "Creating GUI for Excellent Mainwindow" << endl;
	setAcceptDrops(true);

	StdAction::quit     (mPlugin->global(), actionCollection());
	StdAction::stop     (mPlugin->global(), actionCollection());
	StdAction::playpause(mPlugin->global(), actionCollection());
	StdAction::play     (mPlugin->global(), actionCollection());
	StdAction::pause    (mPlugin->global(), actionCollection());
	StdAction::back     (mPlugin->global(), actionCollection());
	StdAction::forward  (mPlugin->global(), actionCollection());
	StdAction::playlist (mPlugin->global(), actionCollection());
	StdAction::effects  (mPlugin->global(), actionCollection(), "show_effects");
	StdAction::equalizer(mPlugin->global(), actionCollection(), "show_equalizer");
	StdAction::loop     (mPlugin->global(), actionCollection(), "loop_style");

	volumeAction = new KToggleAction(i18n("Show &Volume Control"), 0, this,
		SLOT(showVolumeControl()), actionCollection(), "show_volumecontrol");
	volumeAction->setCheckedState(i18n("Hide &Volume Control"));

	menubarAction = KStandardAction::showMenubar(this, SLOT(slotShowMenubar()), actionCollection());

	setupGUI(ToolBar | Keys | StatusBar | Create, "noatun_excellent.rc");

	// TODO: REMOVE
	//napp->pluginActionMenu()->plug(menuBar(), 3);

	//statusBar()->show();
	//slotLoopTypeChanged(Player::None);
	updateTitleDisplay();
	handleTimeStrings();

	//setMinimumWidth(250);
	//resize(300, 75);

	//toolBar("mainToolBar")->applySettings(&config, "ExcellentMainWindow main");
	//applyMainWindowSettings(&config, "excellent");
	setAutoSaveSettings("excellent", true);

	KSharedConfig::Ptr c = KGlobal::config();
	c->setGroup("excellent");
	volumeSlider = 0;
	volumeAction->setChecked(c->readBoolEntry("volumeShown", false) );
	showVolumeControl();

	menubarAction->setChecked(c->readBoolEntry("menuShown", true));
	slotShowMenubar();

	switch((NET::MappingState)c->readNumEntry("mappingState", (int)NET::Visible))
	{
		case NET::Visible:
			showNormal();
			break;
		case NET::Withdrawn:
			hide();
			break;
		case NET::Iconic:
			showMinimized();
			break;
	}
}


ExcellentMainWindow::~ExcellentMainWindow()
{
	kDebug(66666) << k_funcinfo << endl;
	KConfig &config = *KGlobal::config();
	/*
	saveMainWindowSettings(&config, "excellent");
	toolBar("main")->saveSettings(&config, "ExcellentMainWindow main");
	*/
	config.setGroup("excellent");
	config.writeEntry("volumeShown", volumeAction->isChecked());
	config.writeEntry("menuShown", menubarAction->isChecked());
	//config.writeEntry("width", width());
	//config.writeEntry("height", height());
	config.sync();
}


void ExcellentMainWindow::showEvent(QShowEvent *e)
{
	kDebug(66666) << k_funcinfo << endl;
	KSharedConfig::Ptr config = KGlobal::config();
	config->setGroup("excellent");
	config->writeEntry("mappingState", NET::Visible);
	config->sync();

	KMainWindow::showEvent(e);
}


void ExcellentMainWindow::hideEvent(QHideEvent *e)
{
	kDebug(66666) << k_funcinfo << endl;
	KSharedConfig::Ptr config = KGlobal::config();
	config->setGroup("excellent");
	config->writeEntry("mappingState", NET::Withdrawn);
	config->sync();

	KMainWindow::hideEvent(e);
}


void ExcellentMainWindow::dragEnterEvent(QDragEnterEvent *event)
{
	// accept uri drops only
	event->accept(KURLDrag::canDecode(event));
}


void ExcellentMainWindow::dropEvent(QDropEvent *event)
{
	KURL::List uri;
	if (KURLDrag::decode(event, uri))
		mPlugin->global()->playlist()->addFile(uri);
}


bool ExcellentMainWindow::eventFilter(QObject *o, QEvent *e)
{
	if (e->type() == QEvent::Wheel)
	{
		wheelEvent(static_cast<QWheelEvent*>(e));
		return true;
	}
	return QWidget::eventFilter(o, e);
}


void ExcellentMainWindow::wheelEvent(QWheelEvent *e)
{
	mPlugin->global()->player()->setVolume(mPlugin->global()->player()->volume() + e->delta()/120*2);
	e->accept();
}


void ExcellentMainWindow::slotPlaying()
{
	slider->setEnabled(true);
	updateTitleDisplay();
}


void ExcellentMainWindow::slotStopped()
{
	slider->setEnabled(false);
	if(!mPlugin->global()->player()->current())
		return;
	updateTitleDisplay();
	slider->setValue(0);
	handleTimeStrings();
}


void ExcellentMainWindow::slotPaused()
{
	slider->setEnabled(true);
	updateTitleDisplay();
}


void ExcellentMainWindow::slotTimeout()
{
	if(volumeSlider)
		volumeSlider->setValue(100 - mPlugin->global()->player()->volume());

	if (!slider->currentlyPressed())
		handleTimeStrings(mPlugin->global()->player()->positionString(), mPlugin->global()->player()->lengthString());

	if(!mPlugin->global()->player()->current())
		 return;

	if(slider->currentlyPressed())
		 return;

	slider->setRange(0, (int)mPlugin->global()->player()->getLength() / 1000 );
	slider->setValue((int)mPlugin->global()->player()->getPosition() / 1000 );

	updateTitleDisplay();
}


void ExcellentMainWindow::slotSliderMoved(int seconds)
{
	if(mPlugin->global()->player()->current())
		handleTimeStrings(mPlugin->global()->player()->positionString(seconds * 1000), mPlugin->global()->player()->lengthString());
}


void ExcellentMainWindow::skipToWrapper(int second) // wrap int to int _and_ seconds to mseconds
{
	//emit skipTo((int)(second*1000));
	mPlugin->global()->player()->skipTo(second * 1000);
}


void ExcellentMainWindow::slotLoopTypeChanged(int type)
{
	static const int time = 2000;
	switch (type)
	{
	case Player::None:
		statusBar()->message(i18n("No looping"), time);
		break;
	case Player::Song:
		statusBar()->message(i18n("Song looping"), time);
		break;
	case Player::Playlist:
		statusBar()->message(i18n("Playlist looping"), time);
		break;
	case Player::Random:
		statusBar()->message(i18n("Random play"), time);
	}
}


void ExcellentMainWindow::slotShowMenubar()
{
	kDebug(66666) << k_funcinfo << endl;
	if(menubarAction->isChecked())
	{
		menuBar()->show();
	}
	else
	{
		KMessageBox::information(this,
			i18n("<qt>Press %1 to show the menubar.</qt>",
				menubarAction->shortcut().toString()),
			QString::null, "Hide Menu warning");
		menuBar()->hide();
	}
}


void ExcellentMainWindow::showVolumeControl()
{
	if(volumeAction->isChecked())
		growVolumeControl();
	else
		shrinkVolumeControl();
}


void ExcellentMainWindow::updateTitleDisplay()
{
	if(!mPlugin->global()->player()->current().isNull())
		statusBar()->message(mPlugin->global()->player()->current().title());
}


void ExcellentMainWindow::handleTimeStrings(const QString &pos, const QString &len)
{
	elapsed->setText(pos.isNull() ? "--:--" : pos);
	total->setText(len.isNull() ? "--:--" : len);
}


void ExcellentMainWindow::growVolumeControl()
{
	volumeSlider = new L33tSlider(0, 100, 10, 0, Vertical, mainFrame);
	volumeSlider->setValue(100 - mPlugin->global()->player()->volume());
	volumeSlider->show();
	connect(volumeSlider, SIGNAL(sliderMoved(int)), SLOT(changeVolume(int)));
	connect(volumeSlider, SIGNAL(userChanged(int)), SLOT(changeVolume(int)));
}


void ExcellentMainWindow::shrinkVolumeControl()
{
	delete volumeSlider;
	volumeSlider = 0;
}


void ExcellentMainWindow::changeVolume(int slider)
{
	mPlugin->global()->player()->setVolume(100 - slider);
}

#include "userinterface.moc"
