/***************************************************************************
	kjbutton.cpp
	--------------------------------------
	Handles all ordinary Buttons like stop, play, pause, etc.
	--------------------------------------
	Maintainer: Stefan Gehn <sgehn@gmx.net>

 ***************************************************************************/

#include "kjbutton.h"
#include "kjloader.h"
#include "kjconfig.h"

#include <noatun/global.h>
#include <noatun/config.h>
#include <noatun/player.h>

//#include <noatun/equalizer.h>

#include <qcursor.h>

#include <kdebug.h>
#include <klocale.h>
#include <kpixmap.h>
#include <kpixmapeffect.h>
#include <kurl.h>
#include <kfiledialog.h>
#include <khelpmenu.h>
#include <kpopupmenu.h>


/*******************************************
 * KJButton
 *******************************************/

KJButton::KJButton(const QStringList &i, KJMainWindow *parent)
	: QObject(0), KJWidget(parent), mTitle(i[0]), mShowPressed(false)
{
//	kDebug(66666) << k_funcinfo << "new button: " << i[0].latin1() << endl;
	mPushedPixmap = (i.count() >= 7);

	// get the rectangle
	int x, y, xs, ys;
	x=i[1].toInt();
	y=i[2].toInt();
	xs=i[3].toInt()-x; // width
	ys=i[4].toInt()-y; // height
	setRect ( x, y, xs, ys );

	QStringList temp = i;

	// search for selected button-type
	// can be either BMPx where x is a number representing one
	// of the background-images
	// or darken which means just darken the button on click
	bool gotBack = false; // in case any of these keys is duplicated
	for(QStringList::Iterator it = temp.begin(); it != temp.end(); ++it)
	{
		if((*it).contains("bmp"))
		{
			QString pressedTmp = backgroundPressed((*it));
			if(!pressedTmp.isEmpty())
			{
				mPressed = parent->pixmap(pressedTmp);
				gotBack = true;
			}
		}
		else if((*it) == "darken")
		{
			// take background and darken the buttons rectangle
			// FIXME: what KPixmapEffect causes the desired effect?
			//        intensity is the wrong one
			KPixmap temp = parent->pixmap(parser()["backgroundimage"][1]);
			mPressed = (QPixmap)KPixmapEffect::intensity ( temp, 1.2f );
			gotBack = true;
		}
		if(gotBack)
			break;
	}

	if(!gotBack)
	{
		kDebug(66666) << k_funcinfo << "Couldn't find valid background for button '" <<
			mTitle << "', dafulting to backgroundimage" << endl;
		mPressed = parent->pixmap(parser()["backgroundimage"][1]);
	}

	// playlistbutton has to show if playlistwindow is open
	// right after the button appears on screen
	if (mTitle=="playlistbutton")
	{
		mShowPressed = parentWin()->plugin()->global()->playlist()->listVisible();
		connect(parentWin()->plugin()->global()->playlist(), SIGNAL(listShown()),
			this, SLOT(slotPlaylistShown()) );
		connect(parentWin()->plugin()->global()->playlist(), SIGNAL(listHidden()),
			this, SLOT(slotPlaylistHidden()) );
	}
	else if ( mTitle=="equalizeroffbutton") // same goes for EQ buttons
	{
#if 0
		mShowPressed = (!napp->equalizer()->isEnabled());
		connect(napp->equalizer(), SIGNAL(enabled(bool)), SLOT(slotEqEnabled(bool)));
#else
		mShowPressed = false;
#endif
	}
	else if (mTitle=="equalizeronbutton")
	{
#if 0
		mShowPressed = napp->equalizer()->isEnabled();
		connect(napp->equalizer(), SIGNAL(enabled(bool)), SLOT(slotEqEnabled(bool)));
#else
		mShowPressed = false;
#endif
	}
}

QString KJButton::tip()
{
	QString str;
	if (mTitle=="closebutton")
		str=i18n("Close");
	else if (mTitle=="minimizebutton")
		str=i18n("Minimize");
	else if (mTitle=="aboutbutton")
		str=i18n("About");
	else if (mTitle=="stopbutton")
		str=i18n("Stop");
	else if (mTitle=="playbutton")
		str=i18n("Play");
	else if (mTitle=="pausebutton")
		str=i18n("Pause");
	else if (mTitle=="openfilebutton")
		str=i18n("Open");
	else if (mTitle=="playlistbutton")
		str=i18n("Playlist");
	else if (mTitle=="repeatbutton")
		str=i18n("Loop");
	else if (mTitle=="equalizerbutton")
		str=i18n("Show Equalizer Window");
	else if (mTitle=="equalizeronbutton")
		str=i18n("Turn on Equalizer");
	else if (mTitle=="equalizeroffbutton")
		str=i18n("Turn off Equalizer");
	else if (mTitle=="equalizerresetbutton")
		str=i18n("Reset Equalizer");
	else if (mTitle=="nextsongbutton")
		str=i18n("Next");
	else if (mTitle=="previoussongbutton")
		str=i18n("Previous");
	else if (mTitle=="forwardbutton")
		str=i18n("Forward");
	else if (mTitle=="rewindbutton")
		str=i18n("Rewind");
	else if (mTitle=="preferencesbutton")
		str=i18n("K-Jöfol Preferences");
	else if (mTitle=="dockmodebutton")
		str=i18n("Switch to dockmode");
	else if (mTitle=="undockmodebutton")
		str=i18n("Return from dockmode");

	return str;
}

void KJButton::paint(QPainter *, const QRect &)
{
	if (mShowPressed)
		bitBlt(parentWin(), rect().topLeft(), &mPressed, rect(), Qt::CopyROP);
}

bool KJButton::mousePress(const QPoint &)
{
	bitBlt(parentWin(), rect().topLeft(), &mPressed, rect(), Qt::CopyROP);
	return true;
}

void KJButton::showPressed(bool b)
{
	mShowPressed = b;
	if ( mShowPressed )
		repaint(true); // repaint with selected image
	else
		repaint(false); // repaint with default image (player-background)
}

void KJButton::slotPlaylistShown(void)
{
//	kDebug(66666) << "KJButton::slotPlaylistShown()" << endl;
	showPressed(true);
}

void KJButton::slotPlaylistHidden(void)
{
//	kDebug(66666) << "KJButton::slotPlaylistHidden()" << endl;
	showPressed(false);
}

void KJButton::slotEqEnabled(bool on)
{
//	kDebug(66666) << "KJButton::slotEqEnabled(" << on << ") for " << mTitle << endl;
	if (mTitle=="equalizeronbutton")
		showPressed(on);
	else if (mTitle=="equalizeroffbutton")
		showPressed(!on);
}

void KJButton::mouseRelease(const QPoint &, bool in)
{
	// repaint with default image (player-background)
	repaint(false);

	if (!in) // only do something if users is still inside the button
		return;

	Noatun::Player *player = parentWin()->plugin()->global()->player();

	// now, find what widget I am and do the proper action
	if (mTitle=="closebutton")
		parentWin()->close();
	else if (mTitle=="minimizebutton")
		parentWin()->minimize();
	else if (mTitle=="aboutbutton")
		parentWin()->helpMenu()->aboutApplication();
	else if (mTitle=="stopbutton")
		player->stop();
	else if (mTitle=="playbutton")
		player->play();
	else if (mTitle=="pausebutton")
		player->playpause();
	else if (mTitle=="openfilebutton")
	{
		KURL::List files = KFileDialog::getOpenURLs(":mediadir",
			player->mimeTypes().join(" "), parentWin(),
			i18n("Select File(s) to Play"));

		if (files.count() > 0)
			parentWin()->plugin()->global()->playlist()->addFile(files, parentWin()->plugin()->global()->config()->clearOnOpen() /* TODO */, true);
	}
	else if (mTitle=="playlistbutton")
	{
		parentWin()->plugin()->global()->playlist()->toggleList();
	}
	else if (mTitle=="repeatbutton")
	{
		KPopupMenu *loopMenu = new KPopupMenu(parentWin(),"loopMenu");
		int selectedItem = 0;

		loopMenu->setCheckable(true);
		loopMenu->insertTitle(i18n("Loop Style"));
//		loopMenu->insertItem(i18n("&None"), static_cast<int>(Player::None));
//		loopMenu->insertItem(i18n("&Song"), static_cast<int>(Player::Song));
//		loopMenu->insertItem(i18n("&Playlist"), static_cast<int>(Player::Playlist));
//		loopMenu->insertItem(i18n("&Random"), static_cast<int>(Player::Random));

//		loopMenu->setItemChecked(static_cast<int>(player->loopStyle()), true); // select current loopstyle in menu
		selectedItem = loopMenu->exec(QCursor::pos());
//		if (selectedItem != -1)
//			player->loop(selectedItem); // set new loopstyle

		delete loopMenu;

/*
		if ( player->loopStyle() == 1)
		{
//			kDebug(66666) << "loop song is OFF" << endl;

// bah, xlib.h already defined None
#undef None
			player->loop( Player::None );
			showPressed( false );
		}
		else
		{
//			kDebug(66666) << "loop song is ON" << endl;
			player->loop( Player::Song );
			showPressed ( true );
		}
*/
	}
	else if (mTitle=="equalizerbutton")
	{
#if 0
		napp->equalizerView();
#endif
	}
	else if (mTitle=="equalizeronbutton")
	{
#if 0
		if (!napp->equalizer()->isEnabled())
			napp->equalizer()->enable();
#endif
	}
	else if (mTitle=="equalizeroffbutton")
	{
#if 0
		if (napp->equalizer()->isEnabled())
			napp->equalizer()->disable();
#endif
	}
	else if (mTitle=="equalizerresetbutton")
	{
#if 0
		for (int band=0; band<napp->equalizer()->bands(); band++)
			napp->equalizer()->band(band).setLevel(0);
#endif
	}
	else if (mTitle=="nextsongbutton")
		player->next();
	else if (mTitle=="previoussongbutton")
		player->previous();
	else if (mTitle=="forwardbutton")
		player->seek(player->position()+10000); // TODO: make +- 10 secs configurable
	else if (mTitle=="rewindbutton")
		player->seek(player->position()-10000);
	else if (mTitle=="preferencesbutton")
		parentWin()->plugin()->global()->openPreferencesDialog();
	else if (mTitle=="dockmodebutton")
		parentWin()->switchToDockmode();
	else if (mTitle=="undockmodebutton")
		parentWin()->returnFromDockmode();
	else
	{
		kWarning(66666) << "Unknown KJofol button-type: '" <<
			mTitle.latin1() << "'" << endl;
	}
} // void KJButton::mouseRelease()

#include "kjbutton.moc"
