/***************************************************************************
	kjloader.cpp  -  The KJfol-GUI itself
	--------------------------------------
	Maintainer: Stefan Gehn <sgehn@gmx.net>

 ***************************************************************************/

// local includes
#include "kjloader.h"
#include "kjwidget.h"
#include "kjbackground.h"
#include "kjbutton.h"
#include "kjfont.h"
#include "kjseeker.h"
#include "kjsliders.h"
#include "kjtextdisplay.h"
#include "kjvis.h"
#include "kjconfig.h"
#include "kjequalizer.h"

#include "kjofol_prefs.h"

#include "helpers.cpp"

// arts-includes, needed for pitch
#if 0
#include <artsmodules.h>
#include <reference.h>
#include <soundserver.h>
#include <kmedia2.h>

#include <noatunarts/noatunarts.h>
#endif

// noatun-specific includes
#include <noatun/engine.h>
#include <noatun/stdaction.h>
#include <noatun/player.h>
#include <noatun/global.h>
#include <noatun/pluginloader.h>

// system includes
#include <string.h>
#include <math.h>
#include <stdlib.h>
#include <unistd.h>

#include <qdragobject.h>
#include <qimage.h>
#include <qbitmap.h>
#include <qpixmap.h>
#include <qcursor.h>
#include <qpainter.h>
#include <qtooltip.h>
#include <qptrvector.h>
#include <qvbox.h>
#include <qlabel.h>

#include <kapplication.h>
#include <kaction.h>
#include <kdebug.h>
#include <kfiledialog.h>
#include <khelpmenu.h>
#include <kpopupmenu.h>
#include <klocale.h>
#include <kglobalsettings.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <knotifyclient.h>
#include <kurldrag.h>
#include <ksettings/dispatcher.h>

#include <netwm.h>
#include <kwin.h>
#include <kiconloader.h>
#include <kgenericfactory.h>

#include <X11/Xlib.h>
#include <fixx11h.h>

extern Time qt_x_time;

using namespace Noatun;

K_EXPORT_COMPONENT_FACTORY(noatun_kjofol, PluginFactory<KjofolPlugin>("noatun_kjofol"))


KjofolPlugin::KjofolPlugin(const KComponentData &inst, Global *glob, const char* name)
	: Plugin(inst, glob, name)
{
	mWin = new KJMainWindow(this);
	KJPrefs *prefsPage = new KJPrefs(this);
	connect(prefsPage, SIGNAL(saved()), mWin, SLOT(readConfig()));
}

KjofolPlugin::~KjofolPlugin()
{
	delete mWin;
}

Interface *KjofolPlugin::getInterface(const QString &n)
{
	if (n == "FrontendInterface")
		return this;
	return 0;
}

KMainWindow *KjofolPlugin::mainWindow()
{
	return mWin;
}

void KjofolPlugin::init()
{
	kDebug(66666) << k_funcinfo << endl;
	mWin->initGUI();
}



// ----------------------------------------------------------------------------

class KJToolTip : public QToolTip
{
public:
	KJToolTip(KJMainWindow *parent)
		: QToolTip(parent), mParent(parent)
	{}

protected:
	virtual void maybeTip(const QPoint &p)
	{
		if (!mParent->prefs()->displayTooltips())
			return;

		QPtrList<KJWidget> things = mParent->widgetsAt(p);
		KJWidget *i = 0;
		for (i=things.first(); i!=0; i=things.next())
		{
			QString string = i->tip();
			if (!string.isEmpty())
			{
				tip(i->rect(), string);
				return;
			}
		}
	}

private:
	KJMainWindow *mParent;
};

// ----------------------------------------------------------------------------

KJMainWindow *KJMainWindow::kjofol=0;


KJMainWindow::KJMainWindow(Plugin *plug)
	: NMainWindow(plug, true, true, 0, "NoatunKJMainWindow", WStyle_Customize | WStyle_NoBorder | WNoAutoErase),
		mClickedIn(0),
		mText(0),
		mNumbers(0),
		mVolumeFont(0),
		mPitchFont(0),
		splashScreen(0),
		mPlugin(plug)
{
	kjofol = this;
	new KJToolTip(this);

	// Windowname and Icon
	setCaption(i18n("Noatun"));
	setIcon(SmallIcon("noatun"));
	setAcceptDrops(true);

	// We're going to draw over everything, there is no point in drawing the grey background first
	setBackgroundMode(NoBackground);

	// used for dockmode
	mWin = new KWinModule();
	mConfig = KJConfig::self();

	subwidgets.setAutoDelete(true);

	QString skin = mConfig->skinResource();
	if (QFile(skin).exists())
	{
		loadSkin(skin);
	}
	else
	{
		KNotifyClient::event(winId(), "warning",
			i18n("There was trouble loading skin %1. Please select another skin file.",skin));
		mPlugin->global()->openPreferencesDialog();
	}

	mHelpMenu = new KHelpMenu(this, KGlobal::mainComponent().aboutData());
	Player *player = mPlugin->global()->player();
	connect(player, SIGNAL(tick()), SLOT(timeUpdate()));
	connect(player, SIGNAL(stopped()), SLOT(timeUpdate()));
	connect(player, SIGNAL(newSong()), SLOT(newSong()));
}


void KJMainWindow::initGUI()
{
	kDebug(66666) << k_funcinfo << "Creating GUI for KJofol Mainwindow" << endl;

	// - Popupmenu
	StdAction::quit			(mPlugin->global(), actionCollection());
	StdAction::preferences	(mPlugin->global(), actionCollection());
	//   | Player
	StdAction::back			(mPlugin->global(), actionCollection());
	StdAction::stop			(mPlugin->global(), actionCollection());
	StdAction::playpause		(mPlugin->global(), actionCollection());
	StdAction::forward		(mPlugin->global(), actionCollection());
	//   ----------
	StdAction::playlist		(mPlugin->global(), actionCollection());
	StdAction::effects		(mPlugin->global(), actionCollection());
	StdAction::equalizer		(mPlugin->global(), actionCollection());

	// ------------

	setupGUI(Create | Keys, "noatun_kjofol.rc");
}


KJMainWindow::~KJMainWindow()
{
	kDebug(66666) << k_funcinfo << endl;
	delete mWin;
}


Plugin *KJMainWindow::plugin() const
{
	return mPlugin;
}


KJConfig *KJMainWindow::prefs() const
{
	return mConfig;
}


QPtrList<KJWidget> KJMainWindow::widgetsAt(const QPoint &pt) const
{
	QPtrList<KJWidget> things;
	for (QPtrListIterator<KJWidget> i(subwidgets); i.current(); ++i)
	{
		if ((*i)->rect().contains(pt))
			things.append((*i));
	}
	return things;
}


void KJMainWindow::removeChild(KJWidget *c)
{
	if (mClickedIn == c)
		mClickedIn = 0;
	if (subwidgets.findRef(c) != -1)
		subwidgets.take();
}


void KJMainWindow::addChild(KJWidget *c)
{
	subwidgets.append(c);
}


// The BIG one ;)
// this methode does all the hard work on loading these weird skins
void KJMainWindow::loadSkin(const QString &file)
{
	if (file == mCurrentSkin) // we don't load the same skin again
		return;
	mCurrentSkin = file;

	// don't overwrite path to *.rc when we are loading dock- or winshade-mode
	if ( (file != mCurrentWinshadeModeSkin) && (file != mCurrentDockModeSkin) )
	{
		mCurrentDefaultSkin = file;
//		kDebug(66666) << " setting mCurrentDefaultSkin: '" << file.latin1() << "'" << endl;
	}
	unloadSkin();

	Parser::open(file);

	KJPitchText *pitchText=0;
	KJVolumeText *volumeText=0;

	if ( exist("splashscreen") && mConfig->displaySplashScreen() )
		showSplash();

	if ( (file != mCurrentWinshadeModeSkin) && (file != mCurrentDockModeSkin) )
	{
		if ( exist("dockmodercfile") )
		{
			// set path to dockmode rc-file (its not always skinname.dck)
			mCurrentDockModeSkin = file.left(file.findRev("/")+1) + (item("dockmodercfile")[1]);
			mDockPosition  = item("dockmodeposition")[1].toInt();
			mDockPositionX = item("dockmodepositionxy")[1].toInt();
			mDockPositionY = item("dockmodepositionxy")[2].toInt();
		}
		else	// NO DockMode
			mCurrentDockModeSkin="";

		if ( exist("winshademodercfile") )
			mCurrentWinshadeModeSkin = file.left(file.findRev("/")+1) + (item("winshademodercfile")[1]);
		else	// no WinshadeMode
			mCurrentWinshadeModeSkin="";
	}

	// Font-loading
	if ( exist("fontimage") )
		mText = new KJFont("font", this);

	if ( exist("timefontimage") )
		mNumbers = new KJFont("timefont", this);

	if (exist("volumefontimage"))
		mVolumeFont = new KJFont("volumefont", this);

	// our skin-background, There has to be one so no check with exist()
	subwidgets.append( new KJBackground(this) );

	if ( exist("pitchtext") )
	{
		if (exist("pitchfontimage"))
		{
			mPitchFont = new KJFont("pitchfont", this);
		}
		else
		{
			mPitchFont = mNumbers;
			kDebug(66666) << "no pitchfont but pitchtext!" << endl;
			kDebug(66666) << "replacing pitchfont with timefont" << endl;
		}
		subwidgets.append( pitchText=new KJPitchText(item("pitchtext"), this) );
	}

	if (exist("volumetext"))
		subwidgets.append(volumeText=new KJVolumeText(item("volumetext"), this));

	if ( exist("volumecontroltype") )
	{
		if ( item("volumecontroltype")[1] == "bmp" )
		{
			KJVolumeBMP *b;
			subwidgets.append(b=new KJVolumeBMP(item("volumecontrolbutton"), this));
			b->setText(volumeText);
		}
		else if ( item("volumecontroltype")[1] == "bar" )
		{
			KJVolumeBar *b;
			subwidgets.append(b=new KJVolumeBar(item("volumecontrolbutton"), this));
			b->setText(volumeText);
		}
/*		else
		{
			kDebug(66666) << "unknown volumecontrol: " << item("volumecontroltype")[1].latin1() << endl;
		} */
	}
	else
	{
		kDebug(66666) << "guessing type of volumecontrol" << endl;
		if (exist("volumecontrolbutton") &&
			exist("volumecontrolimage") &&
			exist("volumecontrolimagexsize") &&
			exist("volumecontrolimageposition") &&
			exist("volumecontrolimagenb") )
		{
			KJVolumeBMP *b;
			subwidgets.append(b=new KJVolumeBMP(item("volumecontrolbutton"), this));
			b->setText(volumeText);
		}
		else if (exist("volumecontrolimage") &&
				exist("volumecontrolbutton") )
		{
			KJVolumeBar *b;
			subwidgets.append(b=new KJVolumeBar(item("volumecontrolbutton"), this));
			b->setText(volumeText);
		}
/*		else
		{
			kDebug(66666) << " no volumecontrol" << endl;
		} */
	}

	if (exist("pitchcontrolbutton") &&
		exist("pitchcontrolimage") &&
		exist("pitchcontrolimagexsize") &&
		exist("pitchcontrolimageposition") &&
		exist("pitchcontrolimagenb") )
	{
//		kDebug(66666) << "added KJPitchBMP" << endl;
		KJPitchBMP *b;
		subwidgets.append(b=new KJPitchBMP(item("pitchcontrolbutton"), this));
		b->setText(pitchText);
	}
#if 0
	else
	{
		// make sure we reset speed to 100% as the user won't be able
		// to reset it without a pitchcontrol
		Arts::PlayObject playobject = mPlugin->global()->engine()->playObject();
		Arts::PitchablePlayObject pitchable = Arts::DynamicCast(playobject);

		if ( !pitchable.isNull() )
		{
			if ( pitchable.speed() > 1.0f )
				pitchable.speed(1.0f);
		}
	}
#endif

	if (exist("filenamewindow"))
		subwidgets.append(new KJFilename(item("filenamewindow"), this));

	if (exist("mp3timewindow"))
		subwidgets.append(new KJTime(item("mp3timewindow"), this));

	if (exist("mp3kbpswindow"))
		subwidgets.append(new KJFileInfo(item("mp3kbpswindow"), this));

	if (exist("mp3khzwindow"))
		subwidgets.append(new KJFileInfo(item("mp3khzwindow"), this));

	if (exist("analyzerwindow"))
	{
#if 0
		switch (mConfig->analyzerType())
		{
			case KJConfig::None:
				subwidgets.append(new KJNullScope(item("analyzerwindow"), this));
				break;
			case KJConfig::FFT:
				subwidgets.append(new KJFFT(item("analyzerwindow"), this));
				break;
			case KJConfig::StereoFFT:
				subwidgets.append(new KJStereoFFT(item("analyzerwindow"), this));
				break;
			case KJConfig::Mono:
				subwidgets.append(new KJScope(item("analyzerwindow"), this));
				break;
		}
#else
	subwidgets.append(new KJNullScope(item("analyzerwindow"), this));
#endif
	}

	if (exist("EqualizerWindow"))
		subwidgets.append(new KJEqualizer(item("EqualizerWindow"), this));

	// I cant believe it, there are skins without a seeker, now THATS stupid :)
	if (exist("seekregion"))
		QTimer::singleShot(0, this, SLOT(loadSeeker()));

	// all the regular buttons
	for (QDictIterator<QStringList> i(*this); i.current(); ++i)
	{
		QString d=i.currentKey();
		if(d.contains("button") &&
			!d.startsWith("playlistwindow") && // don't add buttons that belong to the playlistwindow
			d != "pitchcontrolbutton" &&  // both already handled above as they aren't buttons but sliders
			d != "volumecontrolbutton" &&
			d != "spectrumanalyzerbutton" && // FIXME: unsupported button
			d != "oscilloscopebutton" && // FIXME: unsupported button
			i.count() >= 7 )
		{
			subwidgets.append(new KJButton(*(*i), this));
		}
	}

	Parser::conserveMemory();
	show();

	repaint();

	// update displays if we are already playing
	// This happens while changing skins
	if (mPlugin->global()->player()->isPlaying())
		newSong();
}

void KJMainWindow::loadSeeker()
{
	QStringList seek = item("seekregion");

	subwidgets.append(new KJSeeker(seek, this));
}

void KJMainWindow::unloadSkin()
{
	KWin::clearState(winId(), NET::SkipTaskbar);
	hide();

	subwidgets.clear();

	// This is special because mPitchfont can also point to mNumbers
	// as some skins use the NumberFont for pitchtext
	if (mPitchFont && mPitchFont != mNumbers)
		delete mPitchFont;
	mPitchFont = 0;

	delete mText;
	mText = 0;

	delete mNumbers;
	mNumbers = 0;

	delete mVolumeFont;
	mVolumeFont = 0;
}

void KJMainWindow::minimize()
{
//	kDebug(66666) << "KJMainWindow::minimize()" << endl;
	showMinimized();
}

void KJMainWindow::dragEnterEvent(QDragEnterEvent *event)
{
	// accept uri drops only
	event->accept(KURLDrag::canDecode(event));
}

void KJMainWindow::dropEvent(QDropEvent *event)
{
	KURL::List urls;
	if ( KURLDrag::decode(event,urls) )
		plugin()->global()->playlist()->addFile(urls);
}

void KJMainWindow::wheelEvent(QWheelEvent *e)
{
	// from QT-Docu: delta() is 120 for one step
	Player *pl = plugin()->global()->player();

	if (e->state() & ControlButton)
		pl->setVolume(pl->volume() + (e->delta()/8)); // 15% volumechange
	else
		pl->setVolume(pl->volume() + (e->delta()/24)); // 5% volumechange
}

// now for some dockmode stuff
void KJMainWindow::switchToDockmode()
{
//	kDebug(66666) << "KJMainWindow::switchToDockmode()" << endl;
	loadSkin(mCurrentDockModeSkin);

	connect(mWin, SIGNAL(activeWindowChanged(WId)), this, SLOT(slotWindowActivate(WId)));
	connect(mWin, SIGNAL(windowRemoved(WId)), this, SLOT(slotWindowRemove(WId)));
	connect(mWin, SIGNAL(stackingOrderChanged()), this, SLOT(slotStackingChanged()));
	connect(mWin, SIGNAL(windowChanged(WId)), this, SLOT(slotWindowChange(WId)));
	connect(mWin, SIGNAL(currentDesktopChanged(int)), this, SLOT(slotDesktopChange(int)));

	WId activeWin = mWin->activeWindow();
	if (activeWin && (activeWin != winId()))
	{
		KWin::WindowInfo winInf = KWin::windowInfo(activeWin, NET::WMKDEFrameStrut);
		if(winInf.valid())
		{
			mDockToWin = activeWin;
			mDockWindowRect = winInf.frameGeometry();
			slotWindowActivate(mDockToWin);
			hide();
			restack();
		}
	}
}

void KJMainWindow::returnFromDockmode()
{
//	kDebug(66666) << "KJMainWindow::returnFromDockmode()" << endl;
	mWin->disconnect();
	loadSkin(mCurrentDefaultSkin);
}

void KJMainWindow::slotWindowActivate(WId win)
{
	if(mCurrentSkin != mCurrentDockModeSkin)
		return;

	KWin::WindowInfo winInf = KWin::windowInfo(
		win, NET::WMWindowType);
	if((win != winId()) && winInf.valid())
	{
		// ensure we dock to the active window _except_ our own
		// and stick to the last window if the NEW current one is a desktop
		NET::WindowType winType = winInf.windowType(
			NET::NormalMask|NET::DesktopMask|NET::DockMask|
			NET::ToolbarMask|NET::MenuMask|NET::DialogMask|
			NET::OverrideMask|NET::TopMenuMask|NET::UtilityMask|
			NET::SplashMask);

		if(winType == NET::Unknown || winType == NET::Normal || winType == NET::Dialog)
		{
			//kDebug(66666) << k_funcinfo << "Now docking to window: " << win << endl;
			mDockToWin = win;
		}

	}

	if(mDockToWin != 0)
	{
		mDockWindowRect = KWin::windowInfo(mDockToWin, NET::WMKDEFrameStrut).frameGeometry();
		/*kDebug(66666) << k_funcinfo << "winrect: " << mDockWindowRect.x() << ", " <<
			mDockWindowRect.y() << endl;*/
		switch (mDockPosition)
		{
			case 0:
				move(mDockWindowRect.x() + mDockPositionX,
					mDockWindowRect.y() + mDockPositionY);
				break;
			case 2:
				move(mDockWindowRect.x() + mDockPositionX,
					mDockWindowRect.y() + mDockWindowRect.height() + mDockPositionY);
				break;
		}

		if(!isVisible())
		{
			show();
			KWin::setState(winId(), NET::SkipTaskbar);
		}
		restack();
	}
	else
	{
		// We don't want to do anything until a window comes into
		// focus.
		//kDebug(66666) << "No window having focus, hiding" << endl;
		hide();
	}

//	kDebug(66666) << "END slotWindowActivate()" << endl;
}

void KJMainWindow::slotWindowRemove(WId win)
{
//	kDebug(66666) << "START slotWindowRemove()" << endl;
	if ( mCurrentSkin != mCurrentDockModeSkin )
		return;

	if (win == mDockToWin)
	{
//		kDebug(66666) << "our window removed: " << win << endl;
		hide();
		mDockToWin = 0;
	}
//	kDebug(66666) << "END slotWindowRemove()" << endl;
}

void KJMainWindow::slotWindowChange(WId win)
{
//	kDebug(66666) << "START slotWindowChange()" << endl;
	if ( mCurrentSkin != mCurrentDockModeSkin )
		return;

	if ( win == mDockToWin )
	{
//		kDebug(66666) << "changed our window:" << win << endl;
		KWin::WindowInfo winInf = KWin::windowInfo(
			mDockToWin, NET::WMKDEFrameStrut|NET::WMWindowType|
			NET::WMState|NET::XAWMState|NET::WMDesktop);

		if(!winInf.valid())
		{
			/*kDebug(66666) << k_funcinfo <<
				"No valid WindowInfo for tracked window: " << win << endl;*/
			hide();
			mDockToWin = 0;
			return;
		}

		NET::WindowType winType = winInf.windowType(
			NET::NormalMask|NET::DesktopMask|NET::DockMask|
			NET::ToolbarMask|NET::MenuMask|NET::DialogMask|
			NET::OverrideMask|NET::TopMenuMask|NET::UtilityMask|
			NET::SplashMask);

		if (
			(winInf.state() & NET::Hidden) ||
			(winInf.state() & NET::FullScreen) ||
			(winType != NET::Unknown && winType != NET::Normal && winType != NET::Dialog)
			)
		{
			/*kDebug(66666) << k_funcinfo <<
				"Our window changed: " << win <<
				". Either iconified or special window" << endl;*/
			// target-window has been iconified or window is desktop
			hide();
			mDockToWin = 0;
			return;
		}

		// Size or position of target-window changed.
		mDockWindowRect = winInf.frameGeometry();
		/*kDebug(66666) << k_funcinfo << "winrect: " << mDockWindowRect.x() << ", " <<
			mDockWindowRect.y() << endl;*/
		// Ensure we are still on the window.
		switch(mDockPosition)
		{
			case 0:
			{
				move(
					mDockWindowRect.x() + mDockPositionX,
					mDockWindowRect.y() + mDockPositionY);
				break;
			}
			case 2:
			{
				move(
					mDockWindowRect.x() + mDockPositionX,
					mDockWindowRect.y() + mDockWindowRect.height() + mDockPositionY);
				break;
			}
		}
		restack();
	}
}

void KJMainWindow::slotDesktopChange(int)
{
//	kDebug(66666) << "START slotDesktopChange()" << endl;
	if ( mCurrentSkin != mCurrentDockModeSkin )
		return;

	hide();
	mDockToWin = 0L;
//	kDebug(66666) << "END slotDesktopChange()" << endl;
}


void KJMainWindow::slotStackingChanged()
{
//	kDebug(66666) << "START slotStackingChanged()" << endl;
	if ( mCurrentSkin != mCurrentDockModeSkin )
		return;

	// We seem to get this signal before the window has been restacked,
	// so we just schedule a restack.
	QTimer::singleShot ( 10, this, SLOT(restack()) );

//	kDebug(66666) << "END slotStackingChanged()" << endl;
}


// Set the animation's stacking order to be just above the target window's
// window decoration, or on top.
void KJMainWindow::restack()
{
//	kDebug(66666) << "START restack()" << endl;

	if ( !mDockToWin )
	{
//		kDebug(66666) << "No window to dock to, no restacking" << endl;
		hide();
		return;
	}

	// simply raise ourselves to the top
	raise();
	// and then ensure our target-window gets focus
//	NET::setActiveWindow (mDockToWin);

//	kDebug(66666) << "END restack()" << endl;
}


void KJMainWindow::paintEvent(QPaintEvent *e)
{
	QPainter p(this);
	for (KJWidget* i=subwidgets.first(); i!=0; i=subwidgets.next())
		if (i->rect().intersects(e->rect()))
			i->paint(&p, e->rect().intersect(i->rect()));
//	QWidget::paintEvent(e);
}


void KJMainWindow::mouseMoveEvent(QMouseEvent *e)
{
//	QWidget::mouseMoveEvent(e);
	// not on background but on a widget: pass event to subwidget
	if ( mClickedIn && subwidgets.findRef(mClickedIn) != -1 )
	{
		mClickedIn->mouseMove (
			e->pos()-mClickedIn->rect().topLeft(),
			mClickedIn->rect().contains(mapFromGlobal(QCursor::pos())) );
	}
}


void KJMainWindow::mousePressEvent(QMouseEvent *e)
{
//	kDebug(66666) << "KJMainWindow::mousePressEvent(QMouseEvent *e)" << endl;
	if (e->button() == RightButton)
	{
		KPopupMenu *popup = static_cast<KPopupMenu *>(factory()->container("NoatunContextMenu",this));
		if (popup)
			popup->exec(QCursor::pos());
	}
	else /* if ( e->button()==LeftButton ) */
	{
		QPoint mousePoint = mapFromGlobal(QCursor::pos());
		// try to find a KJWidget that is here
		for (KJWidget* i=subwidgets.first(); i!=0; i=subwidgets.next())
		{
			if (i->rect().contains(mousePoint))
			{
				if (i->mousePress(mousePoint-i->rect().topLeft()))
				{
					mClickedIn=i;
					return;
				}
			}
		}
		// can't find a widget, so move the window
		KWin::raiseWindow( winId() );
		XUngrabPointer( qt_xdisplay(), qt_x_time );
		NETRootInfo wm_root( qt_xdisplay(), NET::WMMoveResize );
		wm_root.moveResizeRequest(
				winId(), e->globalX(), e->globalY(), NET::Move
			);
		e->accept();
	}
}


void KJMainWindow::mouseReleaseEvent(QMouseEvent *e)
{
//	kDebug(66666) << "KJMainWindow::mouseReleaseEvent(QMouseEvent *e)" << endl;
	if (mClickedIn && subwidgets.findRef(mClickedIn)!=-1)
	{
		mClickedIn->mouseRelease(
			mapFromGlobal(QCursor::pos()) - mClickedIn->rect().topLeft(),
				mClickedIn->rect().contains(mapFromGlobal(QCursor::pos()))
		);
		mClickedIn = 0;
	}

	NETRootInfo wm_root( qt_xdisplay(), NET::WMMoveResize );
	wm_root.moveResizeRequest(
			winId(), e->globalX(), e->globalY(), NET::MoveResizeCancel
		);


}


void KJMainWindow::timeUpdate()
{
	for (KJWidget* widget=subwidgets.first(); widget; widget=subwidgets.next())
		widget->timeUpdate(mPlugin->global()->player()->position()/1000); // pass seconds to all Widgets
}


void KJMainWindow::newSong()
{
	if (!mPlugin->global()->player()->current())
		return;
	for (KJWidget* i=subwidgets.first(); i!=0; i=subwidgets.next())
		i->newFile();
}


void KJMainWindow::readConfig()
{
	kDebug(66666) << "KJMainWindow::readConfig()" << endl;
	mConfig->readConfig();
	loadSkin(mConfig->skinResource());
	for (KJWidget* i=subwidgets.first(); i!=0; i=subwidgets.next())
		i->readConfig();
}

void KJMainWindow::showSplash()
{
	splashScreen = new QLabel(0L, "SplashScreen",
		Qt::WStyle_Customize | Qt::WStyle_Splash | WRepaintNoErase);

	QPixmap splashPix = pixmap(item("splashscreen")[1]);
	splashScreen->setPixmap(splashPix);
	splashScreen->setBackgroundMode(NoBackground);
	splashScreen->setMask(KJgetMask(image(item("splashscreen")[1])));

	QSize sh = splashScreen->sizeHint();

	QRect desk = KGlobalSettings::splashScreenDesktopGeometry();
	splashScreen->move (desk.x() + (desk.width() - sh.width())/2,
		desk.y() + (desk.height() - sh.height())/2 );

	splashScreen->setFixedSize(sh);
	splashScreen->show();
	kapp->processEvents(); // we want this one time to get the splash actually displayed ASAP

	QTimer::singleShot(3000, this, SLOT(hideSplash()));
}

void KJMainWindow::hideSplash()
{
	//splashScreen->hide();
	delete splashScreen;
	splashScreen = 0;
}

#include "kjloader.moc"
