/* This file is part of Noatun

  This file was created on 2007-02-01

  Copyright 2007 by Stefan Gehn <mETz81@web.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/
#include "noatun_phonon.h"

#include <noatun/player.h>
#include <noatun/global.h>

#include <phonon/mediaobject.h>
#include <phonon/audiopath.h>
#include <phonon/audiooutput.h>
//#include <phonon/videopath.h>
//#include <phonon/ui/videowidget.h>
#include <phonon/backendcapabilities.h>

#include <qcoreapplication.h>
#include <qfile.h>

using namespace Noatun;


// ---------------------------------------------------------------------------


K_EXPORT_COMPONENT_FACTORY(noatun_phonon, PluginFactory<PhononPlugin>("noatun_phonon"))


// ---------------------------------------------------------------------------


PhononPlugin::PhononPlugin(const KComponentData &instance, Global *nInst, const char* name)
	: Plugin(instance, nInst, name), EngineInterface(nInst),
	mMediaObject(0), mAudioPath(0), /*mVideoPath(0),*/ mAudioOutput(0)
{
	kDebug(66666) << k_funcinfo << endl;
}


PhononPlugin::~PhononPlugin()
{
	kDebug(66666) << k_funcinfo << endl;
}


Interface *PhononPlugin::getInterface(const QString &interface)
{
	if (interface == "EngineInterface")
		return this;
	return 0;
}


void PhononPlugin::init()
{
	kDebug(66666) << k_funcinfo << "BEGIN ======" << endl;

	mMediaObject = new Phonon::MediaObject(this);
	mAudioPath = new Phonon::AudioPath(this);
	mAudioOutput = new Phonon::AudioOutput(Phonon::MusicCategory, this);
	//mVideoPath = new Phonon::VideoPath(this);

	mMediaObject->addAudioPath(mAudioPath);
	mAudioPath->addOutput(mAudioOutput);
	//mMediaObject->addVideoPath(mVideoPath);
	//mVideoPath->addOutput(mVideoWidget);

	connect(mMediaObject, SIGNAL(stateChanged(Phonon::State, Phonon::State)), SLOT(updateState(Phonon::State, Phonon::State)));
	connect(mMediaObject, SIGNAL(finished()), SLOT(finishedPlaying()));

	//We could make use of additional signals from phonon:
	//connect(mMediaObject, SIGNAL(length(qint64)), SLOT(updateLength(qint64)));
	//connect(mAudioOutput, SIGNAL(volumeChanged(float)), SLOT(updateVolume(float)));

	kDebug(66666) << k_funcinfo << "END ======" << endl;
}


void PhononPlugin::requestUnload()
{
	kDebug(66666) << k_funcinfo << endl;
	stop();
	emit readyForUnload(this);
}


// -----------------------------------------------------------------------------
// EngineInterface implementation:


Player::State PhononPlugin::convertState(Phonon::State s)
{
	switch(s)
	{
	case Phonon::PlayingState:
		return Player::PlayingState;
	case Phonon::PausedState:
		return Player::PausedState;
	case Phonon::LoadingState: // map all these to stopped for now
	case Phonon::StoppedState:
	case Phonon::BufferingState:
	case Phonon::ErrorState:
		break;
	}
	return Player::StoppedState;
}


Player::State PhononPlugin::state() const
{
	if (mMediaObject)
	{
		return PhononPlugin::convertState(mMediaObject->state());
	}
	else
	{
		kWarning(66666) << k_funcinfo << "NO MEDIAOBJECT" << endl;
	}
	return Player::StoppedState;
}


int PhononPlugin::position() const
{
	if (mMediaObject)
	{
		int pos = mMediaObject->currentTime(); // qint64 vs. int
		//kDebug(66666) << k_funcinfo << pos << endl;
		return pos;
	}
	else
	{
		kWarning(66666) << k_funcinfo << "NO MEDIAOBJECT" << endl;
	}
	return -1;
}


int PhononPlugin::length() const
{
	if (mMediaObject)
	{
		int len = mMediaObject->totalTime(); // qint64 vs. int
		kDebug(66666) << k_funcinfo << len << endl;
		return len;
	}
	else
	{
		kWarning(66666) << k_funcinfo << "NO MEDIAOBJECT" << endl;
	}
	return -1;
}


QStringList PhononPlugin::mimeTypes() const
{
	return Phonon::BackendCapabilities::knownMimeTypes();
}


unsigned int PhononPlugin::softwareVolume() const
{
	//kDebug(66666) << k_funcinfo << endl;
	if (mAudioOutput)
	{
		return (unsigned int)(100.00 * mAudioOutput->volume() + 0.5);
	}
	else
	{
		kWarning(66666) << k_funcinfo << "Missing a Phonon::AudioOutput object" << endl;
	}
	return 0u;
}


void PhononPlugin::setSoftwareVolume(unsigned int percent)
{
	//kDebug(66666) << k_funcinfo << endl;
	if (mAudioOutput)
	{
		mAudioOutput->setVolume(percent * 0.01);
	}
	else
	{
		kWarning(66666) << k_funcinfo << "Missing a Phonon::AudioOutput object" << endl;
	}
}


bool PhononPlugin::play(const KUrl &url)
{
	kDebug(66666) << k_funcinfo << "asked to play " << url << endl;
	if (mMediaObject)
	{
		if (url.isEmpty() && state() == Noatun::Player::PausedState)
		{
			kDebug(66666) << k_funcinfo << "Unpausing..." << endl;
			mMediaObject->play();
			return true;
		}

		mMediaObject->setUrl(url);
		mMediaObject->play();
		return true;
	}
	kWarning(66666) << "Cannot play " << url << endl;
	return false;
}


void PhononPlugin::pause()
{
	kDebug(66666) << k_funcinfo << endl;
	if (mMediaObject)
		mMediaObject->pause();
}


void PhononPlugin::stop()
{
	kDebug(66666) << k_funcinfo << endl;
	if (mMediaObject)
		mMediaObject->stop();
}


void PhononPlugin::setPosition(int msec)
{
	if (mMediaObject && msec >= 0)
	{
		kDebug(66666) << k_funcinfo << "msec = " << msec << endl;
		mMediaObject->seek(msec);
		// Phonon is async, do not expect position() to have changed immediately
	}
}


void PhononPlugin::updateState(Phonon::State newState, Phonon::State oldState)
{
	if (newState == Phonon::ErrorState)
	{
		kDebug(66666) << k_funcinfo << "error" << endl;
		stop();
		//TODO: check phonon for updated error handling as promised by Vir
		EngineInterface::errorOccurred(Player::NormalError, QString::null);
	}
	else
	{
		kDebug(66666) << k_funcinfo << "old: " << oldState << "; new: " << newState << endl;
		EngineInterface::stateChanged(convertState(newState));
	}
}


void PhononPlugin::finishedPlaying()
{
	kDebug(66666) << k_funcinfo << " WOOHOO, PLAYING FINISHED" << endl;
	EngineInterface::playbackFinished();
}

#include "noatun_phonon.moc"
