/*  This file is part of the QbtGStreamer project, a Qt GStreamer Wrapper
    Copyright (C) 2006 Tim Beaulen <tbscope@gmail.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License version 2 as published by the Free Software Foundation.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "qbtgstreamerbus.h"

#include "qbtgstreamercallbackmanager.h"

#include <QtDebug>

QbtGStreamerBus* QbtGStreamerBus::m_instance;

QbtGStreamerBus::QbtGStreamerBus()
    : d(new QbtGStreamerBusPrivate)
{
    m_instance = this;

    d->bus = gst_bus_new();
}

bool QbtGStreamerBus::postMessage(QbtGStreamerMessage *message)
{
    return gst_bus_post(d->bus, message->gstMessage());
}

bool QbtGStreamerBus::pendingMessages()
{
    return gst_bus_have_pending(d->bus);
}

QbtGStreamerMessage *QbtGStreamerBus::peek()
{
    GstMessage *gstMessage = gst_bus_peek(d->bus);

    if(!gstMessage)
        return 0;

    QbtGStreamerMessage *btMessage = new QbtGStreamerMessage;
    btMessage->setGstMessage(gstMessage);

    return btMessage;
}

QbtGStreamerMessage *QbtGStreamerBus::pop()
{
    GstMessage *gstMessage = gst_bus_pop(d->bus);

    if(!gstMessage)
        return 0;

    QbtGStreamerMessage *btMessage = new QbtGStreamerMessage;
    btMessage->setGstMessage(gstMessage);

    return btMessage;
}

void QbtGStreamerBus::setFlushing(bool flushing)
{
    gst_bus_set_flushing(d->bus, flushing);
}

void QbtGStreamerBus::messageWatch(bool watch)
{
    if(watch)
        gst_bus_add_signal_watch(d->bus);
    else
        gst_bus_remove_signal_watch(d->bus);
}

void QbtGStreamerBus::syncMessageWatch(bool watch)
{
    if(watch)
        gst_bus_enable_sync_message_emission(d->bus);
    else
        gst_bus_disable_sync_message_emission(d->bus);
}

void QbtGStreamerBus::messageWatchPriority(bool watch, int priority)
{
    if(watch)
        gst_bus_add_signal_watch_full(d->bus, priority);
    else
        gst_bus_remove_signal_watch(d->bus);
}

void QbtGStreamerBus::setGstBus(GstBus *bus)
{
    d->bus = bus;

    g_signal_connect(GST_OBJECT(d->bus), "message", G_CALLBACK(d->bus_message), m_instance);
}

GstBus *QbtGStreamerBus::gstBus()
{
    return d->bus;
}

void QbtGStreamerBusPrivate::bus_message(GstBus *bus, GstMessage *message, gpointer user_data)
{
    QbtGStreamerMessage *btMessage;
    QbtGStreamerDataPointer *btData;

    foreach(QbtGStreamerBus* b, QbtGStreamerCallbackManager::Instance()->busses) {
        if(bus == b->gstBus()) {
            btMessage = new QbtGStreamerMessage;
            btMessage->setGstMessage(message);
            btData = new QbtGStreamerDataPointer;
            btData->setGPointer(user_data);
            emit b->message(btMessage, btData);
        }
    }
}

#include "qbtgstreamerbus.moc"
