/* This file is part of Noatun

  Copyright 2000-2006 by Charles Samuels <charles@kde.org>
  Copyright 2001-2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "player_p.h"
#include "noatun/player.h"

#include "noatun/global.h"
#include "noatun/pluginloader.h"
#include "noatun/plugininterfaces.h"
#include "noatun/plugin.h"
#include "noatun/config.h"

#include <qregexp.h>
#include <qtimer.h>

#include <kdebug.h>
#include <kurl.h>

namespace Noatun
{


Player::State PlayerPrivate::convertState(Phonon::State s)
{
	switch(s)
	{
	case Phonon::PlayingState:
		return Player::PlayingState;
	case Phonon::PausedState:
		return Player::PausedState;
	case Phonon::LoadingState: // map all these to stopped for now
	case Phonon::StoppedState:
	case Phonon::BufferingState:
	case Phonon::ErrorState:
		break;
	}
	return Player::StoppedState;
}

void PlayerPrivate::initPhonon()
{
	mediaObject = new Phonon::MediaObject(q);
	audioOutput = new Phonon::AudioOutput(Phonon::MusicCategory, q);
	//videoPath = new Phonon::VideoPath(q);

	Phonon::createPath(mediaObject, audioOutput);
	//mediaObject->addVideoPath(videoPath);
	//videoPath->addOutput(videoWidget);

	mediaObject->setTickInterval(200);

	QObject::connect(mediaObject, SIGNAL(stateChanged(Phonon::State, Phonon::State)),
		q, SLOT(_n_updateState(Phonon::State, Phonon::State)));
	QObject::connect(mediaObject, SIGNAL(finished()),
		q, SLOT(_n_finishedPlaying()));
	QObject::connect(mediaObject, SIGNAL(length(qint64)),
		q, SLOT(_n_updateLength(qint64)));
	QObject::connect(mediaObject, SIGNAL(metaDataChanged()),
		q, SLOT(_n_updateMetaData()));
	QObject::connect(mediaObject, SIGNAL(tick(qint64)),
		q, SLOT(_n_updatePosition(qint64)));
	QObject::connect(audioOutput, SIGNAL(volumeChanged(float)),
		q, SLOT(_n_updateVolume(float)));
}

void PlayerPrivate::_n_updateState(Phonon::State newState, Phonon::State oldState)
{
	if (newState == Phonon::ErrorState)
	{
		kDebug(66666) << "error " << mediaObject->errorString();
		mediaObject->stop();
		emit q->stopped();
		emit q->stateChanged(Player::StoppedState);
		emit q->errorOccurred(Player::NormalError, mediaObject->errorString());
	}
	else
	{
		kDebug(66666) << "old: " << oldState << "; new: " << newState;
		switch (newState)
		{
		case Phonon::PlayingState:
			emit q->playing();
			break;
		case Phonon::StoppedState:
			emit q->stopped();
			break;
		case Phonon::PausedState:
			emit q->paused();
			break;
		default:
			break;
		}
		emit q->stateChanged(convertState(newState));
	}
}

void PlayerPrivate::_n_finishedPlaying()
{
	kDebug(66666) << " WOOHOO, PLAYING FINISHED";
	// delaying this is helpful for phonon-xine
	QTimer::singleShot(0, q, SLOT(next()));
}

void PlayerPrivate::_n_updateMetaData()
{
/* X == mapped keys
 X ALBUM
 X ARTIST
 X DATE
 DESCRIPTION (unmapped because it contains shoutcast serverinfos, ugly)
 X GENRE
 X TITLE
 TRACKNUMBER (unmapped, did we have a standard key in noatun for that one?)
*/
	foreach(const QString key, mediaObject->metaData().keys())
		kDebug(66666) << key << " => " << mediaObject->metaData(key);

	QVariant val;

	val = mediaObject->metaData(QLatin1String("TITLE"));
	//if (!val.isEmpty())
		currentItem.setProperty("title", val.toString());

	val = mediaObject->metaData(QLatin1String("ARTIST"));
	//if (!val.isEmpty())
		currentItem.setProperty("author", val.toString());

	val = mediaObject->metaData(QLatin1String("ALBUM"));
	//if (!val.isEmpty())
		currentItem.setProperty("album", val.toString());

	val = mediaObject->metaData(QLatin1String("DATE"));
	//if (!val.isEmpty())
		currentItem.setProperty("date", val.toString());

	val = mediaObject->metaData(QLatin1String("GENRE"));
	//if (!val.isEmpty())
		currentItem.setProperty("genre", val.toString());
}

void PlayerPrivate::_n_updateLength(qint64 msecLen)
{
	kDebug(66666) << "new length" << msecLen;
	currentItem.setProperty("length", QString::number(msecLen));
	emit q->lengthChanged((int)msecLen);
}

void PlayerPrivate::_n_updatePosition(qint64 msecPos)
{
	//kDebug(66666) << "new pos" << msecPos;
	emit q->positionChanged((int)msecPos);
}

void PlayerPrivate::_n_updateVolume(float vol)
{
	//kDebug(66666) << "new volume" << vol;
	emit q->volumeChanged((unsigned int)(100.00 * vol + 0.5));
}

void PlayerPrivate::_n_pluginLoaded(Plugin *plug)
{
	if (ItemSourceInterface *f = plug->interface<ItemSourceInterface>())
	{
		kDebug(66666) << "New plugin with ItemSourceInterface";
		itemSource = f;
	}
}

void PlayerPrivate::_n_pluginUnloaded(Plugin *plug)
{
	if (ItemSourceInterface *f = plug->interface<ItemSourceInterface>())
	{
		if (itemSource == f)
		{
			kDebug(66666) << "Plugin with ItemSourceInterface gone";
			itemSource = 0;
			q->stop();
			currentItem = PlaylistItem();
			emit q->currentItemChanged(currentItem);
		}
	}
}








// -----------------------------------------------------------------------------


Player::Player(Global *nInstance) : QObject(nInstance),
	d(new PlayerPrivate())
{
	d->q = this;
	setObjectName("Player");

	d->mediaObject     = 0;
	d->audioOutput     = 0;
	d->nInstance       = nInstance;
	d->itemSource      = 0;

	connect(d->nInstance->pluginHandler(), SIGNAL(pluginLoaded(Plugin *)),
		this, SLOT(_n_pluginLoaded(Plugin *)));
	connect(d->nInstance->pluginHandler(), SIGNAL(pluginUnloaded(Plugin *)),
		this, SLOT(_n_pluginUnloaded(Plugin *)));

	d->initPhonon();
}


Player::~Player()
{
	delete d;
	kDebug(66666) ;
}


PlaylistItem Player::currentItem() const
{
	return d->currentItem;
}


Player::State Player::state() const
{
	if (!d->mediaObject)
	{
		kWarning(66666) << "NO MEDIAOBJECT";
		return Player::StoppedState;
	}
	return d->convertState(d->mediaObject->state());
}


bool Player::isPlaying() const
{
	return state() == Player::PlayingState;
}


bool Player::isPaused() const
{
	return state() == Player::PausedState;
}


bool Player::isStopped() const
{
	return state() == Player::StoppedState;
}


bool Player::isActive() const
{
	Player::State s = state();
	return s == Player::PausedState || s == Player::PlayingState;
}


void Player::stop()
{
	if (isStopped())
		return;
	if (!d->mediaObject)
	{
		kWarning(66666) << "NO MEDIAOBJECT";
		return;
	}
	d->mediaObject->stop();
}


void Player::pause()
{
	if (!isPlaying())
		return;
	if (!d->mediaObject)
	{
		kWarning(66666) << "NO MEDIAOBJECT";
		return;
	}
	d->mediaObject->pause();
}


void Player::play()
{
	Player::State st = state();
	if (st == Player::PlayingState)
		return;
	if (st == Player::PausedState)
		d->mediaObject->play(); // unpause
	else
		play(d->currentItem);
}


void Player::play(const PlaylistItem &item)
{
	kDebug(66666) << "Attempting to play...";
	if (item.isNull() || !d->itemSource || !d->mediaObject)
		return;

	kDebug(66666) << "Starting to play new current track";
	d->currentItem = item; // TODO: remove
	d->itemSource->setCurrent(item);
	d->mediaObject->setCurrentSource(item.url());
	d->mediaObject->play();
	emit currentItemChanged(d->currentItem);
}


void Player::playpause()
{
	if (isPlaying())
		pause();
	else
		play();
}


void Player::next()
{

	if (d->itemSource)
	{
		PlaylistItem item = d->itemSource->forward();
		play(item);
	}
	else
		kWarning(66666) << "Missing an itemsource for switching to next item";
}


void Player::previous()
{
	if (d->itemSource)
	{
		PlaylistItem item = d->itemSource->backward();
		play(item);
	}
	else
		kWarning(66666) << "Missing an itemsource for switching to previous item";
}


void Player::setPosition(unsigned int msec)
{
	if (d->mediaObject && isActive())
	{
		kDebug(66666) << "msec = " << msec;
		d->mediaObject->seek(msec);
		// Phonon is async, do not expect position() to have changed immediately
	}
}

unsigned int Player::position() const
{
	if (!d->mediaObject)
	{
		kWarning(66666) << "NO MEDIAOBJECT";
		return 0;
	}
	return d->mediaObject->currentTime();
}

QString Player::positionString() const
{
	/*if (d->nInstance->config()->displayRemaining())
	{
		int len = currentLength();
		if (len > 0)
			return formatDuration(len - position());
	}*/
	return formatDuration(position());
}

unsigned int Player::currentLength() const
{
	if (!d->mediaObject)
	{
		kWarning(66666) << "NO MEDIAOBJECT";
		return 0;
	}
	return d->mediaObject->totalTime();
}

QString Player::lengthString() const
{
	return Noatun::formatDuration(currentLength());
}

unsigned int Player::volume() const
{
	if (!d->audioOutput)
	{
		kWarning(66666) << "Missing AudioOutput";
		return 0u;
	}
	return (unsigned int)(100.00 * d->audioOutput->volume() + 0.5);
}

void Player::setVolume(unsigned int percent)
{
	percent = qBound(percent, 0u, 100u);
	if (!d->audioOutput)
	{
		kWarning(66666) << "Missing AudioOutput";
		return;
	}
	d->audioOutput->setVolume(percent * 0.01);
}

QStringList Player::mimeTypes() const
{
	return Phonon::BackendCapabilities::availableMimeTypes();
}

QString Player::currentTitle() const
{
	return title(d->currentItem);
}


QString Player::title(PlaylistItem item) const
{
	if (item.isNull())
		return QString();

	if (item.hasProperty("realtitle"))
		return item.property("realtitle");

	// "$(property)"
	QString format = d->nInstance->config()->titleFormat();

	QRegExp find("(?:(?:\\\\\\\\))*\\$\\((.*)");

	int start = 0;
	while (start != -1)
	{
		start = find.indexIn(format, start);
		if (start == -1)
			break;

		// test if there's an odd amount of backslashes
		if (start > 0 && format[start-1] == '\\')
		{
			// yes, so half the amount of backslashes

			// count how many there are first
			QRegExp counter("([\\\\]+)");
			counter.indexIn(format, start - 1);
			uint len = counter.cap(1).length() - 1;

			// and half them, and remove one more
			format.replace(start-1, len/2+1, "");
			start = start - 1 + len / 2 + find.cap(1).length() + 3;
			continue;
		}

		// now replace the backslashes with half as many
		if (format[start] == '\\')
		{
			// count how many there are first
			QRegExp counter("([\\\\]+)");
			counter.indexIn(format, start);
			uint len = counter.cap(1).length();

			// and half them
			format.replace(start, len / 2, "");
			start = start + len / 2;
		}

		// "sth"foo"sth"
		QString cont(find.cap(1));
		QString prefix, suffix, propname;
		int i = 0;
		if (cont[i] == '"')
		{
			i++;
			for (; i < cont.length(); i++)
			{
				if (cont[i] != '"')
					prefix += cont[i];
				else
					break;
			}
			i++;
		}


		for (; i < cont.length(); ++i)
		{
			if (cont[i]!='"' && cont[i]!=')')
				propname += cont[i];
			else
				break;
		}

		if (cont[i] == '"')
		{
			i++;
			for (; i < cont.length(); i++)
			{
				if (cont[i] != '"')
					suffix += cont[i];
				else
					break;
			}
			i++;
		}
		i++;


		QString propval = item.property(propname);
		if (propname == "title" && propval.isEmpty())
		{
			propval = item.url().fileName();
		}
		else if (propname == "comment")
		{
			// comments can contain newlines
			// these are not wanted in a formatted title
			propval.replace('\n', ' ');
		}

		if (propval.length() > 0)
		{
			propval = prefix + propval + suffix;
			format.replace(start, i+2, propval);
			start += propval.length();
		}
		else
		{
			format.replace(start, i+2, "");
		}
	} // while()
	return format;
} // QString title(PlaylistItem item) const

} // namespace Noatun

#include "player.moc"
