/* This file is part of Noatun

  Copyright 2000-2006 by Charles Samuels <charles@kde.org>
  Copyright 2003-2007 by Stefan Gehn <mETz81@web.de>

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "noatun/playlistsaver.h"

#include <qdom.h>
#include <qfile.h>
#include <qregexp.h>
#include <qxml.h>
#include <qtextcodec.h>

#include <kio/netaccess.h>
#include <kconfig.h>
#include <kmimetype.h>
#include <klocale.h>
#include <ksavefile.h>
#include <ktemporaryfile.h>

#include <kdebug.h>

namespace Noatun
{


QStringList PlaylistSaver::loadableMimeTypes() // static member
{
	return QStringList() <<
		"text/xml" << "application/xml" << // Noatun Playlist
		"audio/mpegurl" << "audio/x-mpegurl" << "application/m3u" << // M3U
		"audio/x-scpls" << // PLS
		"audio/x-ms-asx"; // ASX
}


QStringList PlaylistSaver::savableMimeTypes() // static member
{
	return QStringList() <<
		"text/xml" << "application/xml" << // Noatun Playlist
		"audio/mpegurl" << "audio/x-mpegurl" << "application/m3u"; // M3U
}


PlaylistSaver::PlaylistSaver()
{
}


PlaylistSaver::~PlaylistSaver()
{
}


bool PlaylistSaver::save(const KUrl &url, const QString &mimeType)
{
	kDebug(66666) << "url:" << url;

	mLastMimeType.clear();

	if (savableMimeTypes().contains(mimeType) == false)
	{
		kWarning(66666) << "unsupported mimeType " << mimeType;
		return false;
	}

	bool ret = false;
	if (url.isLocalFile())
	{ // local files
		KSaveFile saveFile;
		saveFile.setFileName(url.path());
		//TODO: make use of saveFile.errorString();

		if (saveFile.open())
		{
			if (save(saveFile, mimeType)) // saving worked
			{
				ret = saveFile.finalize();
			}
			else
			{
				saveFile.abort();
			}
		}
		else
		{
			kWarning(66666) << "Couldn't open playlist file";
		}
	}
	else
	{ // remote files
		KTemporaryFile tmpF;

		if (tmpF.open())
		{
			if (save(tmpF, mimeType)) // saving worked
			{
				ret = KIO::NetAccess::upload(tmpF.fileName(), url, 0);
			}
		}
		else
		{
			kWarning(66666) << "Couldn't open tempfile " <<
				tmpF.fileName() << endl;
		}
	}

	if (ret)
		mLastMimeType = mimeType;

	return ret;
}


bool PlaylistSaver::load(const KUrl &url, const QString &mimeType)
{
	kDebug(66666) << "url: " << url << "; mimeType: '" << mimeType << '\'';

	mLastMimeType.clear();

	if (!mimeType.isEmpty())
	{
		if (loadableMimeTypes().contains(mimeType) == false)
		{
			kWarning(66666) << "unsupported mimetype " << mimeType;
			return false;
		}
	}

	QString localFile;
	if(!KIO::NetAccess::download(url, localFile, 0))
		return false;

	kDebug(66666) << "localFile: " << localFile;

	QString mt = mimeType;
	if (mimeType.isEmpty())
	{
		KMimeType::Ptr mimePtr = KMimeType::findByPath(localFile);
		mt = mimePtr->name();
	}

	bool res = false;
	if (mt == "audio/mpegurl" || mt == "audio/x-mpegurl" || mt == "application/m3u")
		res = loadM3U(localFile, url);
	else if (mt == "audio/x-scpls")
		res = loadPLS(localFile, url);
	else if (mt == "audio/x-ms-asx")
		res = loadASX(localFile, url);
	else if (mt == "text/xml" || mt == "application/xml")
		res = loadXML(localFile);

	if (res)
		mLastMimeType = mt;

	KIO::NetAccess::removeTempFile(localFile);
	return res;
}

const QString &PlaylistSaver::lastMimeType() const
{
	return mLastMimeType;
}

bool PlaylistSaver::save(QFile &file, const QString &mt) // private
{
	QTextStream stream(&file);

	if (mt == "audio/mpegurl" || mt == "audio/x-mpegurl" || mt == "application/m3u")
		return saveM3U(stream);

	if (mt == "text/xml" || mt == "application/xml")
		return saveXML(stream);

	return false;
}




class NoatunXMLStructure : public QXmlDefaultHandler
{
public:
	PlaylistSaver *saver;
	bool fresh;

	NoatunXMLStructure(PlaylistSaver *s)
		: saver(s), fresh(true)
	{
	}

	bool startElement(const QString&, const QString &, const QString &name,
		const QXmlAttributes &attr)
	{
		if (fresh)
		{
			fresh = (name != "playlist");
			// if we are still marked as "fresh" then the xml file did not start
			// with a playlist-tag and is thus INVALID.
			return (!fresh);
		}

		if (name == "item")
		{
			Noatun::PropertyMap propMap;
			int attrCount = attr.count();
			for (int i = 0; i < attrCount; i++)
			{
				propMap[attr.qName(i)] = attr.value(i);
			}
			saver->readItem(propMap);
		}

		return true;
	}
};


class MSASXStructure : public QXmlDefaultHandler
{
public:
	PlaylistSaver *saver;
	bool fresh;
	bool inEntry, inTitle;
	Noatun::PropertyMap propMap;
	QString mAbsPath;

	MSASXStructure(PlaylistSaver *s, const QString &absPath)
		: saver(s), fresh(true), inEntry(false),
		inTitle(false), mAbsPath(absPath)
	{
		//kDebug(66666) ;
	}

	bool startElement(const QString&, const QString &,
		const QString &name, const QXmlAttributes &a)
	{
		if (fresh)
		{
			fresh = (name.toLower() != "asx");
			// if we are still marked as "fresh" then the xml file did not start
			// with an asx-tag and is thus INVALID.
			return (!fresh);
		}

		if (name.toLower() == "entry")
		{
			if(inEntry) // WHOOPS, we are already in an entry, this should NEVER happen
			{
				kDebug(66666) << "STOP, ENTRY INSIDE ENTRY!";
				return false;
			}
			inEntry=true;
		}
		else
		{
			if (inEntry) // inside entry block
			{
				// known stuff inside an <entry> ... </entry> block:
				// <title>blah</title>
				// <param album="blub" />
				// <param artist="blah" />
				// <ref HREF="file:/something" />
				if(name.toLower()=="ref")
				{
					for (int i=0; i<a.count(); i++)
					{
						if (a.qName(i).toLower() == "href")
						{
							QString filename = a.value(i);
							if (filename.contains(QRegExp("^[a-zA-Z0-9]+:/")))
							{
								KUrl url(filename);
								KMimeType::Ptr mimetype = KMimeType::findByUrl(url);
								QString type = mimetype->name();
								if (type != "application/octet-stream")
								{
									propMap["url"]=filename;
								}
								else
								{
									//propMap["playObject"]="SplayPlayObject";
									propMap["title"] = i18n("Stream from %1",url.host());
									if (!url.hasPath())
										url.setPath("/");
									propMap["url"] = url.url();
									propMap["stream_"]=propMap["url"];
								}
							}
							else
							{
								KUrl u1;
								// we have to deal with a relative path
								if (filename.contains('/'))
								{
									u1.setPath(mAbsPath); //FIXME: how to get the path in this place?
									u1.setFileName(filename);
								}
								else
								{
									u1.setPath(filename);
								}
								propMap["url"]=u1.url();
							}
						}
					}
				}
				else if(name.toLower()=="param")
				{
					QString keyName="", keyValue="";

					for (int i=0; i<a.count(); i++)
					{
						if(a.value(i).toLower()=="album")
							keyName="album";
						else if(a.value(i).toLower()=="artist")
							keyName="author";
						else if(!keyName.isEmpty()) // successfully found a key, the next key=value pair has to be the value
						{
							keyValue=a.value(i);
						}
					}

					if (!keyName.isEmpty() && !keyValue.isEmpty())
					{
						propMap[keyName]=keyValue;
					}
				}
				else if(name.toLower()=="title")
				{
					if(inTitle) // WHOOPS, we are already in an entry, this should NEVER happen
					{
						kDebug(66666) << "STOP, TITLE INSIDE TITLE!";
						return false;
					}
//					kDebug(66666) << "<TITLE> ======";
					inTitle=true;
				}
/*				else
				{
					kDebug(66666) << "Unknown/unused element inside ENTRY block, NAME='" << name << "'";
					for (int i=0; i<a.count(); i++)
						kDebug(66666) << " | " << a.qName(i) << " = '" << a.value(i) << "'";
				}*/
			}
/*			else
			{
				kDebug(66666) << "Uninteresting element, NAME='" << name << "'";
				for (int i=0; i<a.count(); i++)
					kDebug(66666) << "  | " << a.qName(i) << " = '" << a.value(i) << "'";
			}*/
		}

		return true;
	}

	bool endElement(const QString&,const QString&, const QString& name)
	{
		if (!inEntry)
			return false;

		if (name.toLower() == "entry")
		{
			saver->readItem(propMap);
			propMap.clear();
			inEntry = false;
		}
		else if (name.toLower() == "title")
		{
			if(inTitle)
			{
				inTitle = false;
			}
			else if (inTitle) // found </title> without a start or not inside an <entry> ... </entry>
			{
				//kDebug(66666) << "STOP, </TITLE> without a start";
				return false;
			}
		}
		return true;
	}

	bool characters(const QString &ch)
	{
		if(inTitle && !ch.isEmpty())
			propMap["title"] = ch;
		return true;
	}
};


bool PlaylistSaver::loadASX(const QString &file, const KUrl &originalUrl)
{
	kDebug(66666) << "url:" << originalUrl;
	QFile f(file);
	if (!f.open(QIODevice::ReadOnly))
		return false;

	reset();

	QXmlInputSource source(&f);
	QXmlSimpleReader reader;

	MSASXStructure ASXparser(this, originalUrl.path());
	reader.setContentHandler(&ASXparser);
	reader.parse(source);
	return !ASXparser.fresh;
}

bool PlaylistSaver::loadXML(const QString &file)
{
	kDebug(66666) << "file:" << file;

	QFile f(file);
	if (!f.open(QIODevice::ReadOnly))
		return false;

	reset();

	QXmlInputSource source(&f);
	QXmlSimpleReader reader;

	NoatunXMLStructure parser(this);
	reader.setContentHandler(&parser);
	reader.parse(source);
	return !parser.fresh;
}

bool PlaylistSaver::loadM3U(const QString &file, const KUrl &originalUrl)
{
	kDebug(66666) << "originalUrl: " << originalUrl;
	QFile f(file);
	if (!f.open(QIODevice::ReadOnly))
		return false;

	QTextStream t(&f);

	bool isExt = false; // flag telling if we load an EXTM3U file
	QString filename;
	QString extinf;

	reset();

	while (!t.atEnd())
	{
		Noatun::PropertyMap prop;

		if (isExt)
		{
			extinf = t.readLine().trimmed();
			if (!extinf.startsWith("#EXTINF:"))
			{
				filename = extinf;
				extinf.clear();
			}
			else
			{
				filename = t.readLine().trimmed(); // read in second line containing the filename
			}
		}
		else // old style m3u
		{
			filename = t.readLine().trimmed();
		}

		if (filename == "#EXTM3U") // on first line
		{
			isExt = true;
			continue; // skip parsing the first (i.e. this) line
		}

		if (filename.isEmpty())
			continue;

		kDebug(66666) << "filename: '" << filename << "'";

		if (filename.contains(QRegExp("^[a-zA-Z0-9]*:/")))
		{
			// filename is url-style (i.e. "proto:/foo" format)
			KUrl fileUrl(QUrl::toPercentEncoding(filename));
			/*if (KMimeType::findByUrl(fileUrl)->name() == "application/octet-stream")
				prop["title"] = i18n("Stream from %1", fileUrl.host());*/
			prop["url"] = fileUrl.url();
		}
		else
		{
			// filename that is not in url-style (i.e. plain path without a protocol)

			KUrl localUrl;
			if (KUrl::isRelativeUrl(filename))
			{
				// prepend path from playlist file
				localUrl.setPath(originalUrl.path());
				// append filename
				localUrl.setFileName(filename);
			}
			else
			{
				localUrl = KUrl::fromPath(filename);
			}
			prop["url"] = localUrl.url();
		}

		// parse line of the following format:
		//#EXTINF:length,displayed_title
		if (isExt)
		{
			extinf.remove(0,8); // remove "#EXTINF:"
			int timeTitleSep = extinf.indexOf(',');

			int length = (extinf.left(timeTitleSep)).toInt();
			if (length > 0)
				prop["length"] = QString::number(length * 1000);

			QString displayTitle = extinf.mid(timeTitleSep + 1);
			if (!displayTitle.isEmpty())
			{
				int artistTitleSep = displayTitle.indexOf(" - ");
				if (artistTitleSep == -1) // no "artist - title" like format, just set it as title
				{
					prop["title"] = displayTitle;
				}
				else
				{
					prop["author"] = displayTitle.left(artistTitleSep);
					prop["title"] = displayTitle.mid(artistTitleSep+3);
				}
			} // END !displayTitle.isEmpty()
		} // END if(isExt)

		// pass properties list to playlist
		//kDebug(66666) << "prop:" << prop;
		readItem(prop);
	} // END while()

	return true;
}

static QString findNoCase(const Noatun::PropertyMap &map, const QString &key)
{
	Noatun::PropertyMap::ConstIterator it(map.begin());
	Noatun::PropertyMap::ConstIterator end(map.end());
	const QString lowerKey = key.toLower();
	for ( ; it != end; ++it)
	{
		if (it.key().toLower() == lowerKey)
			return it.value();
	}
	return QString();
}

bool PlaylistSaver::loadPLS(const QString &file, const KUrl &originalUrl)
{
	kDebug(66666) ;
	QFile checkFile(file);
	if (!checkFile.open(QIODevice::ReadOnly))
		return false;
	QTextStream t(&checkFile);
	QString firstLine = t.readLine().trimmed().toLower();
	if (firstLine != "[playlist]")
	{
		kWarning(66666) << "PLS file '" << file <<
			"' did not start with '[playlist]', aborting loading" << endl;
		return false;
	}
	checkFile.close();

	KConfig list(file, KConfig::OnlyLocal);
	// some windows users like to be case insensitive, oh my
	QStringList groups = list.groupList().filter(QRegExp("^playlist$", Qt::CaseInsensitive));
	QMap<QString, QString> group = list.entryMap(groups[0]);

	QString numOfEntries = findNoCase(group, "numberofentries");
	if(numOfEntries.isEmpty())
		return false;

	reset();

	unsigned int nEntries = numOfEntries.toUInt();
	for(unsigned int entry = 1; entry <= nEntries; ++entry )
	{
		Noatun::PropertyMap prop;

		QString filename = findNoCase(group, "file"+QString::number(entry));
		QString title    = findNoCase(group, "title"+QString::number(entry));

		kDebug(66666) << "filename:" << filename;

		KUrl fileUrl;
		if (filename.contains(QRegExp("^[a-zA-Z0-9]*:/")))
		{
			fileUrl = KUrl(QUrl::toPercentEncoding(filename));
		}
		else
		{
			if (KUrl::isRelativeUrl(filename))
			{
				// prepend path from playlist file
				fileUrl.setPath(originalUrl.path());
				// append filename
				fileUrl.setFileName(filename);
			}
			else
			{
				fileUrl = KUrl::fromPath(filename);
			}
		}

		prop["url"] = fileUrl.url();
		if (!title.isEmpty())
			prop["title"] = title;

		//kDebug(66666) << "prop:" << prop;
		readItem(prop);
	}
	return true;
}

bool PlaylistSaver::saveM3U(QTextStream &stream)
{
	reset();
	PlaylistItem i;
	QStringList props;

	stream << "#EXTM3U" << '\n';

	while ((i = writeItem()))
	{
		int length = (int)(i.property("length").toInt() / 1000);

		// special value in an EXTM3U file, means "unknown length"
		if (length == 0)
			length = -1;

		const KUrl u(i.property("url"));
		QString title;

	// if a playlistitem is without a tag or ONLY title is set
		if((i.property("author").isEmpty() && i.property("title").isEmpty())
			|| (i.property("author").isEmpty() && !i.property("title").isEmpty()) )
		{
			title = u.fileName();
			if (title.contains('.'))
				title = title.left(title.indexOf('.'));
		}
		else
		{
			title = i.property("author") + " - " + i.property("title");
		}

		stream << "#EXTINF:"<< QString::number(length) << "," << title << '\n';

		if (u.isLocalFile())
			stream << u.path() << '\n';
		else
			stream << u.url() << '\n';
	}
	return true;
}

bool PlaylistSaver::saveXML(QTextStream &stream)
{
	const PlaylistItem &i = PlaylistItem();
	QDomDocument doc("playlist");
	doc.setContent(QString("<!DOCTYPE XMLPlaylist><playlist version=\"2.0\" client=\"noatun\"/>"));
	QDomElement docElem = doc.documentElement();

	reset();

	while ((i = writeItem()))
	{
		// write all properties
		const QStringList &props = i.properties();
		QDomElement elem = doc.createElement("item");

		foreach(const QString &propertyName, props)
		{
			const QString propertyValue = i.property(propertyName);
			elem.setAttribute(propertyName, propertyValue);
		}
		docElem.appendChild(elem);
	}

	stream.setCodec(QTextCodec::codecForName("UTF-8"));
	stream << doc.toString();
	return true;
}

} // namespace Noatun
