/***************************************************************************
	kjseeker.cpp
	---------------------------------------------
	slider that lets the user jump inside the currently played file
	---------------------------------------------
	Maintainer: Stefan Gehn <sgehn@gmx.net>

 ***************************************************************************/

#include "kjseeker.h"
#include "kjloader.h"

#include "helpers.cpp"
#include <noatun/global.h>
#include <noatun/player.h>
#include <cstring>

#include <kdebug.h>

KJSeeker::KJSeeker(const QStringList &i, KJMainWindow *l) : KJWidget(l), g(0)
{
	QString activeBg = backgroundPressed("bmp1");
	if(activeBg.isEmpty())
	{
		kDebug(66666) << "No pressed background found for seeker," <<
			" using default background!" << endl;
		parentWin()->image(parser()["backgroundimage"][1]);
	}
	else
		mActive = parentWin()->image(activeBg);

	mScale = parentWin()->image(parser()["seekimage"][1]);
	QImage pixmapNoPress = parentWin()->image(parser()["backgroundimage"][1]);
	std::memset(barmode, 0, sizeof(barmode));
	std::memset(barmodeImages, 0, sizeof(barmode));

	// generate transparent mask
	int x, y, xs, ys;
	x=i[1].toInt();
	y=i[2].toInt();
	xs=i[3].toInt()-x;
	ys=i[4].toInt()-y;
	setRect(x,y,xs,ys);
	QImage transmask(xs, ys, 1, 2, QImage::LittleEndian);
#if QT_VERSION < 0x030300
	transmask.setColor(0, qRgb(0,0,0));
	transmask.setColor(1, qRgb(255,255,255));
#else
	transmask.setColor(1, qRgb(0,0,0));
	transmask.setColor(0, qRgb(255,255,255));
#endif

	// clear the pointers
	memset(barmodeImages, 0, 256*sizeof(QImage*));
	memset(barmode, 0, 256*sizeof(QPixmap*));

	// Now do the pixel fking
//	kDebug(66666) << "creating Pixmaps for Seeker";
	for (int iy=y;iy<y+ys; iy++)
	{
		for (int ix=x;ix<x+xs; ix++)
		{
			QRgb checkmScale = mScale.pixel(ix, iy);
			// am I transparent?
			if (!isGray(checkmScale))
			{
				setPixel1BPP(transmask, ix-x, iy-y, 0);
				continue;
			}
			setPixel1BPP(transmask, ix-x, iy-y, 1);

			// what is the level
			int level=grayRgb(checkmScale)+1;
			if (level>255) level=255;
			// allocate the pixmap of the level proper
			// copy the color to the surface proper
			QRgb activeColor=mActive.pixel(ix,iy);
			QRgb inactiveColor=pixmapNoPress.pixel(ix,iy);
			// set this pixel and everything before it
			for(int i=0; i<level; i++)
			{
				if (!barmodeImages[i])
					barmodeImages[i]=new QImage(xs,ys, 32);
				QRgb *l=(QRgb*)barmodeImages[i]->scanLine(iy-y);
				l[ix-x]=inactiveColor;
			}

			do
			{
				if (!barmodeImages[level])
					barmodeImages[level]=new QImage(xs,ys, 32);
				QRgb *l=(QRgb*)barmodeImages[level]->scanLine(iy-y);
				l[ix-x]=activeColor;
			} while (level++<255);
		}
	}
//	kDebug(66666) << "finished creating Pixmaps";

	// create the blank one
	barmode[0]=new QPixmap(xs, ys);
	QPixmap px=parentWin()->pixmap(parser()["backgroundimage"][1]);
	bitBlt(barmode[0], 0, 0, &px, x, y, xs, ys, Qt::CopyROP);
	px.convertFromImage(transmask);
	barModeMask=px;

//	kDebug(66666) << "END KJSeeker constructor";
}

QPixmap *KJSeeker::toPixmap(int n)
{
	if (!barmodeImages[n]) return barmode[n];

	barmode[n]=new QPixmap(
			barmodeImages[n]->width(),
			barmodeImages[n]->height()
		);
	barmode[n]->convertFromImage(*barmodeImages[n]);

	delete barmodeImages[n];
	barmodeImages[n]=0;
	return barmode[n];
}


KJSeeker::~KJSeeker()
{
	for (uint i=0; i<256; ++i)
	{
		if (barmode[i])
			delete barmode[i];
		if (barmodeImages[i])
			delete barmodeImages[i];
	}
}

void KJSeeker::paint(QPainter *p, const QRect &)
{
	closest();
	QPixmap *pixmap = toPixmap(g);
	pixmap->setMask(barModeMask);
	bitBlt(p->device(), rect().topLeft().x(), rect().topLeft().y(),
		pixmap, 0, 0, rect().width(), rect().height(), Qt::CopyROP);
}

bool KJSeeker::mousePress(const QPoint &pos)
{
	return (isGray(mScale.pixel(rect().topLeft().x()+pos.x(), rect().topLeft().y()+pos.y())));
}

void KJSeeker::mouseRelease(const QPoint &pos, bool in)
{
	int x, y;
	QRgb color;
	Player *p;

	x = rect().topLeft().x()+pos.x();
	y = rect().topLeft().y()+pos.y();
	p = global()->player();

	if(p->isStopped() || !mScale.valid(x, y))
		return;

	color = mScale.pixel(x, y);

	// user released mousebutton outside of the seeker-area (which is gray)
	if (!isGray(color) || !in)
		return;

	g = grayRgb(color);
	repaint();

//	kDebug(66666) << "length : " << p->length();
//	kDebug(66666) << "skip to: " << ((long long)g*(long long)p->length())/255;

	// g * titlelength can get REALLY HUGE, that's why I used (long long)
	p->seek(((long long)g * (long long)p->length()) / 255);
}

void KJSeeker::timeUpdate(int sec)
{
	int length = qMax(0, global()->player()->length() / 1000);
	sec = KCLAMP(sec, 0, length);
	g = (length == 0) ? 0 : (sec * 255 / length);
	//kDebug(66666) << "sec: " << sec << " len: " << length << " g: " << g;
	QPainter p(parentWin());
	paint(&p, rect());
}

void KJSeeker::closest()
{
	int south=g, north=g;
	bool southtried=false, northtried=false;
	while (
			!barmode[south] && !barmodeImages[south]
			&& !barmode[north] && !barmodeImages[north])
	{
		if (southtried && northtried) { g=0; return; }
		south--;
		north++;
		if (north>255) {northtried=true; north=g;}
		if (south<0) {southtried=true; south=g;}
	}
	if (barmode[south] || barmodeImages[south])
		g=south;
	else if (barmode[north] || barmodeImages[north])
		g=north;
}
