/***************************************************************************
                     A simple database-driven playlist for Noatun
                    ---------------------------------------------
    begin                : 24.05.2005
    copyright            : (C) 2005 by Stefan Gehn
    email                : Stefan Gehn <mETz81@web.de>
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "simplelistview.h"
#include "simpleplaylist.h"
#include "simpledb.h"

#include <qpainter.h>
#include <qfont.h>
#include <kglobal.h> // KMAX
#include <kurldrag.h>
#include <kiconloader.h>
#include <kaction.h>
#include <kmenu.h>
#include <qevent.h>
#include <qheader.h>
#include <qcursor.h>

SimpleListView::SimpleListView(SimplePlaylist *pl, QWidget *parent, const char *name)
	: K3ListView(parent, name), mCurrentListItem(0), playlist(pl), mDropMenu(0), mDropAfterItem(0)
{
	mLeftColumnIndex = 0;

	addColumn(i18n("Title"), 64);
	addColumn(i18n("Artist"), 64);
	addColumn(i18n("Album"), 32);
	addColumn(i18n("Track"), 32);
	addColumn(i18n("Genre"), 32);
	addColumn(i18n("Year"), 32);
	addColumn(i18n("Length"), 32);
	addColumn(i18n("Bitrate"), 32);
	addColumn(i18n("Comment"), 32);

	setDragEnabled(true);
	setAcceptDrops(true);
	setDropVisualizer(true);
	setItemsMovable(true);
	setSelectionMode(QListView::Extended);
	setShowSortIndicator(true);
	setAllColumnsShowFocus(true);

	mViewColumnsAction = new KActionMenu(i18n("&Show Columns"), this, "view_columns");
	mHeaderMenu = mViewColumnsAction->popupMenu();
	mHeaderMenu->insertTitle(i18n("Show"));
	mHeaderMenu->setCheckable(true);

	for(int i = 0; i < header()->count(); ++i)
	{
		mHeaderMenu->insertItem(header()->label(i), i);
		mHeaderMenu->setItemChecked(i, true);
		//adjustColumn(i);
	}

	connect(mHeaderMenu, SIGNAL(activated(int)),
		this, SLOT(slotToggleColumns(int)));

	connect(playlist->db(), SIGNAL(itemInserted(const DBItem &)),
		this, SLOT(slotItemInserted(const DBItem &)));
	connect(playlist->db(), SIGNAL(itemDeleted(int)),
		this, SLOT(slotItemDeleted(int)));
	connect(playlist->db(), SIGNAL(itemUpdated(int)),
		this, SLOT(slotItemUpdated(int)));
	connect(playlist->db(), SIGNAL(itemUpdatedLength(int, int)),
		this, SLOT(slotItemUpdatedLength(int, int)));
	//playlist->db() SIGNAL(itemUpdatedPosition(int id, int oldNext, int newNext))

	connect(this, SIGNAL(moved(QListViewItem *, QListViewItem *, QListViewItem *)),
		this, SLOT(slotMoved(QListViewItem *, QListViewItem *, QListViewItem *)));
	connect(this, SIGNAL(dropped(QDropEvent*, QListViewItem*)),
		this, SLOT(slotDropped(QDropEvent*, QListViewItem*)));
	connect(this, SIGNAL(aboutToMove()), this, SLOT(slotDisableSorting()));
	connect(this, SIGNAL(executed(QListViewItem *)),
		SLOT(slotExecuted(QListViewItem *)));

	connect(header(), SIGNAL(indexChange(int, int, int)), this,
		SLOT(slotColumnOrderChanged(int, int, int)));

	restoreViewLayout();
}

SimpleListView::~SimpleListView()
{
	saveViewLayout();
}

void SimpleListView::restoreViewLayout()
{
	bool visible;
	restoreLayout(playlist->componentData().config(), "ListViewLayout");
	for (int c = 0; c < SimpleListViewItem::lastColumnIndex(); c++)
	{
		visible = isColumnVisible(c);
		mHeaderMenu->setItemChecked(c, visible);

		if (!visible)
		{
			setColumnWidthMode(c, Manual);
			//header()->moveSection(c, header()->count());
			header()->setResizeEnabled(false, c);
		}
	}
	mLeftColumnIndex = leftMostColumnIndex();
}

void SimpleListView::saveViewLayout()
{
	saveLayout(playlist->componentData().config(), "ListViewLayout");
}

void SimpleListView::refresh()
{
	kDebug(66666) ;
	clear();
	map.clear();
	setSorting(-1);

	int currentId = (mCurrentListItem ? mCurrentListItem->id() : -1);

	QMap<int, DBItem> items = playlist->db()->getOrderedItems();
	QMapIterator<int, DBItem> iter;
	int next = 0;
	while ((iter = items.find(next)) != items.end())
	{
		SimpleListViewItem *lvi = new SimpleListViewItem(this, iter.data());
		updatePosition(lvi, next);
		map.insert(lvi->id(), lvi);
		next = lvi->id();
	}

	if (currentId > 0)
		setCurrentByItemId(currentId);
	else
		setCurrentItem(static_cast<SimpleListViewItem *>(firstChild()));
}

void SimpleListView::slotItemInserted(const DBItem &item)
{
	//kDebug(66666) << "item id = " << item.id;

	QMapIterator<int, SimpleListViewItem*> iter = map.find(item.id);
	SimpleListViewItem *lvi;
	if (iter == map.end())
	{
		setSorting(-1);
		lvi = new SimpleListViewItem(this, item);
		updatePosition(lvi, item.next);
		map.insert(lvi->id(), lvi);
	}
	else
	{
		kDebug(66666) <<
			"OOOPS, ITEM WITH ID " << item.id << "ALREADY IN LIST?" << endl;
	}
}

void SimpleListView::slotItemDeleted(int id)
{
	//kDebug(66666) << "item id = " << item.id;

	QMapIterator<int, SimpleListViewItem*> iter = map.find(id);
	if (iter == map.end())
	{
		kDebug(66666) <<
			"OOOPS, ITEM WITH ID " << id << "MISSING IN LIST?" << endl;
	}
	else
	{
		if (iter.data() == mCurrentListItem)
		{
			// TODO: make it possible to go forward/backward even
			// when the current item has been deleted
			//setCurrentItem(0);
			kDebug(66666) <<
				"Current playlistitem was deleted, hiding until a new current item is selected" << endl;
			iter.data()->setVisible(false);
		}
		else
		{
			delete iter.data();
			map.remove(iter);
		}
	}
}

void SimpleListView::slotItemUpdated(int id)
{
	//kDebug(66666) << "id = " << id;

	QMapIterator<int, SimpleListViewItem*> iter = map.find(id);
	if (iter == map.end())
	{
		kDebug(66666) <<
			"OOOPS, ITEM WITH ID " << id << "MISSING IN LIST?" << endl;
	}
	else
	{
		DBItem item = playlist->db()->getItem(id);
		iter.data()->itemUpdated(item);
	}
}

void SimpleListView::slotItemUpdatedLength(int id, int newLength) // private
{
	//kDebug(66666) << "item id = " << id;

	QMapIterator<int, SimpleListViewItem*> iter = map.find(id);
	if (iter == map.end())
	{
		kDebug(66666) <<
			"OOOPS, ITEM WITH ID " << id << "MISSING IN LIST?" << endl;
	}
	else
	{
		iter.data()->setLength(newLength);
	}
}

void SimpleListView::updatePosition(SimpleListViewItem *lvi, int next) // private
{
	QMapIterator<int, SimpleListViewItem*> iter;
	if (!next) // end of list
	{
		lvi->moveItem(lastItem());
	}
	else
	{
		iter = map.find(next);
		if (iter != map.end())
		{
			lvi->moveItem(iter.data()->itemAbove());
		}
		else
		{
			kDebug(66666) << "Could not find next item (" <<
				next << ") for item " << lvi->id() <<
				"   -------------------------- " << endl;
		}
	}
}

void SimpleListView::slotExecuted(QListViewItem *qlvi) // private
{
	SimpleListViewItem *slvi = static_cast<SimpleListViewItem *>(qlvi);
	if (slvi)
		emit executedItem(slvi->id());
}

void SimpleListView::setCurrentByItemId(int id) // public
{
	if (mCurrentListItem && mCurrentListItem->id() == id)
		return;

	//kDebug(66666) << "id = " << id;

	QMapIterator<int, SimpleListViewItem*> iter = map.find(id);
	if (iter != map.end())
		setCurrentItem(iter.data());
	else
		kDebug(66666) << "ERROR: could not find item with id " << id;
}

void SimpleListView::setCurrentItem(SimpleListViewItem *item) // private
{
	if (mCurrentListItem == item)
		return;

	SimpleListViewItem *previousItem = mCurrentListItem;
	mCurrentListItem = item;

	if (previousItem)
	{
		if (!previousItem->isVisible())
		{
			kDebug(66666) << "previous item invisible, deleting it now";
			map.remove(previousItem->id());
			delete previousItem;
		}
		else
		{
			//previousItem->invalidateHeight();
			previousItem->setup();
		}
	}

	if (mCurrentListItem)
	{
		//mCurrentListItem->invalidateHeight();
		mCurrentListItem->setup();
		ensureItemVisible(mCurrentListItem);
	}

	triggerUpdate();
}

int SimpleListView::currentItemId() const // public
{
	return (mCurrentListItem ? mCurrentListItem->id() : -1);
}

SimpleListViewItem *SimpleListView::currentListItem() const // protected
{
	return mCurrentListItem;
}

int SimpleListView::nextItemId() const // public
{
	int id = -1;
	if (mCurrentListItem)
	{
		SimpleListViewItem *lvi = static_cast<SimpleListViewItem *>(mCurrentListItem->itemBelow());
		if (lvi)
			id = lvi->id();
	}
	return id;
}

int SimpleListView::previousItemId() const
{
	int id = -1;
	if (mCurrentListItem)
	{
		SimpleListViewItem *lvi = static_cast<SimpleListViewItem *>(mCurrentListItem->itemAbove());
		if (lvi)
			id = lvi->id();
	}
	return id;
}

int SimpleListView::firstItemId() const
{
	int id = -1;
	SimpleListViewItem *lvi = static_cast<SimpleListViewItem *>(firstChild());
	if (lvi)
		id = lvi->id();
	return id;
}

void SimpleListView::deleteSelected()
{
	kDebug(66666) << "called";
	QPtrList<QListViewItem> items(selectedItems());
	QPtrListIterator<QListViewItem> it(items);

	SimplePlaylistItem *plItem;
	for (; it.current(); ++it)
	{
		plItem = playlist->getById(static_cast<SimpleListViewItem*>(*it)->id());
		if (plItem)
			plItem->remove();
	}
}

void SimpleListView::executeCurrent() // public
{
	//kDebug(66666) << "called";
	slotExecuted(currentItem());
}

void SimpleListView::slotMoved(QListViewItem *lvi, QListViewItem *, QListViewItem *afterNow)
{
	if (!lvi)
		return;

	SimpleListViewItem *itemLVI = static_cast<SimpleListViewItem*>(lvi);
	SimpleListViewItem *afterLVI = static_cast<SimpleListViewItem*>(afterNow);

	/*DBItem dbitem = playlist->db()->getItem(itemLVI->id());
	if (!dbitem.isNull())
	{
		if (afterLVI)
		{
			playlist->db()->moveItem(dbitem, afterLVI->id(), false);
		}
		else // moved BEFORE the first item
		{
			//NOTE: firstChild() is where we are NOW, i.e. the one we moved before is below us
			SimpleListViewItem *lvi = static_cast<SimpleListViewItem *>(firstChild()->itemBelow());
			playlist->db()->moveItem(dbitem, lvi->id(), true);
		}
	}*/

	if (afterLVI) // moved AFTER afterLVI
	{
		playlist->db()->moveItem(itemLVI->id(), afterLVI->id(), false);
	}
	else // moved BEFORE the first item
	{
		//NOTE: firstChild() is where we are NOW, i.e. the one we moved before is below us
		SimpleListViewItem *lvi = static_cast<SimpleListViewItem *>(firstChild()->itemBelow());
		if (lvi)
			playlist->db()->moveItem(itemLVI->id(), lvi->id(), true);
	}
}

void SimpleListView::slotDropped(QDropEvent *event, QListViewItem *afterItem) // private slot
{
	mDroppedUrlsList.clear();
	if (!KURLDrag::decode(event, mDroppedUrlsList) || mDroppedUrlsList.isEmpty())
		return;

	mDropAfterItem = static_cast<SimpleListViewItem *>(afterItem);
	if (mDropAfterItem)
	{
		kDebug(66666) << "Drop after id " << mDropAfterItem->id();
	}

	if (!mDropMenu)
	{
		mDropMenu = new KMenu(this);
		mDropMenu->insertItem(i18n("Insert here"), 0);
		mDropMenu->insertItem(SmallIconSet("go-bottom"), i18n("Add to end of playlist"), 1);
		mDropMenu->insertSeparator();
		mDropMenu->insertItem(SmallIconSet("cancel"), i18n("Cancel"), 2);
		connect(mDropMenu, SIGNAL(activated(int)), SLOT(slotDropMenuActivated(int)));
	}

	mDropMenu->popup(mapToGlobal(contentsToViewport(event->pos())), 0);
}

void SimpleListView::slotDropMenuActivated(int item)
{
	SimpleListViewItem *addAfterItem;
	int addAfterId;
	KUrl::List::Iterator it;

	switch(item)
	{
		case 0: // add items at selected place in playlist
		{
			addAfterItem = mDropAfterItem;
			addAfterId = (addAfterItem ? addAfterItem->id() : -1);
			playlist->addFiles(mDroppedUrlsList, addAfterId, false);
			break;
		}
		case 1: // add items at end of playlist
		{
			addAfterItem = static_cast<SimpleListViewItem *>(lastItem());
			addAfterId = (addAfterItem ? addAfterItem->id() : -1);
			playlist->addFiles(mDroppedUrlsList, addAfterId, false);
			break;
		}
		case 2: // Cancel
			break;
	}
}

bool SimpleListView::acceptDrag(QDropEvent *event) const // virtual protected
{
	//kDebug(66666) ;
	return K3ListView::acceptDrag(event) || KURLDrag::canDecode(event);
}

QDragObject *SimpleListView::dragObject() // virtual protected
{
	//kDebug(66666) ;
	QPtrList<QListViewItem> items(selectedItems());
	QPtrListIterator<QListViewItem> it(items);
	//SimplePlaylistItem *plItem;
	KUrl::List urlList;

	for (; it.current(); ++it)
	{
		DBItem dbitem = playlist->db()->getItem(static_cast<SimpleListViewItem*>(*it)->id());
		if (!dbitem.isNull())
			urlList.append(dbitem.url);
		/*
		plItem = playlist->getById(static_cast<SimpleListViewItem*>(*it)->id());
		if (plItem)
			urlList.append(plItem->url());
		*/
	}

	KURLDrag *drag = new KURLDrag(urlList, viewport());
	drag->setPixmap(BarIcon("sound"));
	return drag;
}

void SimpleListView::slotDisableSorting()
{
	setSorting(-1);
}

void SimpleListView::sort() //virtual
{
	kDebug(66666) << "TODO";
	setSorting(-1);
	//K3ListView::sort();
	// TODO: Update order in db!
}

bool SimpleListView::eventFilter(QObject *watched, QEvent *e)
{
	if(watched == header())
	{
		if (e->type() == QEvent::MouseButtonPress &&
			static_cast<QMouseEvent *>(e)->button() == Qt::RightButton)
		{
			mHeaderMenu->popup(QCursor::pos());
		}
	}

	return K3ListView::eventFilter(watched, e);
}

KActionMenu *SimpleListView::viewColumnsAction() const // public
{
	return mViewColumnsAction;
}

void SimpleListView::slotToggleColumns(int col)
{
	kDebug(66666) << "For column " << col;
	if (isColumnVisible(col))
		hideColumn(col);
	else
		showColumn(col);
}

void SimpleListView::slotColumnOrderChanged(int, int fromColIndex, int toColIndex)
{
	if (fromColIndex == 0 || toColIndex == 0)
	{
		triggerUpdate();
		mLeftColumnIndex = header()->mapToSection(0);
		kDebug(66666) << "Left column index is now " << mLeftColumnIndex;
	}
}

int SimpleListView::leftColumnIndex() const // protected
{
	return mLeftColumnIndex;
}

int SimpleListView::leftMostColumnIndex() const // private
{
	int i = 0;
	while(!isColumnVisible(header()->mapToSection(i)) && i < SimpleListViewItem::lastColumnIndex())
		i++;
	return header()->mapToSection(i);
}

bool SimpleListView::isColumnVisible(int col) const // private
{
	return columnWidth(col) != 0;
}

void SimpleListView::hideColumn(int c) // private
{
	kDebug(66666) << "For column " << c;
	mHeaderMenu->setItemChecked(c, false);

	if(!isColumnVisible(c))
		return;

	setColumnWidthMode(c, Manual);
	setColumnWidth(c, 0);
	// Moving the column to the end seems to prevent it from randomly
	// popping up.
	//header()->moveSection(c, header()->count());
	header()->setResizeEnabled(false, c);

	if(c == mLeftColumnIndex)
	{
		triggerUpdate(); // update item pixmaps etc.
		mLeftColumnIndex = leftMostColumnIndex();
	}
}

void SimpleListView::showColumn(int c) // private
{
	mHeaderMenu->setItemChecked(c, true);

	if(isColumnVisible(c))
		return;

	setColumnWidth(c, 32); // TODO: sane size?
	header()->setResizeEnabled(true, c);
	header()->moveSection(c, c);
	adjustColumn(c);

	// column is shown at the left side, update mLeftColumnIndex
	if(c == leftMostColumnIndex())
	{
		triggerUpdate(); // update item pixmaps etc.
		mLeftColumnIndex = leftMostColumnIndex();
	}
}



// =============================================================================



SimpleListViewItem::SimpleListViewItem(SimpleListView *parent, const DBItem &item)
	: K3ListViewItem(parent)
{
	mId = item.id;
	itemUpdated(item);
	setLength(item.length);
}

/// Sets everything else EXCEPT length
void SimpleListViewItem::itemUpdated(const DBItem &item)
{
	mURL = item.url;
	Noatun::PropertyMap::ConstIterator it;

	const QString artist = item.properties["author"];
	const QString title = item.properties["title"];

	if (artist.isEmpty() && title.isEmpty())
	{
		setText(ArtistCol, QString());
		// For URLs with no filename we use the full URL as fallback
		if (item.url.fileName().isEmpty())
			setText(TitleCol, item.url.prettyUrl(0, KUrl::NoAdjustements));
		else // otherwise simply display the filename
			setText(TitleCol, item.url.fileName());
	}
	else
	{
		setText(ArtistCol, artist);
		setText(TitleCol, title);
	}

	setText(AlbumCol, item.properties["album"]);

	it = item.properties.find("track");
	if (it != item.properties.end())
		setText(TrackCol, (it.data().toInt() > 0 ? it.data() : QString::null));	//krazy:exclude=nullstrassign for old broken gcc

	setText(GenreCol, item.properties["genre"]);

	it = item.properties.find("date");
	if (it != item.properties.end())
		setText(YearCol, (it.data().toInt() > 0 ? it.data() : QString::null));	//krazy:exclude=nullstrassign for old broken gcc

	it = item.properties.find("bitrate");
	if (it != item.properties.end())
		setText(BitrateCol, (it.data().toInt() > 0 ? it.data() : QString::null));	//krazy:exclude=nullstrassign for old broken gcc

	setText(CommentCol, item.properties["comment"]);
}

int SimpleListViewItem::id() const
{
	return mId;
}

const KUrl &SimpleListViewItem::url() const
{
	return mURL;
}

void SimpleListViewItem::setLength(int len)
{
	if (len < 0)
	{
		setText(LengthCol, "--:--");
	}
	else
	{
		QString lenString;
		len /= 1000; // convert milliseconds -> seconds
		int seconds = len % 60;
		lenString.sprintf("%.2d:%.2d", (len - seconds)/60, seconds);
		setText(LengthCol, lenString);
	}
}

void SimpleListViewItem::paintCell(QPainter *p, const QColorGroup &cg,
	int column, int width, int alignment)
{
	if(static_cast<SimpleListView*>(listView())->currentListItem() == this)
	{
		QFont oldFont = p->font();
		QFont newFont(oldFont);
		newFont.setBold(true);
		p->setFont(newFont);
		K3ListViewItem::paintCell(p, cg, column, width, alignment);
		p->setFont(oldFont);

		QPen pen((isSelected() ? cg.highlightedText() : cg.highlight()), 2, Qt::DotLine);
		p->setPen(pen);
		p->drawLine(0, 1, width, 1);
		p->drawLine(0, height()-1, width, height()-1);
	}
	else
	{
		K3ListViewItem::paintCell(p, cg, column, width, alignment);
	}
}

void SimpleListViewItem::setup() // virtual, public
{
	K3ListViewItem::setup();
	if(static_cast<SimpleListView*>(listView())->currentListItem() == this)
	{
		int fontHeight = listView()->fontMetrics().height();
		// ensure we have enough space for playing icon and border
		setHeight(qMax(int(fontHeight * 1.8), KIconLoader::global()->currentSize(KIcon::Small)+6));
	}
}

int SimpleListViewItem::width(const QFontMetrics &fm, const QListView *lv, int c) const
{
	//if(static_cast<SimpleListView*>(listView())->currentListItem() == this)
	{
		//TODO: optimize me
		QFont boldFont(lv->font());
		boldFont.setBold(true);
		QFontMetrics boldFm(boldFont);
		return qMax(K3ListViewItem::width(fm, lv, c), K3ListViewItem::width(boldFm, lv, c));
	}
	//return K3ListViewItem::width(fm, lv, c);
}

const QPixmap *SimpleListViewItem::pixmap(int col) const
{
	//static QPixmap image(SmallIcon("image"));
	static QPixmap playing(SmallIcon("media-playback-start"));
	/*KMimeType::pixmapForURL(url(), 0, KIcon::Small)*/

	SimpleListView *lv = static_cast<SimpleListView*>(listView());

	if(lv->currentListItem() == this && col == lv->leftColumnIndex())
		return &playing;

	return K3ListViewItem::pixmap(col);
}


#include "simplelistview.moc"
