/* This file is part of Noatun

  Copyright 2000-2006 Charles Samuels <charles@kde.org>
  Copyright 2000-2001 Neil Stevens <neil@qualityassistant.com>
  Copyright 2002 by Ryan Cumming <ryan@kde.org>
  Copyright 2003-2007 by Stefan Gehn <mETz81@web.de>

  Licensed under the "Artistic License"
*/
#ifndef SPL_PLAYLIST_H
#define SPL_PLAYLIST_H

#include <noatun/playlist.h>
#include <noatun/plugin.h>
#include <noatun/plugininterfaces.h>

#include "view.h"

class SafeListViewItem;
class View;
class List;
class Q3ListViewItem;
class KXmlGuiWindow;



class ItemData
{
	SafeListViewItem *mItem;

	Noatun::PlaylistInterface<ItemData*> *mIface;

public:
	ItemData(Noatun::PlaylistInterface<ItemData*> *iface, SafeListViewItem *i);

	ItemData *above();
	ItemData *below();

	SafeListViewItem *item() { return mItem; }

	bool isOn() const { return mItem->isOn(); }

	bool operator==(const SafeListViewItem *o) const { return mItem == o; }
};

class SplitPlaylist : public Noatun::Plugin, public Noatun::PlaylistInterface<ItemData*>, public Noatun::ItemSourceInterface
{
Q_OBJECT
friend class SafeListViewItem;

public:
	SplitPlaylist(const KComponentData &instance, Noatun::Global *parent, const char* name);
	~SplitPlaylist();

public: // ItemSource
	/**
	 * Cycle everthing through forward
	 **/
	virtual Noatun::PlaylistItem forward();
	/**
	 * Cycle through backwards
	 **/
	virtual Noatun::PlaylistItem backward();
	/**
	 * return the one that might/should be playing now
	 **/
	virtual Noatun::PlaylistItem current();

	virtual void setCurrent(const Noatun::PlaylistItem &);


public:
	virtual void init();

	virtual void requestUnload();

	virtual KXmlGuiWindow *mainWindow() const;

	/**
	 * go to the front
	 **/
	virtual void reset();

	virtual void clear();
	virtual void addFile(const KUrl &file, bool purgeList, bool autoplay);
	Noatun::PlaylistItem next(bool play);

	virtual Noatun::PlaylistItem getFirst();
	virtual Noatun::PlaylistItem getAfter(const Noatun::PlaylistItem &item);

	static SplitPlaylist *SPL() { return Self; }

	Noatun::Interface *getInterface(const QString &name)
	{
		if (name=="PlaylistInterface")
			return static_cast<Noatun::PlaylistInterface<ItemData*>*>(this);
		if (name=="ItemSourceInterface")
			return static_cast<Noatun::ItemSourceInterface*>(this);
		return 0;
	}

public: // all the playlistitem functions
	/**
	 * Noatun asks your playlist for properties.  It is your
	 * responsiblity to store the information. But usually a QMap<QString,QString>
	 * is enough.
	 *
	 * If you return the default value, the default should not
	 * be written.
	 *
	 * This returns the property, or @p def if such a property doesn't exist
	 **/
	virtual QString property(ItemData *item, const QString &key, const QString &def=QString());

	/**
	 * This sets the property with the given key and value.
	 *
	 * Important: If you use a QMap, you'll have to remove the current
	 * item before adding a new one
	 **/
	virtual void setProperty(ItemData *item, const QString &key, const QString &value);

	/**
	 * remove the item with given key
	 **/
	virtual void clearProperty(ItemData *item, const QString &key);

	/**
	 * return a list of property keys
	 **/
	virtual QStringList properties(ItemData *item);

	/**
	 * return whether if the given key exists
	 **/
	virtual bool hasProperty(ItemData *item, const QString &key);

	/**
	 * compare yourself with the given PlaylistItemData
	 * This is implemented in the slow fashion of
	 * comparing all the properties.  You may
	 * have a more accurate way of implementing this
	 *
	 * if one==two, this will not be called, normally
	 **/
	virtual bool compare(ItemData *one, ItemData *two);

	/**
	 * remove this item from the playlist, this does not necessarily include
	 * deleting this object right now as something might still reference it.
	 **/
	virtual void remove(ItemData *data);

	/**
	 * Playlists should not download files if this returns true
	 **/
	bool streamable(ItemData *data);

public Q_SLOTS:
	virtual void remove(const Noatun::PlaylistItem&);
	virtual void sort();

public Q_SLOTS:
	void setCurrent(SafeListViewItem *item, bool emitC);
	void listItemSelected(Q3ListViewItem*);
	void randomize();

Q_SIGNALS:
	void play(Noatun::PlaylistItem *);

private:
	SafeListViewItem *currentItem, *randomPrevious;
	//bool mExiting;
	View *mView;
	static SplitPlaylist *Self;
};


#endif
