/*
  This file is part of libkdepim.

  Copyright (c) 2004 Tobias Koenig <tokoe@kde.org>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this library; see the file COPYING.LIB.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
  Boston, MA 02110-1301, USA.
*/

#include "addresseeemailselection.h"
#include "recentaddresses.h"

#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>

#include <QPixmap>

using namespace KPIM;

AddresseeEmailSelection::AddresseeEmailSelection()
  : Selection()
{
}

int AddresseeEmailSelection::fieldCount() const
{
  return 3;
}

QString AddresseeEmailSelection::fieldTitle( int index ) const
{
  switch ( index ) {
    case 0:
      return i18n( "To" );
      break;
    case 1:
      return i18n( "Cc" );
      break;
    case 2:
      return i18n( "Bcc" );
      break;
    default:
      return QString();
  }
}

QStringList AddresseeEmailSelection::to() const
{
  return mToEmailList;
}

QStringList AddresseeEmailSelection::cc() const
{
  return mCcEmailList;
}

QStringList AddresseeEmailSelection::bcc() const
{
  return mBccEmailList;
}

KABC::Addressee::List AddresseeEmailSelection::toAddresses() const
{
  return mToAddresseeList;
}

KABC::Addressee::List AddresseeEmailSelection::ccAddresses() const
{
  return mCcAddresseeList;
}

KABC::Addressee::List AddresseeEmailSelection::bccAddresses() const
{
  return mBccAddresseeList;
}

QStringList AddresseeEmailSelection::toDistributionLists() const
{
  return mToDistributionList;
}

QStringList AddresseeEmailSelection::ccDistributionLists() const
{
  return mCcDistributionList;
}

QStringList AddresseeEmailSelection::bccDistributionLists() const
{
  return mBccDistributionList;
}

void AddresseeEmailSelection::setSelectedTo( const QStringList &emails )
{
  setSelectedItem( 0, emails );
}

void AddresseeEmailSelection::setSelectedCC( const QStringList &emails )
{
  setSelectedItem( 1, emails );
}

void AddresseeEmailSelection::setSelectedBCC( const QStringList &emails )
{
  setSelectedItem( 2, emails );
}

int AddresseeEmailSelection::itemCount( const KABC::Addressee &addressee ) const
{
  return addressee.emails().count();
}

QString AddresseeEmailSelection::itemText( const KABC::Addressee &addressee, int index ) const
{
  return addressee.formattedName() + ' ' + email( addressee, index );
}

QPixmap AddresseeEmailSelection::itemIcon( const KABC::Addressee &addressee, int ) const
{
  if ( !addressee.photo().data().isNull() ) {
    return QPixmap::fromImage(
      addressee.photo().data().scaled( 16, 16, Qt::IgnoreAspectRatio,
                                       Qt::SmoothTransformation ) );
  } else {
    return KIconLoader::global()->loadIcon( "personal", KIconLoader::Small );
  }
}

bool AddresseeEmailSelection::itemEnabled( const KABC::Addressee &addressee, int ) const
{
  return addressee.emails().count() != 0;
}

bool AddresseeEmailSelection::itemMatches( const KABC::Addressee &addressee,
                                           int index, const QString &pattern ) const
{
  return addressee.formattedName().startsWith( pattern, Qt::CaseInsensitive ) ||
         email( addressee, index ).startsWith( pattern, Qt::CaseInsensitive );
}

bool AddresseeEmailSelection::itemEquals( const KABC::Addressee &addressee,
                                          int index, const QString &pattern ) const
{
  return
    pattern == ( addressee.formattedName() + ' ' + email( addressee, index ) ) ||
    addressee.emails().contains( pattern );
}

QString AddresseeEmailSelection::distributionListText( const KABC::DistributionList *list ) const
{
  return list->name();
}

QPixmap AddresseeEmailSelection::distributionListIcon( const KABC::DistributionList *list ) const
{
  Q_UNUSED( list );
  return KIconLoader::global()->loadIcon( "kdmconfig", KIconLoader::Small );
}

bool AddresseeEmailSelection::distributionListEnabled( const KABC::DistributionList *list ) const
{
  Q_UNUSED( list );
  return true;
}

bool AddresseeEmailSelection::distributionListMatches( const KABC::DistributionList *list,
                                                       const QString &pattern ) const
{
  // check whether the name of the distribution list matches the pattern or one of its entries.
  bool ok = list->name().startsWith( pattern, Qt::CaseInsensitive );

  KABC::DistributionList::Entry::List entries = list->entries();
  KABC::DistributionList::Entry::List::ConstIterator it;
  for ( it = entries.begin(); it != entries.end(); ++it ) {
    ok = ok || (*it).addressee().formattedName().startsWith( pattern, Qt::CaseInsensitive ) ||
               (*it).email().startsWith( pattern, Qt::CaseInsensitive );
  }

  return ok;
}

int AddresseeEmailSelection::addressBookCount() const
{
  // we provide the recent email addresses via the custom addressbooks
  return 1;
}

QString AddresseeEmailSelection::addressBookTitle( int index ) const
{
  if ( index == 0 ) {
    return i18n( "Recent Addresses" );
  } else {
    return QString();
  }
}

KABC::Addressee::List AddresseeEmailSelection::addressBookContent( int index ) const
{
  if ( index == 0 ) {
    KConfig config( "kmailrc" );
    return RecentAddresses::self( &config )->kabcAddresses();
  } else {
    return KABC::Addressee::List();
  }
}

QString AddresseeEmailSelection::email( const KABC::Addressee &addressee, int index ) const
{
  return addressee.emails()[ index ];
}

void AddresseeEmailSelection::setSelectedItem( int fieldIndex, const QStringList &emails )
{
  QStringList::ConstIterator it;
  for ( it = emails.begin(); it != emails.end(); ++it ) {
    KABC::Addressee addr;
    addr.insertEmail( *it, true );

    selector()->setItemSelected( fieldIndex, addr, 0, *it );
  }
}

void AddresseeEmailSelection::addSelectedAddressees( int fieldIndex,
                                                     const KABC::Addressee &addressee,
                                                     int itemIndex )
{
  switch ( fieldIndex ) {
    case 0:
      mToAddresseeList.append( addressee );
      mToEmailList.append( email( addressee, itemIndex ) );
      break;
    case 1:
      mCcAddresseeList.append( addressee );
      mCcEmailList.append( email( addressee, itemIndex ) );
      break;
    case 2:
      mBccAddresseeList.append( addressee );
      mBccEmailList.append( email( addressee, itemIndex ) );
      break;
    default:
      // oops
      break;
  }
}

void AddresseeEmailSelection::addSelectedDistributionList( int fieldIndex,
                                                           const KABC::DistributionList *list )
{
  switch ( fieldIndex ) {
    case 0:
      mToDistributionList.append( list->name() );
      break;
    case 1:
      mCcDistributionList.append( list->name() );
      break;
    case 2:
      mBccDistributionList.append( list->name() );
      break;
    default:
      // oops
      break;
  }
}
