/*
 *   khexedit - Versatile hex editor
 *   Copyright (C) 1999  Espen Sand, espensa@online.no
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */
#include "optiondialog.h"

#include <limits.h>
#include <KLineEdit>
#include <q3vbox.h>
#include <Q3VButtonGroup>
#include <kicon.h>
#include <qpainter.h>

#include <QCheckBox>
#include <QLayout>
#include <QLabel>
#include <qradiobutton.h>
#include <QSpinBox>
#include <qwhatsthis.h>

#include <kcolordialog.h>
#include <k3colordrag.h>
#include <kfontdialog.h> // For KFontChooser
#include <kiconloader.h>
#include <klocale.h>
#include <kglobalsettings.h>

#include <QPushButton>
#include <KComboBox>

#if 0
#include <qobjectlist.h>
static void enableWidget( QWidget *w, bool state )
{
  if( w->children() )
  {
    QObjectList *l = (QObjectList*)w->children(); // silence please
    for( uint i=0; i < l->count(); i++ )
    {
      QObject *o = l->at(i);
      if( o->isWidgetType() )
      {
	enableWidget( (QWidget*)o, state );
      }
    }
  }
  w->setEnabled( state );
}
#endif



COptionDialog::COptionDialog( QWidget *parent )
  :KPageDialog( parent )
{
  setCaption( i18n("Configure") );
  setFaceType( List );
  setButtons( Help|Default|Apply|Ok|Cancel );
  setDefaultButton( Ok );
  setModal( false );

  setHelp( "khexedit/khexedit.html", QString() );

  setupLayoutPage();
  setupCursorPage();
  setupFontPage();
  setupColorPage();
  setupFilePage();
  setupMiscPage();
  enableButton( Apply, false );
  configChanged = false;
}


COptionDialog::~COptionDialog( void )
{
}

void COptionDialog::slotChanged()
{
  enableButton( Apply, true );
  configChanged = true;
}

void COptionDialog::showEvent( QShowEvent *e )
{
  KPageDialog::showEvent(e);
  setCurrentPage( mLayoutPage );
  mLayout.lineSizeSpin->setFocus();
}


void COptionDialog::setupLayoutPage( void )
{
  QString text;

  mLayoutPage = new KPageWidgetItem( new QFrame(), i18n("Layout") );
  mLayoutPage->setHeader( i18n("Data Layout in Editor" ) );
  mLayoutPage->setIcon( KIcon( BarIcon("khexedit", KIconLoader::SizeMedium ) ));
  addPage( mLayoutPage );

  QGridLayout *gbox = new QGridLayout( mLayoutPage->widget() );
  gbox->setSpacing( spacingHint() );
  gbox->setColumnStretch( 1, 10 );

  mLayout.formatCombo  = new KComboBox( mLayoutPage->widget() );
  QStringList modeList;
  modeList.append( i18n("Hexadecimal Mode") );
  modeList.append( i18n("Decimal Mode") );
  modeList.append( i18n("Octal Mode") );
  modeList.append( i18n("Binary Mode") );
  modeList.append( i18n("Text Only Mode") );
  mLayout.formatCombo->addItems( modeList );
  connect( mLayout.formatCombo, SIGNAL(activated(int)),
	   SLOT(slotModeSelectorChanged(int)) );
  connect( mLayout.formatCombo, SIGNAL(activated(int)),
    SLOT(slotChanged()) );
  gbox->addWidget( mLayout.formatCombo, 0, 1 );

  //
  // I am setting the min. width for one widget in the second column
  // This width will be used by every widget in this column.
  //
  mLayout.lineSizeSpin = new QSpinBox( mLayoutPage->widget() );
  mLayout.lineSizeSpin->setMinimumWidth( fontMetrics().width("M") * 10 );
  mLayout.lineSizeSpin->setRange( 1, 10000 );
  connect( mLayout.lineSizeSpin, SIGNAL(valueChanged(int)),
	   SLOT(slotLineSizeChanged(int) ) );
  connect( mLayout.lineSizeSpin, SIGNAL(valueChanged(int)),
    SLOT(slotChanged()) );
  gbox->addWidget( mLayout.lineSizeSpin, 1, 1 );

  mLayout.columnSizeSpin = new QSpinBox( mLayoutPage->widget() );
  mLayout.columnSizeSpin->setRange( 1, 10000 );
  connect( mLayout.columnSizeSpin, SIGNAL(valueChanged(int)),
	   SLOT(slotColumnSizeChanged(int) ) );
  connect( mLayout.columnSizeSpin, SIGNAL(valueChanged(int)),
    SLOT(slotChanged() ) );
  gbox->addWidget( mLayout.columnSizeSpin, 2, 1 );

  text = i18n("Default l&ine size [bytes]:");
  QLabel *label = new QLabel( text, mLayoutPage->widget() );
  label->setBuddy( mLayout.lineSizeSpin );
  gbox->addWidget( label, 1, 0 );

  text = i18n("Colu&mn size [bytes]:");
  label = new QLabel( text, mLayoutPage->widget() );
  label->setBuddy( mLayout.columnSizeSpin );
  gbox->addWidget( label, 2, 0 );

  QFrame *hline = new QFrame( mLayoutPage->widget() );
  hline->setFrameStyle( QFrame::Sunken | QFrame::HLine );
  gbox->addWidget( hline, 3, 0, 1, 2 );

  text = i18n("Line size is &fixed (use scrollbar when required)");
  mLayout.lockLineCheck = new QCheckBox( text, mLayoutPage->widget() );
  gbox->addWidget( mLayout.lockLineCheck, 4, 0, 1, 2, Qt::AlignLeft );
  connect( mLayout.lockLineCheck, SIGNAL(toggled(bool)),
    SLOT(slotChanged()) );

  text = i18n("Loc&k column at end of line (when column size>1)");
  mLayout.lockColumnCheck = new QCheckBox( text, mLayoutPage->widget() );
  gbox->addWidget( mLayout.lockColumnCheck, 5, 0, 1, 2 );
  connect( mLayout.lockColumnCheck, SIGNAL(toggled(bool)),
    SLOT(slotChanged()) );

  hline = new QFrame( mLayoutPage->widget() );
  hline->setFrameStyle( QFrame::Sunken | QFrame::HLine );
  gbox->addWidget( hline, 6, 0, 1, 2 );

  QStringList gridList;
  gridList.append( i18n("None") );
  gridList.append( i18n("Vertical Only") );
  gridList.append( i18n("Horizontal Only") );
  gridList.append( i18n("Both Directions") );

  mLayout.gridCombo = new KComboBox( mLayoutPage->widget() );
  mLayout.gridCombo->addItems( gridList );
  connect( mLayout.gridCombo, SIGNAL(activated(int)),
    SLOT(slotChanged()) );

  text = i18n("&Gridlines between text:");
  label = new QLabel( text, mLayoutPage->widget() );
  label->setBuddy( mLayout.gridCombo );

  gbox->addWidget( label, 7, 0 );
  gbox->addWidget( mLayout.gridCombo, 7, 1 );

  mLayout.leftSepWidthSpin = new QSpinBox( mLayoutPage->widget() );
  mLayout.leftSepWidthSpin->setRange( 0, 20 );
  gbox->addWidget( mLayout.leftSepWidthSpin, 8, 1 );
  connect( mLayout.leftSepWidthSpin, SIGNAL(valueChanged(int)),
    SLOT(slotChanged()) );

  mLayout.rightSepWidthSpin = new QSpinBox( mLayoutPage->widget() );
  mLayout.rightSepWidthSpin->setRange( 0, 20 );
  gbox->addWidget( mLayout.rightSepWidthSpin, 9, 1 );
  connect( mLayout.rightSepWidthSpin, SIGNAL(valueChanged(int)),
    SLOT(slotChanged()) );

  text = i18n("&Left separator width [pixels]:");
  mLayout.leftSepLabel = new QLabel( text, mLayoutPage->widget() );
  mLayout.leftSepLabel->setBuddy( mLayout.leftSepWidthSpin );
  gbox->addWidget( mLayout.leftSepLabel, 8, 0 );

  text = i18n("&Right separator width [pixels]:");
  mLayout.rightSepLabel = new QLabel( text, mLayoutPage->widget() );
  mLayout.rightSepLabel->setBuddy( mLayout.rightSepWidthSpin );
  gbox->addWidget( mLayout.rightSepLabel, 9, 0 );

  mLayout.separatorSpin = new QSpinBox( mLayoutPage->widget() );
  mLayout.separatorSpin->setRange( 0, 20 );
  gbox->addWidget( mLayout.separatorSpin, 10, 1 );
  connect( mLayout.separatorSpin, SIGNAL(valueChanged(int)),
    SLOT(slotChanged()) );

  mLayout.edgeSpin = new QSpinBox( mLayoutPage->widget() );
  mLayout.edgeSpin->setRange( 0, 20 );
  gbox->addWidget( mLayout.edgeSpin, 11, 1 );
  connect( mLayout.edgeSpin, SIGNAL(valueChanged(int)),
    SLOT(slotChanged()) );

  text = i18n("&Separator margin width [pixels]:");
  label = new QLabel( text, mLayoutPage->widget() );
  label->setBuddy( mLayout.separatorSpin );
  gbox->addWidget( label, 10, 0 );

  text = i18n("&Edge margin width [pixels]:");
  label = new QLabel( text, mLayoutPage->widget() );
  label->setBuddy( mLayout.edgeSpin );
  gbox->addWidget( label, 11, 0 );

  text = i18n("Column separation is e&qual to one character");
  mLayout.columnCheck = new QCheckBox( text, mLayoutPage->widget() );
  gbox->addWidget( mLayout.columnCheck, 12, 0, 1, 2, Qt::AlignLeft );
  connect( mLayout.columnCheck, SIGNAL(toggled(bool)),
	   SLOT( slotColumnSepCheck(bool)));
  connect( mLayout.columnCheck, SIGNAL(toggled(bool)),
    SLOT( slotChanged()));

  mLayout.columnSepSpin = new QSpinBox( mLayoutPage->widget() );
  mLayout.columnSepSpin->setRange( 1, 100 );
  connect( mLayout.columnSepSpin, SIGNAL(valueChanged(int)),
    SLOT(slotChanged()) );

  text = i18n("Column separa&tion [pixels]:");
  mLayout.columnSepLabel = new QLabel( text, mLayoutPage->widget() );
  mLayout.columnSepLabel->setBuddy( mLayout.columnSepSpin );

  gbox->addWidget( mLayout.columnSepLabel, 13, 0 );
  gbox->addWidget( mLayout.columnSepSpin, 13, 1 );

  gbox->setRowStretch( 14, 10 );
}


void COptionDialog::setupCursorPage( void )
{
  QString text;

  KPageWidgetItem *mCursorPage = new KPageWidgetItem( new QFrame(), i18n("Cursor") );
  mCursorPage->setHeader( i18n("Cursor Behavior (only valid for editor)" ) );
  mCursorPage->setIcon(KIcon(  BarIcon("mouse", KIconLoader::SizeMedium ) ));
  addPage( mCursorPage );

  QVBoxLayout *topLayout = new QVBoxLayout( mCursorPage->widget() );
  topLayout->setSpacing( spacingHint() );
  topLayout->setMargin( 0 );

  Q3VButtonGroup *group = new Q3VButtonGroup( i18n("Blinking"), mCursorPage->widget() );
  group->layout()->setMargin( spacingHint() );
  topLayout->addWidget( group );

  text = i18n("Do not b&link");
  mCursor.blinkCheck = new QCheckBox( i18n("Do not b&link"), group );
  connect( mCursor.blinkCheck, SIGNAL(toggled(bool)),
	   SLOT( slotBlinkIntervalCheck(bool)));
  connect( mCursor.blinkCheck, SIGNAL(toggled(bool)),
    SLOT( slotChanged()));

  Q3HBox *hbox = new Q3HBox( group );
  mCursor.blinkLabel = new QLabel( i18n("&Blink interval [ms]:" ), hbox );
  mCursor.blinkSpin = new QSpinBox( hbox );
  mCursor.blinkSpin->setMinimumWidth( fontMetrics().width("M") * 10 );
  mCursor.blinkSpin->setRange( 100, 1000 );
  mCursor.blinkSpin->setSingleStep( 100 );
  mCursor.blinkSpin->setValue( 500 );
  mCursor.blinkLabel->setBuddy(mCursor.blinkSpin);
  connect( mCursor.blinkSpin, SIGNAL(valueChanged(int)),
    SLOT( slotChanged()));

  group = new Q3VButtonGroup( i18n("Shape"), mCursorPage->widget() );
  group->layout()->setMargin( spacingHint() );
  topLayout->addWidget( group );

  text = i18n("Always &use block (rectangular) cursor");
  mCursor.blockCheck = new QCheckBox( text, group );
  connect( mCursor.blockCheck, SIGNAL(toggled(bool)),
	   SLOT( slotBlockCursorCheck(bool)));
  connect( mCursor.blockCheck, SIGNAL(toggled(bool)),
    SLOT( slotChanged()));
  text = i18n("Use &thick cursor in insert mode");
  mCursor.thickCheck = new QCheckBox( text, group );
  connect( mCursor.thickCheck, SIGNAL(toggled(bool)),
    SLOT( slotChanged()));

  text = i18n("Cursor Behavior When Editor Loses Focus");
  group = new Q3VButtonGroup( text, mCursorPage->widget() );
  group->layout()->setMargin( spacingHint() );
  topLayout->addWidget( group );

  text = i18n("&Stop blinking (if blinking is enabled)");
  mCursor.stopRadio = new QRadioButton( text, group );
  mCursor.hideRadio = new QRadioButton( i18n("H&ide"), group );
  text = i18n("Do &nothing");
  mCursor.nothingRadio = new QRadioButton( text, group );
  connect( mCursor.stopRadio, SIGNAL(toggled(bool)),
    SLOT( slotChanged()));
  connect( mCursor.hideRadio, SIGNAL(toggled(bool)),
    SLOT( slotChanged()));
  connect( mCursor.nothingRadio, SIGNAL(toggled(bool)),
    SLOT( slotChanged()));

  topLayout->addStretch(10);
}


void COptionDialog::setupColorPage( void )
{
  QString text;

  KPageWidgetItem *mColorPage = new KPageWidgetItem( new QFrame(), i18n("Colors") );
  mColorPage->setHeader( i18n("Editor Colors (system selection color is always used)" ) );
  mColorPage->setIcon(KIcon(  BarIcon("colorize", KIconLoader::SizeMedium ) ));
  addPage( mColorPage );

  QVBoxLayout *topLayout = new QVBoxLayout( mColorPage->widget() );
  topLayout->setSpacing( spacingHint() );
  topLayout->setMargin( 0 );

  text = i18n("&Use system colors (as chosen in Control Center)");
  mColor.checkSystem = new QCheckBox( text, mColorPage->widget() );
  connect( mColor.checkSystem, SIGNAL(toggled(bool)),
	   SLOT( slotColorSystem(bool)));
  connect( mColor.checkSystem, SIGNAL(toggled(bool)),
    SLOT( slotChanged()));
  topLayout->addWidget( mColor.checkSystem );

  QFrame *hline = new QFrame( mColorPage->widget() );
  hline->setFrameStyle( QFrame::Sunken | QFrame::HLine );
  topLayout->addWidget( hline );

  QStringList modeList;
  modeList.append( i18n("First, Third ... Line Background") );
  modeList.append( i18n("Second, Fourth ... Line Background") );
  modeList.append( i18n("Offset Background") );
  modeList.append( i18n("Inactive Background") );
  modeList.append( i18n("Even Column Text") );
  modeList.append( i18n("Odd Column Text") );
  modeList.append( i18n("Non Printable Text") );
  modeList.append( i18n("Offset Text") );
  modeList.append( i18n("Secondary Text") );
  modeList.append( i18n("Marked Background") );
  modeList.append( i18n("Marked Text") );
  modeList.append( i18n("Cursor Background") );
  modeList.append( i18n("Cursor Text (block shape)") );
  modeList.append( i18n("Bookmark Background") );
  modeList.append( i18n("Bookmark Text") );
  modeList.append( i18n("Separator") );
  modeList.append( i18n("Grid Lines") );

  mColor.colorList = new CColorListBox( mColorPage->widget() );
  topLayout->addWidget( mColor.colorList, 10 );
  for( uint i=0; i<17; i++ )
  {
    CColorListItem *listItem = new CColorListItem( modeList[i] );
    mColor.colorList->insertItem( listItem );
  }
  mColor.colorList->setCurrentItem(0);
  connect( mColor.colorList, SIGNAL( dataChanged() ), this, SLOT( slotChanged() ) );
}


void COptionDialog::setupFontPage( void )
{
  QString text;

  KPageWidgetItem *mFontPage = new KPageWidgetItem( new QFrame(), i18n("Font") );
  mFontPage->setHeader( i18n("Font Selection (editor can only use a fixed font)") );
  mFontPage->setIcon(KIcon(  BarIcon("fonts", KIconLoader::SizeMedium ) ) );
  addPage( mFontPage );

  QVBoxLayout *topLayout = new QVBoxLayout( mFontPage->widget() );
  topLayout->setSpacing( spacingHint() );
  topLayout->setMargin( 0 );

  text = i18n("&Use system font (as chosen in Control Center)");
  mFont.checkSystem = new QCheckBox( text, mFontPage->widget() );
  connect( mFont.checkSystem, SIGNAL(toggled(bool)),
	   this, SLOT( slotFontSystem(bool)));
  connect( mFont.checkSystem, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));
  topLayout->addWidget( mFont.checkSystem );

  QFrame *hline = new QFrame( mFontPage->widget() );
  hline->setFrameStyle( QFrame::Sunken | QFrame::HLine );
  topLayout->addWidget( hline );

  mFont.chooser = new KFontChooser( mFontPage->widget(), KFontChooser::FixedFontsOnly, QStringList(), 4 );
  mFont.chooser->setObjectName( "font" );
  topLayout->addWidget( mFont.chooser );
  QFont fixFont( KGlobalSettings::fixedFont() );
  fixFont.setBold(true);
  mFont.chooser->setFont( fixFont, true );
  mFont.chooser->setSampleText( i18n("KHexEdit editor font") );
  connect( mFont.chooser, SIGNAL(fontSelected(const QFont &)),
    this, SLOT( slotChanged()));

  hline = new QFrame( mFontPage->widget() );
  hline->setFrameStyle( QFrame::Sunken | QFrame::HLine );
  topLayout->addWidget( hline );

  QHBoxLayout *hbox = new QHBoxLayout();
  topLayout->addLayout( hbox );

  mFont.nonPrintInput = new KLineEdit( mFontPage->widget() );
  mFont.nonPrintInput->setMaxLength( 1 );
  connect( mFont.nonPrintInput, SIGNAL(textChanged(const QString &)),
    this, SLOT( slotChanged()));

  text = i18n("&Map non printable characters to:");
  QLabel *nonPrintLabel = new QLabel( text, mFontPage->widget() );
  nonPrintLabel->setBuddy( mFont.nonPrintInput );

  hbox->addWidget( nonPrintLabel, 0, Qt::AlignLeft );
  hbox->addSpacing( spacingHint() );
  hbox->addWidget( mFont.nonPrintInput, 10 );

  topLayout->addStretch(10);
}


void COptionDialog::setupFilePage( void )
{
  QString text;

  KPageWidgetItem *mFilePage = new KPageWidgetItem( new QFrame(), i18n("Files") );
  mFilePage->setHeader( i18n("File Management") );
  mFilePage->setIcon(KIcon(  BarIcon("kmultiple", KIconLoader::SizeMedium ) ) );
  addPage( mFilePage );

  QVBoxLayout *topLayout = new QVBoxLayout( mFilePage->widget() );
  topLayout->setSpacing( spacingHint() );
  topLayout->setMargin( 0 );

  QGridLayout *gbox = new QGridLayout();
  gbox->setSpacing( spacingHint() );
  topLayout->addLayout( gbox );

  mFile.openCombo = new KComboBox( mFilePage->widget() );
  QStringList modeList;
  modeList.append( i18n("None") );
  modeList.append( i18n("Most Recent Document") );
  modeList.append( i18n("All Recent Documents") );
  mFile.openCombo->addItems( modeList );
  mFile.openCombo->setMinimumWidth( mFile.openCombo->sizeHint().width() );
  connect( mFile.openCombo, SIGNAL(activated(int)),
    this, SLOT( slotChanged()));

  text = i18n("Open doc&uments on startup:");
  QLabel *label = new QLabel( text, mFilePage->widget() );
  label->setBuddy(  mFile.openCombo );

  gbox->addWidget( label, 0, 0 );
  gbox->addWidget( mFile.openCombo, 0, 1 );

  text = i18n("&Jump to previous cursor position on startup");
  mFile.gotoOffsetCheck = new QCheckBox( text, mFilePage->widget() );
  // ### TODO: this is currently not available.
  mFile.gotoOffsetCheck->setChecked( false );
  mFile.gotoOffsetCheck->setEnabled( false );
  // ### END
  topLayout->addWidget( mFile.gotoOffsetCheck, 0, Qt::AlignLeft );
  connect( mFile.gotoOffsetCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));

  QFrame *hline = new QFrame( mFilePage->widget() );
  hline->setFrameStyle( QFrame::Sunken | QFrame::HLine );
  topLayout->addWidget( hline );

  text = i18n("Open document with &write protection enabled");
  mFile.writeProtectCheck = new QCheckBox( text, mFilePage->widget() );
  topLayout->addWidget( mFile.writeProtectCheck, 0, Qt::AlignLeft );
  connect( mFile.writeProtectCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));

  text = i18n("&Keep cursor position after reloading document");
  mFile.reloadOffsetCheck = new QCheckBox( text, mFilePage->widget() );
  topLayout->addWidget( mFile.reloadOffsetCheck, 0, Qt::AlignLeft );
  connect( mFile.reloadOffsetCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));

  text = i18n("&Make a backup when saving document");
  mFile.backupCheck = new QCheckBox( text, mFilePage->widget() );
  topLayout->addWidget( mFile.backupCheck, 0, Qt::AlignLeft );
  connect( mFile.backupCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));

  hline = new QFrame( mFilePage->widget() );
  hline->setFrameStyle( QFrame::Sunken | QFrame::HLine );
  topLayout->addWidget( hline );

  text = i18n("Do not &save \"Recent\" document list on exit");
  mFile.discardRecentCheck = new QCheckBox( text, mFilePage->widget() );
  topLayout->addWidget( mFile.discardRecentCheck, 0, Qt::AlignLeft );
  mFile.discardRecentCheck->setWhatsThis(
                   i18n( "Clicking this check box makes KHexEdit forget his recent document list "
                          "when the program is closed.\n"
                          "Note: it will not erase any document of the recent document list "
                          "created by KDE." ) );
  connect( mFile.discardRecentCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));

  text = i18n("Cl&ear \"Recent\" Document List");
  QPushButton *discardRecentButton = new QPushButton( mFilePage->widget() );
  discardRecentButton->setText( text );
  discardRecentButton->setWhatsThis(
                   i18n( "Clicking this button makes KHexEdit forget his recent document list.\n"
                          "Note: it will not erase any document of the recent document list "
                          "created by KDE." ) );
  topLayout->addWidget( discardRecentButton, 0, Qt::AlignCenter );
  connect( discardRecentButton, SIGNAL(clicked()),
	   SIGNAL(removeRecentFiles()) );

  topLayout->addStretch(10);
}



void COptionDialog::setupMiscPage( void )
{
  QString text;
  QLabel *label;

  KPageWidgetItem *mMiscPage = new KPageWidgetItem( new QFrame(), i18n("Miscellaneous") );
  mMiscPage->setHeader( i18n("Various Properties") );
  mMiscPage->setIcon(KIcon(  BarIcon("gear", KIconLoader::SizeMedium ) ) );
  addPage( mMiscPage );

  QVBoxLayout *topLayout = new QVBoxLayout( mMiscPage->widget() );
  topLayout->setSpacing( spacingHint() );
  topLayout->setMargin( 0 );


  text = i18n("Auto&matic copy to clipboard when selection is ready");
  mMisc.autoCheck = new QCheckBox( text, mMiscPage->widget() );
  topLayout->addWidget( mMisc.autoCheck, 0, Qt::AlignLeft );
  connect( mMisc.autoCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));

  text = i18n("&Editor starts in \"insert\" mode" );
  mMisc.insertCheck = new QCheckBox( text, mMiscPage->widget() );
  topLayout->addWidget( mMisc.insertCheck, 0, Qt::AlignLeft );
  connect( mMisc.insertCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));

  text = i18n("Confirm &wrapping (to beginning or end) during search");
  mMisc.confirmWrapCheck = new QCheckBox( text, mMiscPage->widget() );
  topLayout->addWidget( mMisc.confirmWrapCheck, 0, Qt::AlignLeft );
  connect( mMisc.confirmWrapCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));

  text = i18n("Cursor jumps to &nearest byte when moved");
  mMisc.cursorJumpCheck = new QCheckBox( text, mMiscPage->widget() );
  topLayout->addWidget( mMisc.cursorJumpCheck, 0, Qt::AlignLeft );
  connect( mMisc.cursorJumpCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));

  Q3VButtonGroup *group = new Q3VButtonGroup( i18n("Sounds"), mMiscPage->widget() );
  group->layout()->setMargin( spacingHint() );
  topLayout->addWidget( group );
  text = i18n("Make sound on data &input (eg. typing) failure");
  mMisc.inputCheck = new QCheckBox( text, group );
  connect( mMisc.inputCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));
  text = i18n("Make sound on &fatal failure");
  mMisc.fatalCheck = new QCheckBox( text, group );
  connect( mMisc.fatalCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));

  group = new Q3VButtonGroup( i18n("Bookmark Visibility"), mMiscPage->widget() );
  group->layout()->setMargin( spacingHint() );
  topLayout->addWidget( group );
  text = i18n("Use visible bookmarks in the offset column");
  mMisc.bookmarkColumnCheck = new QCheckBox( text, group );
  connect( mMisc.bookmarkColumnCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));
  text = i18n("Use visible bookmarks in the editor fields");
  mMisc.bookmarkEditorCheck = new QCheckBox( text, group );
  connect( mMisc.bookmarkEditorCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));

  text = i18n("Confirm when number of printed pages will e&xceed limit");
  mMisc.thresholdCheck = new QCheckBox( text, mMiscPage->widget() );
  connect( mMisc.thresholdCheck, SIGNAL(clicked()),
	   SLOT( slotThresholdConfirm()));
  connect( mMisc.thresholdCheck, SIGNAL(toggled(bool)),
    this, SLOT( slotChanged()));
  topLayout->addWidget( mMisc.thresholdCheck, 0, Qt::AlignLeft );

  QGridLayout *glay = new QGridLayout( );
  glay->setColumnStretch(2,10);
  topLayout->addLayout( glay );

  mMisc.thresholdSpin = new QSpinBox( mMiscPage->widget() );
  mMisc.thresholdSpin->setMinimumWidth( fontMetrics().width("M") * 10 );
  mMisc.thresholdSpin->setRange( 5, INT_MAX );
  mMisc.thresholdSpin->setSingleStep( 5 );
  mMisc.thresholdSpin->setValue( 5 );
  connect( mMisc.thresholdSpin, SIGNAL(valueChanged(int)),
    this, SLOT( slotChanged()));

  text = i18n("&Threshold [pages]:" );
  mMisc.thresholdLabel = new QLabel( text, mMiscPage->widget() );
  mMisc.thresholdLabel->setBuddy( mMisc.thresholdSpin );

  glay->addWidget( mMisc.thresholdLabel, 0, 0 );
  glay->addWidget( mMisc.thresholdSpin, 0, 1 );

  QFrame *hline = new QFrame( mMiscPage->widget() );
  hline->setFrameStyle( QFrame::Sunken | QFrame::HLine );
  glay->addWidget( hline, 1, 0, 1, 3 );

  mMisc.undoSpin = new QSpinBox( mMiscPage->widget() );
  mMisc.undoSpin->setRange( 10, 10000 );
  mMisc.undoSpin->setSingleStep( 5 );
  mMisc.undoSpin->setValue( 50 );
  connect( mMisc.undoSpin, SIGNAL(valueChanged(int)),
    this, SLOT( slotChanged()));

  label = new QLabel( i18n("&Undo limit:"), mMiscPage->widget() );
  label->setBuddy( mMisc.undoSpin );
  glay->addWidget( label, 2, 0 );
  glay->addWidget( mMisc.undoSpin, 2, 1 );

  topLayout->addStretch(10);
}



void COptionDialog::slotModeSelectorChanged( int index )
{
  int lineSize = mDisplayState.line.getLineSize( index );
  int colSize  = mDisplayState.line.getColumnSize( index );
  mLayout.lineSizeSpin->setValue( lineSize );
  mLayout.columnSizeSpin->setValue( colSize );
}

void COptionDialog::slotLineSizeChanged( int value )
{
  int index = mLayout.formatCombo->currentIndex();
  Q_ASSERT(index >= 0);
  mDisplayState.line.setLineSize( index, value );
  if( index == SDisplayLine::textOnly )
  {
    mLayout.columnSizeSpin->setValue( value );
  }
}

void COptionDialog::slotColumnSizeChanged( int value )
{
  int index = mLayout.formatCombo->currentIndex();
  Q_ASSERT(index >= 0);
  mDisplayState.line.setColumnSize( index, value );
  if( index == SDisplayLine::textOnly )
  {
    mLayout.lineSizeSpin->setValue( value );
  }
}

void COptionDialog::slotColumnSepCheck( bool state )
{
  state = state == true ? false: true;
  mLayout.columnSepLabel->setEnabled( state );
  mLayout.columnSepSpin->setEnabled( state );
}


void COptionDialog::slotBlinkIntervalCheck( bool state )
{
  state = state == true ? false : true;
  mCursor.blinkSpin->setEnabled( state );
  mCursor.blinkLabel->setEnabled( state );
}

void COptionDialog::slotBlockCursorCheck( bool state )
{
  state = state == true ? false : true;
  mCursor.thickCheck->setEnabled( state );
}

void COptionDialog::slotFontSystem( bool state )
{
  mFont.chooser->setEnabled( state == true ? false : true );
}

void COptionDialog::slotColorSystem( bool state )
{
  state = state == true ? false : true;
  mColor.colorList->setEnabled( state );
}



void COptionDialog::slotThresholdConfirm( void )
{
  bool state = mMisc.thresholdCheck->isChecked();
  mMisc.thresholdLabel->setEnabled( state );
  mMisc.thresholdSpin->setEnabled( state );
}


void COptionDialog::slotDefault( void )
{
  //
  // We use the constructor settings as default values
  //
  if ( currentPage() == mLayoutPage )
  {
    SDisplayLayout layout;
    SDisplayLine line;
    setLayout( layout, line );
  }
  else if ( currentPage() == mCursorPage )
  {
    SDisplayCursor cursor;
    setCursor( cursor );
  }
  else if ( currentPage() == mColorPage )
  {
    SDisplayColor color;
    setColor( color );
  }
  else if ( currentPage() == mFontPage )
  {
    SDisplayFont font;
    setFont( font );
  }
  else if ( currentPage() == mFilePage )
  {
    SDisplayMisc misc;
    setFile( misc );
  }
  else if ( currentPage() == mMiscPage )
  {
    SDisplayMisc misc;
    setMisc( misc );
  }
}


void COptionDialog::slotOk( void )
{
  if( configChanged )
    slotApply();
  accept();
}

void COptionDialog::slotApply( void )
{
  if ( currentPage() == mLayoutPage )
  {
    int index = mLayout.formatCombo->currentIndex();
    Q_ASSERT(index >= 0);
    mDisplayState.line.setLineSize(index, mLayout.lineSizeSpin->value());
    mDisplayState.line.setColumnSize(index, mLayout.columnSizeSpin->value());
    emit lineSizeChoice( mDisplayState.line );

    SDisplayLayout &l = mDisplayState.layout;
    l.lockLine = mLayout.lockLineCheck->isChecked();
    l.lockColumn = mLayout.lockColumnCheck->isChecked();
    l.leftSeparatorWidth = mLayout.leftSepWidthSpin->value();
    l.rightSeparatorWidth = mLayout.rightSepWidthSpin->value();
    l.separatorMarginWidth = mLayout.separatorSpin->value();
    l.edgeMarginWidth = mLayout.edgeSpin->value();
    l.leftSeparatorWidth = mLayout.leftSepWidthSpin->value();
    l.rightSeparatorWidth = mLayout.rightSepWidthSpin->value();
    l.columnCharSpace = mLayout.columnCheck->isChecked();
    l.columnSpacing = mLayout.columnSepSpin->value();

    if( mLayout.gridCombo->currentIndex() == 0 )
    {
      l.horzGridWidth = l.vertGridWidth = 0;
    }
    else if( mLayout.gridCombo->currentIndex() == 1 )
    {
      l.horzGridWidth = 0;
      l.vertGridWidth = 1;
    }
    else if( mLayout.gridCombo->currentIndex() == 2 )
    {
      l.horzGridWidth = 1;
      l.vertGridWidth = 0;
    }
    else
    {
      l.horzGridWidth = l.vertGridWidth = 1;
    }

    emit layoutChoice( mDisplayState.layout );
  }
  else if ( currentPage() == mCursorPage )
  {
    SDisplayCursor &c = mDisplayState.cursor;
    c.alwaysVisible    = mCursor.blinkCheck->isChecked();
    c.interval         = mCursor.blinkSpin->value();
    c.alwaysBlockShape = mCursor.blockCheck->isChecked();
    c.thickInsertShape = mCursor.thickCheck->isChecked();
    c.focusMode        = cursorFocusMode();
    emit cursorChoice( mDisplayState.cursor );
  }
  else if ( currentPage() == mColorPage )
  {
    SDisplayColor &c = mDisplayState.color;
    c.useSystemColor = mColor.checkSystem->isChecked();
    c.textBg = mColor.colorList->color( SColorWidgets::FirstTextBg );
    c.secondTextBg = mColor.colorList->color( SColorWidgets::SecondTextBg );
    c.offsetBg = mColor.colorList->color( SColorWidgets::OffsetBg );
    c.inactiveBg = mColor.colorList->color( SColorWidgets::InactiveBg );
    c.markBg = mColor.colorList->color( SColorWidgets::MarkedBg );
    c.markFg = mColor.colorList->color( SColorWidgets::MarkedFg );
    c.primaryFg[0] = mColor.colorList->color( SColorWidgets::EvenColumnFg );
    c.primaryFg[1] = mColor.colorList->color( SColorWidgets::OddColumnFg );
    c.offsetFg  = mColor.colorList->color( SColorWidgets::OffsetFg );
    c.secondaryFg = mColor.colorList->color( SColorWidgets::SecondaryFg );
    c.cursorBg  = mColor.colorList->color( SColorWidgets::CursorBg );
    c.cursorFg  = mColor.colorList->color( SColorWidgets::CursorFg );
    c.leftSeparatorFg = mColor.colorList->color(SColorWidgets::SeparatorFg);
    c.rightSeparatorFg = mColor.colorList->color(SColorWidgets::SeparatorFg);
    c.bookmarkBg = mColor.colorList->color( SColorWidgets::BookmarkBg );
    c.bookmarkFg = mColor.colorList->color( SColorWidgets::BookmarkFg );
    c.nonPrintFg  = mColor.colorList->color( SColorWidgets::NonPrintFg );
    c.gridFg = mColor.colorList->color( SColorWidgets::GridFg );
    emit colorChoice( mDisplayState.color );
  }
  else if ( currentPage() == mFontPage )
  {
    SDisplayFont &f = mDisplayState.font;
    f.useSystemFont = mFont.checkSystem->isChecked();
    f.localFont = mFont.chooser->font();

    QString nonPrintText = mFont.nonPrintInput->text();
    if( nonPrintText.isEmpty() )
    {
      f.nonPrintChar = ' ';
    }
    else
    {
      f.nonPrintChar = nonPrintText[0];
    }

    emit fontChoice( mDisplayState.font );
  }
  else if ( currentPage() == mFilePage || currentPage() == mMiscPage )
  {
    SDisplayMisc &m = mDisplayState.misc;
    m.undoLevel = mMisc.undoSpin->value();
    m.autoCopyToClipboard = mMisc.autoCheck->isChecked();
    m.insertMode = mMisc.insertCheck->isChecked();
    m.inputSound = mMisc.inputCheck->isChecked();
    m.confirmWrap = mMisc.confirmWrapCheck->isChecked();
    m.cursorJump = mMisc.cursorJumpCheck->isChecked();
    m.fatalSound = mMisc.fatalCheck->isChecked();
    m.confirmThreshold = mMisc.thresholdCheck->isChecked();
    m.thresholdValue = mMisc.thresholdSpin->value();
    m.bookmarkOffsetColumn = mMisc.bookmarkColumnCheck->isChecked();
    m.bookmarkEditor = mMisc.bookmarkEditorCheck->isChecked();

    m.openFile  = (SDisplayMisc::EOpenFile)mFile.openCombo->currentIndex();
    m.gotoOnStartup = mFile.gotoOffsetCheck->isChecked();
    m.writeProtect = mFile.writeProtectCheck->isChecked();
    m.gotoOnReload = mFile.reloadOffsetCheck->isChecked();
    m.makeBackup = mFile.backupCheck->isChecked();
    m.discardRecent = mFile.discardRecentCheck->isChecked();

    emit miscChoice( mDisplayState.misc );
  }
  enableButton( Apply, false );
  configChanged = false;
}

void COptionDialog::setLayout( SDisplayLayout &layout, SDisplayLine &line )
{
  mDisplayState.line = line;
  mDisplayState.layout = layout;

  slotModeSelectorChanged( mLayout.formatCombo->currentIndex() );
  mLayout.lockLineCheck->setChecked( layout.lockLine );
  mLayout.lockColumnCheck->setChecked( layout.lockColumn );
  mLayout.leftSepWidthSpin->setValue( layout.leftSeparatorWidth );
  mLayout.rightSepWidthSpin->setValue( layout.rightSeparatorWidth );
  mLayout.separatorSpin->setValue( layout.separatorMarginWidth );
  mLayout.edgeSpin->setValue( layout.edgeMarginWidth );
  mLayout.leftSepWidthSpin->setValue( layout.leftSeparatorWidth );
  mLayout.rightSepWidthSpin->setValue( layout.rightSeparatorWidth );
  mLayout.columnCheck->setChecked( layout.columnCharSpace );
  slotColumnSepCheck( layout.columnCharSpace );
  mLayout.columnSepSpin->setValue( layout.columnSpacing );

  if( layout.horzGridWidth == 0 && layout.vertGridWidth == 0 )
  {
    mLayout.gridCombo->setCurrentIndex(0);
  }
  else if( layout.horzGridWidth != 0 && layout.vertGridWidth != 0 )
  {
    mLayout.gridCombo->setCurrentIndex(3);
  }
  else
  {
    mLayout.gridCombo->setCurrentIndex( layout.vertGridWidth != 0 ? 1 : 2 );
  }

}


void COptionDialog::setCursor( SDisplayCursor &cursor )
{
  mDisplayState.cursor = cursor;

  mCursor.blinkCheck->setChecked( cursor.alwaysVisible );
  mCursor.blinkSpin->setValue( cursor.interval );
  mCursor.blockCheck->setChecked( cursor.alwaysBlockShape );
  mCursor.thickCheck->setChecked( cursor.thickInsertShape );
  bool state = cursor.focusMode == SDisplayCursor::stopBlinking ? true : false;
  mCursor.stopRadio->setChecked( state );
  state = cursor.focusMode == SDisplayCursor::hide ? true : false;
  mCursor.hideRadio->setChecked( state );
  state = cursor.focusMode == SDisplayCursor::ignore ? true : false;
  mCursor.nothingRadio->setChecked( state );
}


void COptionDialog::setColor( SDisplayColor &color )
{
  mDisplayState.color = color;

  mColor.checkSystem->setChecked( color.useSystemColor );
  mColor.colorList->setColor( SColorWidgets::FirstTextBg, color.textBg );
  mColor.colorList->setColor( SColorWidgets::SecondTextBg, color.secondTextBg);
  mColor.colorList->setColor( SColorWidgets::OffsetBg, color.offsetBg );
  mColor.colorList->setColor( SColorWidgets::InactiveBg, color.inactiveBg );
  mColor.colorList->setColor( SColorWidgets::MarkedBg, color.markBg );
  mColor.colorList->setColor( SColorWidgets::MarkedFg, color.markFg );
  mColor.colorList->setColor( SColorWidgets::EvenColumnFg, color.primaryFg[0]);
  mColor.colorList->setColor( SColorWidgets::OddColumnFg, color.primaryFg[1] );
  mColor.colorList->setColor( SColorWidgets::OffsetFg, color.offsetFg );
  mColor.colorList->setColor( SColorWidgets::SecondaryFg, color.secondaryFg );
  mColor.colorList->setColor( SColorWidgets::CursorBg, color.cursorBg );
  mColor.colorList->setColor( SColorWidgets::CursorFg, color.cursorFg );
  mColor.colorList->setColor(SColorWidgets::SeparatorFg,color.leftSeparatorFg);
  mColor.colorList->setColor( SColorWidgets::BookmarkBg, color.bookmarkBg );
  mColor.colorList->setColor( SColorWidgets::BookmarkFg, color.bookmarkFg );
  mColor.colorList->setColor( SColorWidgets::NonPrintFg, color.nonPrintFg );
  mColor.colorList->setColor( SColorWidgets::GridFg, color.gridFg );
}


void COptionDialog::setFont( SDisplayFont &font )
{
  mDisplayState.font = font;

  mFont.checkSystem->setChecked( font.useSystemFont );
  mFont.chooser->setFont( font.localFont, true );
  QString buf = font.nonPrintChar;
  mFont.nonPrintInput->setText( buf );
}


void COptionDialog::setMisc( SDisplayMisc &misc )
{
  mDisplayState.misc.undoLevel            = misc.undoLevel;
  mDisplayState.misc.inputSound           = misc.inputSound;
  mDisplayState.misc.fatalSound           = misc.fatalSound;
  mDisplayState.misc.autoCopyToClipboard  = misc.autoCopyToClipboard;
  mDisplayState.misc.insertMode           = misc.insertMode;
  mDisplayState.misc.confirmWrap          = misc.confirmWrap;
  mDisplayState.misc.cursorJump           = misc.cursorJump;
  mDisplayState.misc.confirmThreshold     = misc.confirmThreshold;
  mDisplayState.misc.thresholdValue       = misc.thresholdValue;
  mDisplayState.misc.bookmarkOffsetColumn = misc.bookmarkOffsetColumn;
  mDisplayState.misc.bookmarkEditor       = misc.bookmarkEditor;

  mMisc.autoCheck->setChecked( misc.autoCopyToClipboard );
  mMisc.insertCheck->setChecked( misc.insertMode );
  mMisc.confirmWrapCheck->setChecked( misc.confirmWrap );
  mMisc.cursorJumpCheck->setChecked( misc.cursorJump );

  mMisc.inputCheck->setChecked( misc.inputSound );
  mMisc.fatalCheck->setChecked( misc.fatalSound );
  mMisc.bookmarkColumnCheck->setChecked( misc.bookmarkOffsetColumn );
  mMisc.bookmarkEditorCheck->setChecked( misc.bookmarkEditor );

  mMisc.thresholdCheck->setChecked( misc.confirmThreshold );
  mMisc.thresholdSpin->setValue( misc.thresholdValue );
  slotThresholdConfirm();

  mMisc.undoSpin->setValue( misc.undoLevel );
}

void COptionDialog::setFile( SDisplayMisc &misc )
{
  mDisplayState.misc.openFile      = misc.openFile;
  mDisplayState.misc.gotoOnStartup = misc.gotoOnStartup;
  mDisplayState.misc.writeProtect  = misc.writeProtect;
  mDisplayState.misc.gotoOnReload  = misc.gotoOnReload;
  mDisplayState.misc.discardRecent = misc.discardRecent;
  mDisplayState.misc.makeBackup    = misc.makeBackup;

  mFile.openCombo->setCurrentIndex( (uint)misc.openFile );
  mFile.gotoOffsetCheck->setChecked( misc.gotoOnStartup );
  mFile.writeProtectCheck->setChecked( misc.writeProtect );
  mFile.reloadOffsetCheck->setChecked( misc.gotoOnReload );
  mFile.discardRecentCheck->setChecked( misc.discardRecent );
  mFile.backupCheck->setChecked( misc.makeBackup );
}


void COptionDialog::setState( SDisplayState &state )
{
  setLayout( state.layout, state.line );
  setCursor( state.cursor );
  setColor( state.color );
  setFont( state.font );
  setFile( state.misc );
  setMisc( state.misc );
  enableButton( Apply, false );
  configChanged = false;
}


SDisplayCursor::EFocusMode COptionDialog::cursorFocusMode( void )
{
  if( mCursor.stopRadio->isChecked() == true )
  {
    return( SDisplayCursor::stopBlinking );
  }
  else if( mCursor.hideRadio->isChecked() == true )
  {
    return( SDisplayCursor::hide );
  }
  else
  {
    return( SDisplayCursor::ignore );
  }
}



CColorListBox::CColorListBox( QWidget *parent, Qt::WFlags f )
  :K3ListBox( parent, 0, f ), mCurrentOnDragEnter(-1)
{
  connect( this, SIGNAL(selected(int)), this, SLOT(newColor(int)) );
  setAcceptDrops( true);
}


void CColorListBox::setEnabled( bool state )
{
  if( state == isEnabled() )
  {
    return;
  }

  Q3ListBox::setEnabled( state );
  for( uint i=0; i<count(); i++ )
  {
    updateItem( i );
  }
}


void CColorListBox::setColor( uint index, const QColor &color )
{
  if( index < count() )
  {
    CColorListItem *colorItem = (CColorListItem*)item(index);
    colorItem->setColor(color);
    updateItem( colorItem );
  }
}


const QColor CColorListBox::color( uint index )
{
  if( index < count() )
  {
    CColorListItem *colorItem = (CColorListItem*)item(index);
    return( colorItem->color() );
  }
  else
  {
    return( Qt::black );
  }
}


void CColorListBox::newColor( int index )
{
  if( isEnabled() == false )
  {
    return;
  }

  if( (uint)index < count() )
  {
    QColor c = color( index );
    if( KColorDialog::getColor( c, this ) != QDialog::Rejected )
    {
      setColor( index, c );
      emit dataChanged();
    }
  }
}


void CColorListBox::dragEnterEvent( QDragEnterEvent *e )
{
  if( e->mimeData()->hasColor() && isEnabled() )
  {
    mCurrentOnDragEnter = currentItem();
    e->accept();
  }
  else
  {
    mCurrentOnDragEnter = -1;
    e->ignore();
  }
}


void CColorListBox::dragLeaveEvent( QDragLeaveEvent * )
{
  if( mCurrentOnDragEnter != -1 )
  {
    setCurrentItem( mCurrentOnDragEnter );
    mCurrentOnDragEnter = -1;
  }
}


void CColorListBox::dragMoveEvent( QDragMoveEvent *e )
{
  if( e->mimeData()->hasColor() && isEnabled() )
  {
    CColorListItem *item = (CColorListItem*)itemAt( e->pos() );
    if( item != 0 )
    {
      setCurrentItem( item );
    }
  }
}


void CColorListBox::dropEvent( QDropEvent *e )
{
  QVariant color = e->mimeData()->colorData();
  if( color.isValid() )
  {
    int index = currentItem();
    if( index != -1 )
    {
      CColorListItem *colorItem = (CColorListItem*)item(index);
      colorItem->setColor(color.value<QColor>());
      triggerUpdate( false ); // Redraw item
    }
    mCurrentOnDragEnter = -1;
  }
}



CColorListItem::CColorListItem( const QString &text, const QColor &color )
  : Q3ListBoxItem(), mColor( color ), mBoxWidth( 30 )
{
  setText( text );
}


const QColor &CColorListItem::color( void )
{
  return( mColor );
}


void CColorListItem::setColor( const QColor &color )
{
  mColor = color;
}


void CColorListItem::paint( QPainter *p )
{
  QFontMetrics fm = p->fontMetrics();
  int h = fm.height();

  p->drawText( mBoxWidth+3*2, fm.ascent() + fm.leading()/2, text() );

  p->setPen( Qt::black );
  p->drawRect( 3, 1, mBoxWidth, h-1 );
  p->fillRect( 4, 2, mBoxWidth-2, h-3, mColor );
}


int CColorListItem::height(const Q3ListBox *lb ) const
{
  return( lb->fontMetrics().lineSpacing()+1 );
}


int CColorListItem::width(const Q3ListBox *lb ) const
{
  return( mBoxWidth + lb->fontMetrics().width( text() ) + 6 );
}






#include "optiondialog.moc"
