/***************************************************************************
 *   Copyright (C) 2005 Novell, Inc.                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA           *
 ***************************************************************************/

#include <qstring.h>
#include <qstringlist.h>
#include <qdir.h>
#include <qclipboard.h>
#include <kfiledialog.h>
#include <kcmdlineargs.h>
#include <fcntl.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdebug.h>
#include <khelpmenu.h>
#include <kprocess.h>

#include <qtooltip.h>
#include <qdom.h>

#include "searchdlg.h"
#include "kerryapp.h"
#include "configdialog.h"
#include <kaction.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <klineedit.h>

#define MAX_HISTORY_ITEMS 5
#define HISTORY_ITEMS_START_ID 100


KerryApplication::KerryApplication() : KUniqueApplication(),
    hitListWindow(0), sysTrayIcon(0), m_history(0)
{
}

KerryApplication::~KerryApplication()
{
}

int KerryApplication::newInstance()
{
    if (!hitListWindow)
        init(KGlobal::instance()->aboutData());

        KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
        if (args->isSet("show-searchdialog")) {
            hitListWindow->showSearchDialog();
        }

        if (args->count()==1)
           search(args->arg(0));

        args->clear();

    return KUniqueApplication::newInstance();
}

void KerryApplication::init(const KAboutData* /*about*/)
{
  if (hitListWindow)
      return;

  hitListWindow = new SearchDlg();
  connect(hitListWindow,SIGNAL(searchStarted(const QString&)),SLOT(addToHistory(const QString&)));

  KConfig *config = KGlobal::config();
  config->setGroup("General");
  hitListWindow->setDisplayAmount(config->readNumEntry("DisplayAmount", 5));
  hitListWindow->setSortOrder(config->readNumEntry("DefaultSortOrder",0));
  m_history = config->readListEntry("History");

  sysTrayIcon = new KSystemTray(hitListWindow);
  KPopupMenu *menu = sysTrayIcon->contextMenu();
  connect(menu,SIGNAL(aboutToShow()),SLOT(aboutToShowSysTrayMenu()));
  connect(menu,SIGNAL(activated(int)),SLOT(historySelected(int)));

  menu->insertSeparator();
  menu->insertItem(SmallIconSet("history_clear"),
                   i18n("Clear Search History"), this,
                   SLOT(clearHistory()));

  menu->insertItem(SmallIconSet("configure"),
                   i18n("Configure Kerry..."), this,
                   SLOT(configure()));
/*
  KHelpMenu *helpmenu = new KHelpMenu(hitListWindow, about, false);
  menu->insertItem( SmallIconSet("help"), KStdGuiItem::help().text(), helpmenu->menu() );
*/
  globalKeys = new KGlobalAccel(this);
  KGlobalAccel* keys = globalKeys;
  keys->insert( "Program:kerry", i18n("Kerry Beagle Search") );

  KShortcut showDialogShortcut = KShortcut(Key_F12);
  showDialogShortcut.append( KKey( ALT+Key_Space) );
  keys->insert( "Show Kerry Dialog", i18n("Show Search Dialog"), QString::null, showDialogShortcut, showDialogShortcut, hitListWindow, SLOT(showSearchDialog()) );
  keys->insert( "Search Primary Selection with Kerry", i18n("Search Primary Selection"), QString::null, CTRL+ALT+Key_Space, CTRL+ALT+Key_Space, this, SLOT(searchPrimarySelection()) );
  globalKeys->readSettings();
  globalKeys->updateConnections();

  sysTrayIcon->setPixmap(sysTrayIcon->loadIcon("filefind"));
  QToolTip::add(sysTrayIcon, i18n("Kerry Beagle Search (%1)").arg(globalKeys->shortcut("Show Kerry Dialog").seq(0).toString()));

  sysTrayIcon->show();

  sysTrayIcon->actionCollection()->action("file_quit")->setShortcut(KShortcut());
  disconnect(sysTrayIcon->actionCollection()->action("file_quit"), SIGNAL(activated()), sysTrayIcon, SLOT(maybeQuit()));
  connect(sysTrayIcon->actionCollection()->action("file_quit"), SIGNAL(activated()), this, SLOT(quitKerry()));
}

void KerryApplication::search(const QString& text)
{
  if (hitListWindow)
    hitListWindow->search(text);
}

void KerryApplication::aboutToShowSysTrayMenu()
{
  KPopupMenu *menu = sysTrayIcon->contextMenu();

  for (int id=HISTORY_ITEMS_START_ID;id<=HISTORY_ITEMS_START_ID+MAX_HISTORY_ITEMS;id++)
    menu->removeItem(id);

  if (m_history.count()==0) {
    menu->insertItem(i18n("<No Recent Searches>"),HISTORY_ITEMS_START_ID,1);
    menu->setItemEnabled(HISTORY_ITEMS_START_ID,false);
    return;
  }

  for (int i=0;i<(int)m_history.count();i++)
    menu->insertItem(m_history[i],i+HISTORY_ITEMS_START_ID,i+1);
}

void KerryApplication::historySelected(int id)
{
  if (id<HISTORY_ITEMS_START_ID)
    return;

  if (hitListWindow)
    hitListWindow->search(sysTrayIcon->contextMenu()->text(id));
}

void KerryApplication::searchPrimarySelection()
{
  QApplication::clipboard()->setSelectionMode( true );
  QString text = QApplication::clipboard()->text();
  if (!text.isEmpty() && hitListWindow)
    hitListWindow->search(text);
}

void KerryApplication::quitKerry()
{
    int autoStart = KMessageBox::questionYesNoCancel( 0L, i18n("Should Kerry start automatically\nwhen you login?"), i18n("Automatically Start Kerry?"), i18n("&Start"), i18n("&Do Not Start") );

  KConfig *config = KGlobal::config();
  config->setGroup("General");
  if ( autoStart == KMessageBox::Yes ) {
    config->writeEntry("AutoStart", true);
  } else if ( autoStart == KMessageBox::No) {
    config->writeEntry("AutoStart", false);
  } else  // cancel chosen don't quit
    return;
  config->writeEntry("History",m_history);
  config->sync();

  if (hitListWindow)
  	hitListWindow->hide();

  KProcess *proc = new KProcess;
  *proc << "beagle-shutdown";
  if (!proc->start())
      kdDebug() << "Couldn't run beagle-shutdown." << endl;

  qApp->closeAllWindows();
  qApp->quit();
}

void KerryApplication::addToHistory(const QString& term)
{
  if (m_history.contains(term))
    return;

  m_history.push_front(term);

  if (m_history.count()>MAX_HISTORY_ITEMS)
    m_history.pop_back();
}

void KerryApplication::clearHistory()
{
  m_history.clear();
}

void KerryApplication::readIndexConfig(bool &indexHomeDir, QStringList &roots, QStringList &excludeTypes, QStringList &excludeValues)
{
    indexHomeDir = true;
    roots.clear();
    excludeTypes.clear();
    excludeValues.clear();

    QDomDocument doc( "mydocument" );
    QFile file( QDir::home().absPath()+"/.beagle/config/indexing.xml" );
    if ( !file.open( IO_ReadOnly ) )
        return;

    if ( !doc.setContent( &file ) ) {
        file.close();
        return;
    }
    file.close();

    QDomElement docElem = doc.documentElement();

    QDomNode n = docElem.firstChild();
    while( !n.isNull() ) {
        QDomElement e = n.toElement();
        if( !e.isNull() ) {
            if (e.tagName()=="IndexHomeDir")
               indexHomeDir = e.text()=="true";
            else if (e.tagName()=="Roots") {
               QDomNode ro = n.firstChild();
               while( !ro.isNull() ) {
                  QDomElement exel = ro.toElement();
                  if( !exel.isNull() )
                    roots << exel.text();
                  ro = ro.nextSibling();
                }
            }
            else if (e.tagName()=="Excludes") {
                QDomNode ex = n.firstChild();
                while( !ex.isNull() ) {
                  QDomElement exel = ex.toElement();
                  if( !exel.isNull() ) {
                    excludeTypes << exel.attribute("Type");
		    excludeValues << exel.attribute("Value");
                  }
                  ex = ex.nextSibling();
                }
            }
        }
        n = n.nextSibling();
    }
}

bool KerryApplication::saveIndexConfig(bool indexHomeDir, QStringList roots, QStringList excludeTypes, QStringList excludeValues)
{
  QDir beagleDir(QDir::home().absPath()+"/.beagle");
  if (!beagleDir.exists())
    beagleDir.mkdir(QDir::home().absPath()+"/.beagle");

  QDir beagleConfigDir(QDir::home().absPath()+"/.beagle/config");
  if (!beagleConfigDir.exists())
    beagleConfigDir.mkdir(QDir::home().absPath()+"/.beagle/config");

  QFile configFile( QDir::home().absPath()+"/.beagle/config/indexing.xml" );
  if ( !configFile.open( IO_WriteOnly ) )
        return false;

  QDomDocument doc( QString::null );
  doc.appendChild( doc.createProcessingInstruction(
                   "xml", "version=\"1.0\" encoding=\"UTF-8\"" ) );

  QDomElement root = doc.createElement( "IndexingConfig" );
  root.setAttribute("xmlns:xsd","http://www.w3.org/2001/XMLSchema");
  root.setAttribute("xmlns:xsi","http://www.w3.org/2001/XMLSchema-instance");
  doc.appendChild( root );

  QDomElement rootselement = doc.createElement( "Roots" );
  root.appendChild(rootselement);

  for ( QStringList::Iterator it = roots.begin(); it != roots.end(); ++it ) {
    QDomElement tag = doc.createElement( "Root" );
    rootselement.appendChild( tag );

    QDomText t = doc.createTextNode( *it );
    tag.appendChild( t );
  }

  QDomElement tag = doc.createElement( "IndexHomeDir" );
  root.appendChild( tag );

  QDomText t = doc.createTextNode( indexHomeDir ? "true" : "false" );
  tag.appendChild( t );

  QDomElement excludeselement = doc.createElement( "Excludes" );
  root.appendChild(excludeselement);

  QStringList::Iterator it_types = excludeTypes.begin();
  for ( QStringList::Iterator it = excludeValues.begin(); it != excludeValues.end(); ++it ) {
    QDomElement t = doc.createElement( "ExcludeItem" );
    t.setAttribute("Type", *it_types);
    t.setAttribute("Value", *it);
    excludeselement.appendChild( t );

    ++it_types;
  }

  QTextStream stream( &configFile );
  stream << doc.toString();
  configFile.close();

  KProcess *proc = new KProcess;
  *proc << "beagle-config";
  *proc << "--beagled-reload-config";
  if (!proc->start()) {
    kdError("Could not make Beagle reload its config.");
    return false;
  }

  return true;
}

void KerryApplication::configure()
{
  ConfigDialog *dlg = new ConfigDialog( globalKeys );

  KConfig *config = KGlobal::config();
  config->setGroup("Beagle");
  dlg->setStartBeagle(config->readBoolEntry("AutoStart",false));
  config->setGroup("General");
  dlg->setDefaultSortOrder(config->readNumEntry("DefaultSortOrder",0));

  dlg->setMaxResultsDisplayed(hitListWindow->getDisplayAmount());

  bool indexHomeDir;
  QStringList roots, excludeTypes, excludeValues;
  readIndexConfig(indexHomeDir, roots, excludeTypes, excludeValues);

  dlg->setIndexHome(indexHomeDir);
  dlg->setRoots(roots);
  dlg->setExcludes(excludeTypes, excludeValues);

  if ( dlg->exec() == QDialog::Accepted ) {
        dlg->commitShortcuts();
        // the keys need to be written to kdeglobals, not kickerrc --ellis, 22/9/02
        globalKeys->writeSettings(0, true);
        globalKeys->updateConnections();

	QToolTip::remove(sysTrayIcon);
	QToolTip::add(sysTrayIcon, i18n("Kerry Beagle Search (%1)").arg(globalKeys->shortcut("Show Kerry Dialog").seq(0).toString()));

        config->setGroup("General");
        const int sortOrder = dlg->getDefaultSortOrder();
        config->writeEntry("DefaultSortOrder",sortOrder);
        hitListWindow->setSortOrder(sortOrder);

	const int maxresults = dlg->getMaxResultsDisplayed();
        hitListWindow->setDisplayAmount(maxresults);
        config->writeEntry("DisplayAmount",maxresults);

	config->setGroup("Beagle");
	config->writeEntry("AutoStart",dlg->getStartBeagle());
	config->sync();

        saveIndexConfig(dlg->getIndexHome(), dlg->getRoots(), dlg->getTypes(), dlg->getValues());
  }

  delete dlg;
}

#include "kerryapp.moc"
