/***************************************************************************
 *   Copyright (C) 2005 Novell, Inc.                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA           *
 ***************************************************************************/

#include "kerryapp.h"
#include "searchdlg.h"
#include "hitwidget.h"
#include "kwidgetlistbox.h"
#include "kerrylabel.h"

#include <qlayout.h>
#include <qtextcodec.h>
#include <kdebug.h>
#include <kfiledialog.h>
#include <kpushbutton.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <kapplication.h>
#include <kfileitem.h>
#include <kiconloader.h>
#include <klocale.h>
#include <klineedit.h>
#include <kurllabel.h>
#include <krun.h>
#include <krfcdate.h>
#include <qtable.h>
#include <qdir.h>
#include <qlabel.h>
#include <qcombobox.h>
#include <kio/job.h>
#include <kprotocolinfo.h>
#include <ktrader.h>
#include <kprocess.h>
#include <qregexp.h>
#include <kaction.h>
#include <kpopupmenu.h>
#include <dcopclient.h>
#include <qcheckbox.h>
#include <kdesktopfile.h>
#include <qtoolbutton.h>
#include <kcombobox.h>
#include <kservicegroup.h>
#include <kbookmarkmanager.h>
#include <qcursor.h>

#include <kwin.h>
#include <kwinmodule.h>

SearchDlg::SearchDlg(QWidget *parent, const char *name)
 : DCOPObject("search"), HitsLayout(parent, name)
{
#define switchLabelCount 19
static QLabel* switchLabel[switchLabelCount] = {showEverything, showApplications, showContacts, showDocuments,
                                  showConversations, showImages, showMedia, showWebPages, showFilePathName, sortByType, sortByDate, sortByName, sortByRelevance, showAnyDate, showToday, showSinceYesterday, showThisWeek, 
                                  showThisMonth, showThisYear};

  g_type_init();
  beagle_search = NULL;
  editSearch->setMaxCount(MAX_HISTORY_ITEMS);
  editSearch->setDuplicatesEnabled(false);

#if KDE_IS_VERSION( 3, 5, 4 )
  KLineEdit *lineedit = new KLineEdit(editSearch);
  lineedit->setClickMessage(i18n( "Applications, Contacts, Conversations, Files and more..." ) );
  editSearch->setLineEdit(lineedit);
#endif

  connect(editSearch->lineEdit(), SIGNAL(returnPressed()), SLOT(search()));
  connect(editSearch->lineEdit(), SIGNAL(textChanged(const QString &)), SLOT(searchChanged(const QString &)));
  showMode = Everywhere;
  dateRange = AnyDate;
  showBigTiles = false;

  QFont f= sortByRelevance->font();
  f.setBold(true);
  QFontMetrics fm(f);
  int minWidth=0;
  for (int i=0; i<switchLabelCount; ++i)
     minWidth = MAX(minWidth, fm.width(switchLabel[i]->text()));
  frame3->setMinimumWidth(minWidth+25);

  showEverything->installEventFilter(this);
  showApplications->installEventFilter(this);
  showContacts->installEventFilter(this);
  showDocuments->installEventFilter(this);
  showConversations->installEventFilter(this);
  showImages->installEventFilter(this);
  showMedia->installEventFilter(this);
  showWebPages->installEventFilter(this);
  showFilePathName->installEventFilter(this);
  sortByType->installEventFilter(this);
  sortByDate->installEventFilter(this);
  sortByName->installEventFilter(this);
  sortByRelevance->installEventFilter(this);
  showAnyDate->installEventFilter(this);
  showToday->installEventFilter(this);
  showSinceYesterday->installEventFilter(this);
  showThisWeek->installEventFilter(this);
  showThisMonth->installEventFilter(this);
  showThisYear->installEventFilter(this);
  connect(buttonFind, SIGNAL(clicked()), SLOT(search()));
  connect(buttonClear, SIGNAL(clicked()), SLOT(slotButtonClear()));
  connect(buttonPrevious, SIGNAL(clicked()), SLOT(slotPrevious()));
  connect(buttonNext, SIGNAL(clicked()), SLOT(slotNext()));
  connect(tableHits, SIGNAL(contextMenuRequested (int, int, const QPoint &)), SLOT(slotContextMenu(int, int, const QPoint &)));
  buttonFind->setIconSet( BarIconSet( "key_enter", 22 ) );
  buttonConfigure->setGuiItem(KStdGuiItem::configure());
  connect(buttonConfigure, SIGNAL(clicked()), SIGNAL(configure()));
  setMouseTracking(true);
  results.setAutoDelete(true);
  results.clear();
  displayed_results.clear();
  displayAmount = 5;
  displayOffset = 0;
  labelStatus->setAlignment(Qt::SingleLine);
  pPreviewJob = 0;
  pPreviewMimeTypes = 0;
  previewItems.clear();
  previewItems.setAutoDelete(true);
  pending_showQuickTips=true;
  updateStatus();
  defaultSortOrder = Modified;
  currentSortOrder = defaultSortOrder;
  kapp->dcopClient()->setDefaultObject( objId() );
  beagleJustStarted = false;
  m_searchPixmap->setPixmap( BarIcon( "find", 32 ) );

  QTextCodec::setCodecForCStrings(QTextCodec::codecForName( "utf8" ));
  encodingRegexp = QRegExp("%[\\dA-F][\\dA-F]");

  bookmarkManager = 0;
  m_addressBook = 0;
  still_searching = false;
}

SearchDlg::~SearchDlg()
{
}

void SearchDlg::showQuickTips()
{
  tableHits->clear();

  HitWidget* item = new HitWidget(QString::null, QString::null);
  QLabel *headerLabel = new QLabel(item);
  headerLabel->setText(i18n("Quick Tips"));
  item->insertHeaderWidget(0,headerLabel);

  item->setIcon("messagebox_info");
  item->setDescriptionText("<qt>"+i18n("- You can use upper and lower case; search is case-insensitive.<br>"
                           "- To search for optional terms, use OR.  ex: <b>George OR Ringo</b><br>"
                           "- To exclude search terms, use the minus symbol in front, such as <b>-cats</b><br>"
                           "- When searching for a phrase, add quotes. ex: <b>\"There be dragons\"</b><br>"
                           "- Add ext:type to specify a file extension, ex: <b>ext:txt</b> or <b>ext:</b> for none")+"</qt>");
  tableHits->insertItem(item);

  item = new HitWidget(QString::null, QString::null);
  headerLabel = new QLabel(item);
  headerLabel->setText(i18n("Configuration"));
  item->insertHeaderWidget(0,headerLabel);

  item->setIcon("package_settings");
  item->setDescriptionText("<qt>"+i18n("- Choose what folders and resources shall be indexed - or not.<br>"
                           "- Change the sort order and the number of shown results.<br>"
                           "- Define your own shortcuts to invoke the search dialog.")+"</qt>");
  KURLLabel *buttonStart = new KURLLabel(item);
  buttonStart->setPixmap(SmallIcon("exec"));
  item->insertHitWidget(0,buttonStart);
  connect(buttonStart, SIGNAL(leftClickedURL()), SIGNAL(configure()));

  buttonStart = new KURLLabel(item);
  buttonStart->setText(i18n("Open configuration dialog"));
  item->insertHitWidget(1,buttonStart);
  connect(buttonStart, SIGNAL(leftClickedURL()), SIGNAL(configure()));

  tableHits->insertItem(item);

  labelStatus->setText("");
}

void SearchDlg::search(const QString & search)
{
  SearchDlg::search(search, "everything");
}

void SearchDlg::search(const QString & search, const QString &scope)
{
#define showSwitchesCount 9
static QLabel* showSwitches[showSwitchesCount] = {showEverything, showApplications, showContacts, showDocuments,
                                   showConversations, showImages, showMedia, showWebPages, showFilePathName};
static QString nameSwitches[showSwitchesCount] = {"everything", "applications", "contacts", "documents",
                                  "conversations", "images", "media", "webpages", "path"};

  pending_showQuickTips=false;
  showSearchDialog();

  QLabel* newShowLabel = showEverything;
  ScopeType newShowMode = Everywhere;
  for (int i=0; i<showSwitchesCount; ++i)
     if (scope == nameSwitches[i]) {
        newShowLabel = showSwitches[i];
        newShowMode = (ScopeType)i;
        break;
     }

  if (showMode!=newShowMode) {
     QFont f= showSwitches[showMode]->font();
     f.setBold(false);
     showSwitches[showMode]->setFont(f);
     showMode = newShowMode;
     f.setBold(true);
     newShowLabel->setFont(f);
  }
  editSearch->lineEdit()->setText(search);
  this->search();
}

void SearchDlg::searchChanged(const QString & search)
{
  QString _search = search;
  buttonFind->setEnabled(_search.replace("*", QString::null).length()>2);
  if (!search.isEmpty() && displayed_results.count()==0 && tableHits->count()==1)
    tableHits->clear();
  if (search.isEmpty() && displayed_results.count()==0 && tableHits->count()==0)
    showQuickTips();
}

void SearchDlg::setDisplayAmount(int amount)
{
    if (amount<1 || displayAmount==amount)
	return;

    buttonPrevious->setShown(amount>1);
    buttonNext->setShown(amount>1);

    displayAmount = amount;
    displayOffset = 0;
    tableHits->setUpdatesEnabled(false);
    fillTableHits();
    tableHits->setUpdatesEnabled(true);
    updateStatus();
}

void SearchDlg::setSortOrder(int order)
{
static QLabel* sortSwitches[4] = {sortByType, sortByDate, sortByName, sortByRelevance};

    defaultSortOrder = order;

    if (currentSortOrder==order)
        return;

    QFont f= sortSwitches[currentSortOrder]->font();
    f.setBold(false);
    sortSwitches[currentSortOrder]->setFont(f);
    f.setBold(true);
    sortSwitches[order]->setFont(f);

    currentSortOrder = order;
    if (displayed_results.count())
        sortFilterResults();
}

void SearchDlg::slotButtonClear()
{
    editSearch->clear();
    slotClear();
    showQuickTips();
    currentSortOrder = defaultSortOrder;
}

void SearchDlg::slotClear()
{
    if (beagle_search != NULL) {
	kdDebug () << "Previous client w/id " << beagle_search->id << " running ... stopping it." << endl;
	beagle_search->stopClient ();
    }

    displayOffset = 0;
    stopPreview();
    tableHits->clear();
    displayed_results.clear();
    results.clear();
    updateStatus();
}

void SearchDlg::slotPrevious()
{
    if (displayOffset==0)
      return;

    displayOffset-=displayAmount;
    tableHits->setUpdatesEnabled(false);
    fillTableHits();
    tableHits->setUpdatesEnabled(true);
    updateStatus();
}

void SearchDlg::slotNext()
{
    if (displayAmount==1 || displayOffset+displayAmount>=(int)displayed_results.count())
      return;

    displayOffset+=displayAmount;
    tableHits->setUpdatesEnabled(false);
    fillTableHits();
    tableHits->setUpdatesEnabled(true);
    updateStatus();
}

void SearchDlg::sortFilterResults()
{
    displayOffset = 0;
    stopPreview();
    tableHits->clear();
    displayed_results.clear();
    displayResults(results);
    updateStatus();
}

void SearchDlg::fillTableHits()
{
    stopPreview();
    tableHits->clear();
    previewItems.clear();
    if (displayAmount==1) {
        for(int i = 0; i < (int)displayed_results.count(); ++i) {
	    insertResult(displayed_results.at(i),i);
        }
    }
    else {
      if (displayOffset+displayAmount<=(int)displayed_results.count()) {
        for(int i = displayOffset; i < displayOffset+displayAmount; ++i) {
	    insertResult(displayed_results.at(i),i-displayOffset);
        }
      }
      else
      for(uint i = displayOffset; i < (displayed_results.count() % displayAmount)+displayOffset; ++i) {
	    insertResult(displayed_results.at(i),i-displayOffset);
      }
    }
    if (previewItems.count()) {
      startPreview(previewItems);
    }
}

void SearchDlg::updateStatus()
{
    buttonPrevious->setEnabled(displayOffset>0);
    buttonNext->setEnabled(displayOffset+displayAmount<(int)displayed_results.count());
    labelStatus->setAlignment(Qt::SingleLine);
    const int count = displayed_results.count();
    QString text;
    if (displayAmount==1)
       text=i18n("<b>%1 results</b> found.").arg(displayed_results.count());
    else if (count==0)
       text=i18n("<qt>No results.</qt>").replace("<qt>",QString::null).replace("</qt>",QString::null);
    else if (displayOffset==0)
       text=i18n("Best <b>%1 results of %2</b> shown.").arg(tableHits->count()).arg(displayed_results.count());
    else
       text=i18n("Results <b>%1 through %2 of %3</b> are shown.").arg(displayOffset+1).arg(displayOffset+tableHits->count()).arg(displayed_results.count());

#if 0
    if (still_searching)
       text=text+" <b>"+i18n("(still searching)")+"</b>";
#endif

    labelStatus->setText(text);
}

void SearchDlg::search()
{
    current_query.set(editSearch->lineEdit()->text());
    if (current_query.get().replace("*", QString::null).length()<3)
       return;

    editSearch->addToHistory(current_query.get());

    if (!beagle_util_daemon_is_running()) {
       tableHits->clear();
       HitWidget* item = new HitWidget(QString::null, QString::null);
       QLabel *headerLabel = new QLabel(item);
       headerLabel->setText(i18n("The query for \"%1\" failed.").arg(current_query.get()));
       item->insertHeaderWidget(0,headerLabel);

       item->setIcon("messagebox_critical");
       item->setDescriptionText("<qt>"+i18n("The likely cause is that the Beagle daemon is not running.")+"</qt>");

       cb_beagleStart = new QCheckBox(i18n("Automatically start Beagle daemon at login"),item);
       item->insertTextWidget(1,cb_beagleStart);

       KURLLabel *buttonStart = new KURLLabel(item);
       buttonStart->setPixmap(SmallIcon("exec"));
       item->insertHitWidget(0,buttonStart);
       connect(buttonStart, SIGNAL(leftClickedURL()), SLOT(slotStartBeagle()));

       buttonStart = new KURLLabel(item);
       buttonStart->setText(i18n("Click to start the Beagle daemon"));
       item->insertHitWidget(1,buttonStart);
       connect(buttonStart, SIGNAL(leftClickedURL()), SLOT(slotStartBeagle()));

       tableHits->insertItem(item);
       labelStatus->setText("");
       return;
    }

    slotClear();
    labelStatus->setText(i18n("Searching..."));

    // Beagle search
    if (beagle_search != NULL) {
	kdDebug () << "Previous client w/id " << beagle_search->id << " still running ... ignoring it." << endl;
	beagle_search->stopClient ();
    }

    current_beagle_client_id = KApplication::random ();
    kdDebug () << "Creating client with id:" << current_beagle_client_id << endl;

    m_searchPixmap->setMovie(QMovie(locate( "appdata", "search-running.mng" )));

    new_results.clear();

    kdDebug() << "searchProgramList" << endl;
    searchProgramList(QString::null);

    kdDebug() << "searchBookmars" << endl;
    if (!bookmarkManager)
      bookmarkManager = KBookmarkManager::userBookmarksManager();
    searchBookmarks(bookmarkManager->root());

    kdDebug() << "searchAddressBook" << endl;
    searchAddressbook();

    displayResults(new_results);

    kdDebug() << "starting BeagleSearch" << endl;
    beagle_search = new BeagleSearch(current_beagle_client_id, this, current_query.get());
    beagle_search->start();
    still_searching = true;
}

QString SearchDlg::takeProperty( const QString& property, QStringList& propertyList )
{
      QString ret = QString::null;
      BeagleSearch::PropertyList::iterator it;
      for ( it = propertyList.begin(); it != propertyList.end(); ++it ) {
         const QString search = property+'=';
         if ((*it).startsWith(search)) {
           ret = (*it).remove(search);
           propertyList.erase(it);
           break;
         }
      }
      return ret;
}

QDateTime SearchDlg::datetimeFromString( const QString& s)
{
      int year( s.mid( 0, 4 ).toInt() );
      int month( s.mid( 4, 2 ).toInt() );
      int day( s.mid( 6, 2 ).toInt() );
      int hour( s.mid( 8, 2 ).toInt() );
      int min( s.mid( 10, 2 ).toInt() );
      int sec( s.mid( 12, 2 ).toInt() );
      return QDateTime(QDate(year,month,day),QTime(hour,min,sec));
}

void SearchDlg::slotCleanClientList ()
{
    toclean_list_mutex.lock ();
    BeagleSearch *old_client = toclean_client_list.take (0);
    if (old_client != NULL) { // failsafe
	kdDebug () << "Cleanup old client " << old_client->id << endl;
	delete old_client;
    }
    toclean_list_mutex.unlock ();
}

void SearchDlg::customEvent (QCustomEvent *e)
{
    if (e->type () == RESULTFOUND) {
        BeagleSearch::BeagleResultList* items = (BeagleSearch::BeagleResultList*) e->data();
        if (items->count() == 0 || current_beagle_client_id != items->first()->client_id) {
            kdDebug () << "Stale result" << endl;
	    delete items;
        } else {
            kdDebug () << "Good results ...total=" << items->count() << endl;
            searchHasOutput(*items);
        }
    } else if (e->type () == RESULTGONE) {
        BeagleSearch::BeagleVanishedURIList* items = (BeagleSearch::BeagleVanishedURIList*) e->data();
        if (items->list.count() == 0 || current_beagle_client_id != items->client_id) {
            kdDebug () << "Stale leaving from " << items->client_id << endl;
	    delete items;
        } else {
            kdDebug () << "Good leavings ...total=" << items->list.count() << endl;
            searchLostOutput(items->list);
        }
    } else if (e->type () == SEARCHOVER) {
        BeagleSearch *client = (BeagleSearch *) e->data ();
	if (client == NULL) {
	    kdDebug () << "Query finished but client is already deleted" << endl;
	    searchFinished();
	    return;
	}
//        kdDebug () << "Query finished for id=" << client->id << endl;
	if (current_beagle_client_id == client->id) {
	    searchFinished();
	}
    } else if (e->type () == KILLME) {
        BeagleSearch *client = (BeagleSearch *) e->data ();
        if (client==beagle_search)
            beagle_search = NULL;

	if (client->finished ())
	    delete client;
	else {
	    // add client to cleanup list
	    toclean_list_mutex.lock ();
	    toclean_client_list.append (client);
	    kdDebug () << "Scheduling client to be deleted in 500ms" << endl;
	    toclean_list_mutex.unlock ();
	    QTimer::singleShot (500, this, SLOT (slotCleanClientList ()));
	}
    }
}

void SearchDlg::insertResult(BeagleSearch::beagle_result_struct *result,int index)
{
      KURL url(*(result->uri));

      HitWidget* item = new HitWidget(*(result->uri),*(result->mime_type),tableHits);
      item->setCollapsible(result);
      item->setCollapsed(!result->show_expanded);
      connect(item, SIGNAL(uncollapsed(HitWidget*)), SLOT(itemUncollapsed(HitWidget*)));

      item->icon->setURL(*(result->uri));
      connect(item->icon, SIGNAL(leftClickedURL()), SLOT(slotOpen()));

      KURLLabel *buttonGo = new KerryLabel(item);
      buttonGo->setPixmap(SmallIcon( *(result->mime_type)=="application/x-desktop" ? "exec" : "fileopen") );
      buttonGo->setURL(*(result->uri));
      item->insertHitWidget(0,buttonGo);
      connect(buttonGo, SIGNAL(leftClickedURL()), SLOT(slotOpen()));

      buttonGo = new KerryLabel(item);
      buttonGo->setText( *(result->mime_type)=="application/x-desktop" ? i18n("Run") : i18n("Open") );
      buttonGo->setURL(*(result->uri));
      item->insertHitWidget(1,buttonGo);
      connect(buttonGo, SIGNAL(leftClickedURL()), SLOT(slotOpen()));

      if (result->tilegroup == BeagleSearch::Website) {
        item->setIcon("network");

        QString description = "<qt>";
        if (result->hit_type!="Google") {
          QDateTime datetime;
          datetime.setTime_t(result->last_index_time);
          if (datetime.date().year()>1970)
             description = description + i18n("Last viewed: %1").arg(KGlobal::locale()->formatDateTime(datetime,false))+"<br>";
        }

        item->setDescriptionText(description +i18n("URL:")+" "+*(result->uri)+"</qt>");

        if (result->snippet)
            item->setPropertiesText("<qt>"+*(result->snippet)+"</qt>");

        KerryLabel *headerFileLabel = new KerryLabel(item);
        QStringList _properties(result->properties);
        QString title = takeProperty("dc:title",_properties);
        if (title.isEmpty())
           title = takeProperty("Title",_properties);

        headerFileLabel->setText(title.isEmpty() ? i18n("Untitled Page") : title);
        headerFileLabel->setAlignment(headerFileLabel->alignment() | Qt::SingleLine);
        headerFileLabel->setURL(*(result->uri));
        item->insertHeaderWidget(0,headerFileLabel);
        connect(headerFileLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
      }
      else if (result->tilegroup == BeagleSearch::Feed) {
        item->setIcon("network");

        QDateTime datetime;
        datetime.setTime_t(result->last_index_time);
        item->setDescriptionText("<qt>"+ i18n("Published: %1").arg(KGlobal::locale()->formatDateTime(datetime,false))+"<br>");

        if (result->snippet)
          item->setPropertiesText("<qt>"+*(result->snippet)+"</qt>");

        QLabel *headerLabel = new QLabel(item);
        headerLabel->setText(i18n("Weblog:"));
        headerLabel->setAlignment(headerLabel->alignment() | Qt::SingleLine);
        item->insertHeaderWidget(0,headerLabel);

        QStringList _properties(result->properties);
        QString title = takeProperty("dc:title",_properties);
        QString identifier = takeProperty("dc:identifier", _properties);
        item->setUri(identifier);
        buttonGo->setURL(identifier);

        KerryLabel *headerFileLabel = new KerryLabel(item);
        headerFileLabel->setText(title.isEmpty() ? i18n("Untitled Entry") : title);
        headerFileLabel->setAlignment(headerFileLabel->alignment() | Qt::SingleLine);
        headerFileLabel->setURL(identifier);
        item->insertHeaderWidget(1,headerFileLabel);
        connect(headerFileLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
      }
      else if ((*(result->uri)).startsWith("knotes:/")) {
        item->setIcon("knotes");

        QDateTime datetime;
        datetime.setTime_t(result->last_index_time);
        item->setDescriptionText("<qt>"+ i18n("Last modified: %1").arg(KGlobal::locale()->formatDateTime(datetime,false))+"<br>");

        if (result->snippet)
          item->setPropertiesText("<qt>"+*(result->snippet)+"</qt>");

        QStringList _properties(result->properties);
        QString title = takeProperty("dc:title",_properties);

        KerryLabel *headerNameLabel = new KerryLabel(item);
        headerNameLabel->setText(title.isEmpty() ? i18n("Untitled") : title);
        headerNameLabel->setURL(item->uri());
        item->insertHeaderWidget(0,headerNameLabel);
        connect(headerNameLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
      }
      else if (result->tilegroup == BeagleSearch::Note) {
        item->setIcon("contents2");
        QDateTime datetime;
        datetime.setTime_t(result->last_index_time);
        item->setDescriptionText("<qt>"+ i18n("Last modified: %1").arg(KGlobal::locale()->formatDateTime(datetime,false))+"<br>");


        if (result->snippet)
          item->setPropertiesText("<qt>"+*(result->snippet)+"</qt>");

        QStringList _properties(result->properties);
        QString title = takeProperty("dc:title",_properties);

        KerryLabel *headerNoteLabel = new KerryLabel(item);
        headerNoteLabel->setText(title.isEmpty() ? i18n("Untitled Entry") : title);
        headerNoteLabel->setAlignment(headerNoteLabel->alignment() | Qt::SingleLine);
        headerNoteLabel->setURL(item->uri());
        item->insertHeaderWidget(0,headerNoteLabel);
        connect(headerNoteLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
      }
      else if ((*(result->uri)).startsWith("calendar:/")) {
        item->setIcon("ximian-evolution-calendar");

        QStringList _properties(result->properties);
        QString summary = takeProperty("fixme:summary",_properties);

        QString properties;
        QDateTime datetime;
        datetime = datetimeFromString(takeProperty("fixme:starttime",_properties));
        properties = properties + i18n("Start time: %1").arg(KGlobal::locale()->formatDateTime(datetime,false)) + "<br>";
        datetime = datetimeFromString(takeProperty("fixme:endtime",_properties));
        properties = properties + i18n("End time: %1").arg(KGlobal::locale()->formatDateTime(datetime,false)) + "<br>";

        BeagleSearch::PropertyList::iterator it;
        for ( it = _properties.begin(); it != _properties.end(); ++it )
                properties=properties+(*it);

        item->setDescriptionText("<qt>"+properties+"</qt>");

        KerryLabel *headerSummaryLabel = new KerryLabel(item);
        headerSummaryLabel->setText(summary.isEmpty() ? i18n("No Summary Specified") : summary);
        headerSummaryLabel->setURL(item->uri());
        item->insertHeaderWidget(0,headerSummaryLabel);
        connect(headerSummaryLabel, SIGNAL(leftClickedURL(const QString&)), SLOT(slotOpenEvolution(const QString&)));
      }
      else if ((*(result->uri)).startsWith("contacts:/")) {
        item->setIcon("ximian-evolution-addressbook");

        QStringList _properties(result->properties);
        QString name = takeProperty("fixme:Name",_properties);
        QString email = takeProperty("fixme:Email1",_properties);

        QString properties;
        BeagleSearch::PropertyList::iterator it;
        for ( it = _properties.begin(); it != _properties.end(); ++it )
                properties=properties+(*it);

        if (!properties.isEmpty())
           item->setDescriptionText("<qt>"+properties+"</qt>");

        KerryLabel *headerNameLabel = new KerryLabel(item);
        headerNameLabel->setText(name.isEmpty() ? i18n("No Name Known") : name);
        headerNameLabel->setURL(item->uri());
        item->insertHeaderWidget(0,headerNameLabel);
        connect(headerNameLabel, SIGNAL(leftClickedURL(const QString&)), SLOT(slotOpenEvolution(const QString&)));

        QLabel *headerLabel = new QLabel(item);
        headerLabel->setText(i18n(","));
        item->insertHeaderWidget(1,headerLabel);

        KerryLabel *headerEmailLabel = new KerryLabel(item);
        headerEmailLabel->setText(email);
        headerEmailLabel->setURL(email);
        item->insertHeaderWidget(2,headerEmailLabel);
        connect(headerEmailLabel, SIGNAL(leftClickedURL(const QString&)), SLOT(slotMailTo(const QString&)));
      }
      else if ((*(result->uri)).startsWith("kabc:/")) {
        item->setIcon("kaddressbook");

        QStringList _properties(result->properties);
        QString name = takeProperty("vCard:FN",_properties);
        QString email = takeProperty("vCard:EMAIL",_properties);

        QString properties;
        BeagleSearch::PropertyList::iterator it;
        for ( it = _properties.begin(); it != _properties.end(); ++it )
                properties=properties+(*it);

        if (!m_addressBook)
           m_addressBook = KABC::StdAddressBook::self( false );

        KABC::Addressee addresseee = m_addressBook->findByUid( item->uri().mid(8,10) );
        QString number;

        number = addresseee.phoneNumber(KABC::PhoneNumber::Work).number();
        if (!number.isEmpty())
           properties=properties+i18n("Business phone: %1").arg(number)+"<br>";

        number = addresseee.phoneNumber(KABC::PhoneNumber::Cell).number();
        if (!number.isEmpty())
           properties=properties+i18n("Mobile phone: %1").arg(number)+"<br>";

        number = addresseee.phoneNumber(KABC::PhoneNumber::Home).number();
        if (!number.isEmpty())
           properties=properties+i18n("Home phone: %1").arg(number)+"<br>";

        if (!properties.isEmpty())
           item->setDescriptionText("<qt>"+properties+"</qt>");

        KerryLabel *headerNameLabel = new KerryLabel(item);
        headerNameLabel->setText(name.isEmpty() ? i18n("No Name Known") : name);
        headerNameLabel->setURL(item->uri());
        item->insertHeaderWidget(0,headerNameLabel);
        connect(headerNameLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));

        if (!email.isEmpty()) {
        	QLabel *headerLabel = new QLabel(item);
        	headerLabel->setText(i18n(","));
        	item->insertHeaderWidget(1,headerLabel);
		
        	KerryLabel *headerEmailLabel = new KerryLabel(item);
        	headerEmailLabel->setText(email);
        	headerEmailLabel->setURL(email);
        	item->insertHeaderWidget(2,headerEmailLabel);
        	connect(headerEmailLabel, SIGNAL(leftClickedURL(const QString&)), SLOT(slotMailTo(const QString&)));
        }
      }
      else if (*(result->mime_type)=="message/rfc822" || (*(result->uri)).startsWith("email:/") ) {
        item->setIcon("mail_generic");

        QStringList _properties(result->properties);
        QString subject = takeProperty("dc:title",_properties);
        QString from = takeProperty("fixme:from",_properties);
        QString received = takeProperty("fixme:date",_properties);

        QDateTime received_datetime;
        received_datetime = datetimeFromString(received);
        if (!received.isEmpty())
          item->setDescriptionText("<qt>"+i18n("Received: %1").arg(KGlobal::locale()->formatDateTime(received_datetime,false))+"</qt>");

        QString properties;
        BeagleSearch::PropertyList::iterator it;
        for ( it = _properties.begin(); it != _properties.end(); ++it )
                properties=properties+(*it);

        if (result->snippet) {
          if (!properties.isEmpty())
             properties=properties+"<br>";
          properties=properties+*(result->snippet);
        }

        if (!properties.isEmpty())
            item->setPropertiesText("<qt>"+properties+"</qt>");

        KerryLabel *headerSubjectLabel = new KerryLabel(item);
        headerSubjectLabel->setText(subject.isEmpty() ? i18n("No Subject") : subject);
        item->insertHeaderWidget(0,headerSubjectLabel);
        headerSubjectLabel->setURL(*(result->uri));
        connect(headerSubjectLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));

        QLabel *headerLabel = new QLabel(item);
        headerLabel->setText(i18n("From"));
        headerLabel->setAlignment(headerLabel->alignment() | Qt::SingleLine);
        item->insertHeaderWidget(1,headerLabel);

        KerryLabel *headerFromLabel = new KerryLabel(item);
        headerFromLabel->setText(from.isEmpty() ? i18n("Unknown Person") : from);
        headerFromLabel->setURL(from);
        item->insertHeaderWidget(2,headerFromLabel);
        connect(headerFromLabel, SIGNAL(leftClickedURL(const QString&)), SLOT(slotMailTo(const QString&)));
      }
      else if (*(result->mime_type)=="application/x-desktop") {
        KFileItem *fileitem=new KFileItem(*(result->uri),*(result->mime_type),KFileItem::Unknown);
        item->setIcon(fileitem->iconName());
        if (!item->isCollapsed() && canPreview(fileitem))
            previewItems.append(fileitem);
        else
            delete fileitem;

        KDesktopFile desktopfile(url.path(),true);
        if (!desktopfile.readGenericName().isNull()) {
	    item->setDescriptionText("<qt>"+desktopfile.readGenericName()+"</qt>");
        }

        QLabel *headerLabel = new QLabel(item);
        headerLabel->setText(i18n("Application:"));
        headerLabel->setAlignment(headerLabel->alignment() | Qt::SingleLine);
        item->insertHeaderWidget(0,headerLabel);

        KerryLabel *headerFileLabel = new KerryLabel(item);
        QStringList _properties(result->properties);
        QString title = takeProperty("fixme:Name",_properties);
        headerFileLabel->setText(title.isEmpty() ? desktopfile.readName() : title);
        headerFileLabel->setURL(*(result->uri));
        item->insertHeaderWidget(1,headerFileLabel);
        connect(headerFileLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
      }
      else if (*(result->mime_type)=="beagle/x-kopete-log" || *(result->mime_type)=="beagle/x-gaim-log") {
        if (*(result->mime_type)=="beagle/x-kopete-log")
          item->setIcon("kopete");
        else
          item->setIcon("gaim");

        QDateTime datetime;
        datetime.setTime_t(result->last_index_time);
        item->setDescriptionText("<qt>"+i18n("Date: %1").arg(KGlobal::locale()->formatDateTime(datetime,false)+"</qt>"));

        if (result->snippet)
            item->setPropertiesText("<qt>"+*(result->snippet)+ "</qt>");

        KerryLabel *headerFileLabel = new KerryLabel(item);
        headerFileLabel->setURL(*(result->uri));
        QStringList _properties(result->properties);
        QString person = takeProperty("fixme:speakingto",_properties);
        headerFileLabel->setText(i18n("Conversation With %1").arg(person.isEmpty() ? i18n("Unknown Person") : person));
        item->insertHeaderWidget(0,headerFileLabel);
        connect(headerFileLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
      }
      else if (result->tilegroup == BeagleSearch::Packages) {
        QString icon;
        if (*(result->mime_type)=="application/x-rpm")
	    icon = "rpm";
	else if (*(result->mime_type)=="application/x-deb")
	    icon = "deb";
	else
	    icon = "kuroo";
        item->setIcon(icon);
    
        KerryLabel *headerFileLabel = new KerryLabel(item);
        headerFileLabel->setText(url.fileName());
        headerFileLabel->setTipText(url.prettyURL());
        headerFileLabel->setUseTips();
        headerFileLabel->setURL(*(result->uri));
        item->insertHeaderWidget(0,headerFileLabel);
        connect(headerFileLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
    
        QStringList _properties(result->properties);
        QString homepage = takeProperty("dc:source",_properties);
	if (!homepage.isEmpty())
	{
		QLabel *headerLabel = new QLabel(item);
		headerLabel->setText(i18n("From"));
		headerLabel->setAlignment(headerLabel->alignment() | Qt::SingleLine);
		item->insertHeaderWidget(1,headerLabel);
    
		KerryLabel *headerFileLabel = new KerryLabel(item);
		headerFileLabel->setText(homepage);
		headerFileLabel->setAlignment(headerFileLabel->alignment() | Qt::SingleLine);
		headerFileLabel->setURL(homepage);
		item->insertHeaderWidget(2,headerFileLabel);
		connect(headerFileLabel, SIGNAL(leftClickedURL(const QString&)), SLOT(slotOpenURL(const QString&)));
	}

        item->setDescriptionText(takeProperty("dc:subject", _properties));
    
        QString properties;
	QString install_time_str = takeProperty("fixme:install_time", _properties);
	if (!install_time_str.isEmpty())
	{
            properties=properties+i18n("Installed on: %1").arg(KGlobal::locale()->formatDateTime(datetimeFromString(install_time_str),false))+"<br>";
	}

	QString install_size_str = takeProperty("fixme:contents_byte_count", _properties);
	if (!install_size_str.isEmpty())
	{
            properties=properties+i18n("Installed size: %1").arg(formatBytes(install_size_str))+"<br>";
	}

	QString download_size_str = takeProperty("fixme:size", _properties);
	if (!download_size_str.isEmpty())
	{
            properties=properties+i18n("Download size: %1").arg(formatBytes(download_size_str))+"<br>";
	}
		
        BeagleSearch::PropertyList::iterator it;
        for ( it = _properties.begin(); it != _properties.end(); ++it )
	{
                properties=properties+(*it);
	}

        if (result->snippet) {
          if (!properties.isEmpty())
             properties=properties+"<br>";
          properties=properties+*(result->snippet);
        }
   
        if (!properties.isEmpty())
            item->setPropertiesText("<qt>"+properties+"</qt>");
    

        item->insertHitSpacing(2,10);
        buttonGo = new KURLLabel(item);
        buttonGo->setPixmap(SmallIcon("kfm"));
        item->insertHitWidget(3,buttonGo);
        connect(buttonGo, SIGNAL(leftClickedURL()), SLOT(slotOpenDir()));
    
        buttonGo = new KURLLabel(item);
        buttonGo->setText(i18n("Reveal in File Manager"));
        item->insertHitWidget(4,buttonGo);
        connect(buttonGo, SIGNAL(leftClickedURL()), SLOT(slotOpenDir()));
      }
      else if ((*(result->mime_type)).startsWith("audio/")) {
        KFileItem *fileitem=new KFileItem(*(result->uri),*(result->mime_type),KFileItem::Unknown);
        item->setIcon(fileitem->iconName());
        if (!item->isCollapsed() && canPreview(fileitem))
            previewItems.append(fileitem);
        else
            delete fileitem;
    
        QStringList _properties(result->properties);
        QString title = takeProperty("fixme:title",_properties);

        KerryLabel *headerFileLabel = new KerryLabel(item);
        if (title.isEmpty())
            headerFileLabel->setText(url.fileName());
        else
            headerFileLabel->setText(title);
        
        headerFileLabel->setTipText(url.prettyURL());
        headerFileLabel->setUseTips();
        headerFileLabel->setURL(*(result->uri));
        item->insertHeaderWidget(0,headerFileLabel);
        connect(headerFileLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
    
        QLabel *headerLabel = new QLabel(item);
        headerLabel->setText(i18n("In Folder"));
        headerLabel->setAlignment(headerLabel->alignment() | Qt::SingleLine);
        item->insertHeaderWidget(1,headerLabel);

        KerryLabel *headerDirLabel = new KerryLabel(item);
        QString directory=url.directory();
        int i = directory.findRev('/');
        if (i>0)
            directory=directory.mid(i+1);
        headerDirLabel->setText(directory);
        headerDirLabel->setURL(url.directory());
        headerDirLabel->setTipText(url.directory());
        headerDirLabel->setUseTips();
        item->insertHeaderWidget(2,headerDirLabel);
        connect(headerDirLabel, SIGNAL(leftClickedURL()), SLOT(slotOpenDir()));

        QDateTime datetime;
        datetime.setTime_t(result->last_index_time);
        title = i18n("Last modified: %1").arg(KGlobal::locale()->formatDateTime(datetime,false));
        item->setDescriptionText("<qt>"+title+"</qt>");

        QString properties;
        BeagleSearch::PropertyList::iterator it;
        for ( it = _properties.begin(); it != _properties.end(); ++it )
                properties=properties+(*it);

        if (result->snippet) {
          if (!properties.isEmpty())
             properties=properties+"<br>";
          properties=properties+*(result->snippet);
        }

        if (!properties.isEmpty())
            item->setPropertiesText("<qt>"+properties+"</qt>");

        item->insertHitSpacing(2,10);
        buttonGo = new KURLLabel(item);
        buttonGo->setPixmap(SmallIcon("kfm"));
        item->insertHitWidget(3,buttonGo);
        connect(buttonGo, SIGNAL(leftClickedURL()), SLOT(slotOpenDir()));

        buttonGo = new KURLLabel(item);
        buttonGo->setText(i18n("Reveal in File Manager"));
        item->insertHitWidget(4,buttonGo);
        connect(buttonGo, SIGNAL(leftClickedURL()), SLOT(slotOpenDir()));
      }
      else {
        KFileItem *fileitem=new KFileItem(*(result->uri),*(result->mime_type),KFileItem::Unknown);
        item->setIcon(fileitem->iconName());
        if (!item->isCollapsed() && canPreview(fileitem))
            previewItems.append(fileitem);
        else
            delete fileitem;
    
        KerryLabel *headerFileLabel = new KerryLabel(item);
        headerFileLabel->setText(url.fileName());
        headerFileLabel->setTipText(url.prettyURL());
        headerFileLabel->setUseTips();
        headerFileLabel->setURL(*(result->uri));
        item->insertHeaderWidget(0,headerFileLabel);
        connect(headerFileLabel, SIGNAL(leftClickedURL()), SLOT(slotOpen()));
    
        QLabel *headerLabel = new QLabel(item);
        headerLabel->setText(i18n("In Folder"));
        headerLabel->setAlignment(headerLabel->alignment() | Qt::SingleLine);
        item->insertHeaderWidget(1,headerLabel);
    
        KerryLabel *headerDirLabel = new KerryLabel(item);
        QString directory=url.directory();
        int i = directory.findRev('/');
        if (i>0)
            directory=directory.mid(i+1);
        headerDirLabel->setText(directory);
        headerDirLabel->setURL(url.directory());
        headerDirLabel->setTipText(url.directory());
        headerDirLabel->setUseTips();
        item->insertHeaderWidget(2,headerDirLabel);
        connect(headerDirLabel, SIGNAL(leftClickedURL()), SLOT(slotOpenDir()));


        QString title;
        if (*(result->mime_type)=="inode/directory") {
            int count = QDir(url.path()).count() -2 ;
            if (count==0)
              title=i18n("Empty");
            else
              title=i18n("Contains 1 item","Contains %n items",count);
        }
        else {
            QDateTime datetime;
            datetime.setTime_t(result->last_index_time);
            title = i18n("Last modified: %1").arg(KGlobal::locale()->formatDateTime(datetime,false));
        }
        item->setDescriptionText("<qt>"+title+"</qt>");
    
        QString properties;
        BeagleSearch::PropertyList::iterator it;
        for ( it = result->properties.begin(); it != result->properties.end(); ++it )
                properties=properties+(*it);

        if (result->snippet) {
          if (!properties.isEmpty())
             properties=properties+"<br>";
          properties=properties+*(result->snippet);
        }
   
        if (!properties.isEmpty())
            item->setPropertiesText("<qt>"+properties+"</qt>");
    

        item->insertHitSpacing(2,10);
        buttonGo = new KURLLabel(item);
        buttonGo->setPixmap(SmallIcon("kfm"));
        item->insertHitWidget(3,buttonGo);
        connect(buttonGo, SIGNAL(leftClickedURL()), SLOT(slotOpenDir()));
    
        buttonGo = new KURLLabel(item);
        buttonGo->setText(i18n("Reveal in File Manager"));
        item->insertHitWidget(4,buttonGo);
        connect(buttonGo, SIGNAL(leftClickedURL()), SLOT(slotOpenDir()));
      }

      item->score->setText(i18n("<p align=\"center\">Score: %1</p>").arg(result->score,0,'f',1));
    
      tableHits->insertItem(item,index);
}

void SearchDlg::itemUncollapsed(HitWidget* item)
{
   KFileItem *fileitem=new KFileItem(item->uri(),item->mimetype(),KFileItem::Unknown);
   if (canPreview(fileitem))
       previewItems.append(fileitem);
   startPreview(previewItems);
}

void SearchDlg::searchHasOutput(BeagleSearch::BeagleResultList &items)
{
    BeagleSearch::BeagleResultList* searchResults = new BeagleSearch::BeagleResultList;

    for (BeagleSearch::BeagleResultList::ConstIterator it = items.begin(); it != items.end(); ++it) {
      BeagleSearch::beagle_result_struct* result = (*it); 
      if ((*(result->uri)).find(encodingRegexp)>0) {
          KURL kurl(*(result->uri));
          *(result->uri)=kurl.url();
      }
      if (!checkUriInResults(*(result->uri)) && !(*(result->uri)).startsWith("kbookmark:/")) {
         result->show_expanded=showBigTiles;
         results.append(result);
         searchResults->append(result);
      }
    }
    displayResults(*searchResults);
}

void SearchDlg::displayResults(BeagleSearch::BeagleResultList &items)
{
    bool rebuildDisplay = false;
    if (displayAmount==1)
      tableHits->setUpdatesEnabled(false);
    for (BeagleSearch::BeagleResultList::ConstIterator it = items.begin(); it != items.end(); ++it) {
        BeagleSearch::beagle_result_struct *result = *it;

        bool show = false;
        switch (showMode)
        {
	  case Everywhere:
		show = true;
		break;
	  case Applications:
                show = ( result->tilegroup == BeagleSearch::Application );
		break;
          case Contacts:
		show = ( result->tilegroup == BeagleSearch::Contact );
		break;
          case Documents:
		show = ( result->tilegroup == BeagleSearch::Documents );
		break;
          case Conversations:
		show = ( result->tilegroup == BeagleSearch::Conversations );
		break;
          case Images:
		show = ( result->tilegroup == BeagleSearch::Image );
		break;
          case Media:
		show = ( result->tilegroup == BeagleSearch::Audio || result->tilegroup == BeagleSearch::Video);
		break;
          case Website:
		show = ( result->tilegroup == BeagleSearch::Website);
		break;
          case FilePathName:
		show = current_query.matches(*(result->uri));
          default:
		break;
        }

        if (!show)
           continue;

        QDateTime datetime;
        datetime.setTime_t(result->last_index_time);
        QDate date = datetime.date();

	show = false;
        switch (dateRange)
        {
           case AnyDate:
		show = true;
		break;
           case Today:
		show = ( datetime.date()==QDate::currentDate() );
		break;
           case SinceYesterday:
		show = ( datetime.date()>QDate::currentDate().addDays(-1) );
		break;
           case ThisWeek:
		show = ( datetime.date().year()==QDate::currentDate().year() &&
			datetime.date().weekNumber()==QDate::currentDate().weekNumber() );
		break;
           case ThisMonth:
		show = ( datetime.date().year()==QDate::currentDate().year() &&
			datetime.date().month()==QDate::currentDate().month() );
                break;
           case ThisYear:
		show = ( datetime.date().year()==QDate::currentDate().year() );
		break;
	   default:
                break;
        }
        if (!show)
           continue;

        int i = 0;
        for (BeagleSearch::BeagleResultList::ConstIterator result_it = displayed_results.begin(); result_it != displayed_results.end(); ++result_it) {
           bool foundplace = false;
           switch (currentSortOrder)
           {
                case Type:
                        if ( result->tilegroup < (*result_it)->tilegroup || 
                            (result->tilegroup == (*result_it)->tilegroup &&                           (*result->mime_type) < (*(*result_it)->mime_type)) ||
                            (result->tilegroup == (*result_it)->tilegroup &&                           (*result->mime_type) == (*(*result_it)->mime_type) && result->last_index_time >= (*result_it)->last_index_time) ) 
                                foundplace = true;
                        break;
		case Name:
           		if (KURL(*(result->uri)).fileName().lower() < KURL(*((*(result_it))->uri)).fileName().lower())
              			foundplace = true;
			break;
		case Modified:
           		if (result->last_index_time >= (*result_it)->last_index_time)
              			foundplace = true;
			break;
		default:
           		if (result->score >= (*result_it)->score)
              			foundplace = true;
			break;
           }
           if (foundplace)
		break;
           ++i;
        }
        if (displayed_results.count()==0)
          tableHits->clear();
        displayed_results.insert(i, result);
        if (displayAmount==1)
          insertResult(result,i);
        else if (i<=displayOffset+displayAmount)
          rebuildDisplay = true;
    }
    if (displayAmount==1)
      tableHits->setUpdatesEnabled(true);
    else if (rebuildDisplay) {
      tableHits->setUpdatesEnabled(false);
      fillTableHits();
      tableHits->setUpdatesEnabled(true);
    }
    updateStatus();
}

void SearchDlg::slotOpen()
{
    HitWidget* item = static_cast<HitWidget*>(sender()->parent());
    if (item) {
      QString mimetype = item->mimetype();
      if (mimetype=="beagle/x-kopete-log" || mimetype=="beagle/x-gaim-log") {
        KProcess *proc = new KProcess;
        *proc << "beagle-imlogviewer";
        KURL kuri = KURL(item->uri());
        QString uri = kuri.path();
        if (mimetype=="beagle/x-kopete-log")
          *proc << "--client" << "kopete" << "--highlight-search" << current_query.get() << uri;
        else
          *proc << "--client" << "gaim" << "--highlight-search" << current_query.get() << uri;
        if (!proc->start()) {
//        KMessageBox::error(0,i18n("Could not start instant message log viewer."));
          if (mimetype=="beagle/x-kopete-log")
            KRun::runURL(uri, "text/plain", false, true);
          else
            KRun::runURL(uri, "text/html", false, true);
          return;
        }
      }
      else if (item->uri().startsWith("calendar:/") || item->uri().startsWith("contacts:/") 
            || item->uri().startsWith("email:/"))
      {
        slotOpenEvolution(item->uri());
      }
      else if (item->uri().startsWith("mailbox:/") && item->uri().find("thunderbird")>0)
      {
        slotOpenThunderbird(item->uri());
      }
      else if (item->uri().startsWith("kabc:/") )
      {
        slotOpenKAddressBook(item->uri().mid(8,10));
      }
      else if (item->uri().startsWith("knotes:/") )
      {
        slotOpenKNotes(item->uri().mid(9,22));
      }
      else if (item->uri().startsWith("note:/")) {
        KProcess *proc = new KProcess;
        *proc << "tomboy";
        *proc << "--open-note" << item->uri() << "--highligh-search" << "\""+current_query.get()+"\"";
        if (!proc->start()) {
          KMessageBox::error(0,i18n("Could not start Tomboy."));
          return;
        }
      }
      else {
        if (mimetype=="beagle/x-konq-cache")
          mimetype = "text/html";
        KRun::runURL(item->uri(), mimetype, false, true);
      }
    }
}

void SearchDlg::slotOpenDir()
{
    HitWidget* item = static_cast<HitWidget*>(sender()->parent());
    if (item)
      KRun::runURL(KURL(item->uri()).directory(), "inode/directory", false, true);
}

void SearchDlg::slotMailTo(const QString &address)
{
    kapp->invokeMailer(address, QString::null);
}

void SearchDlg::slotOpenEvolution(const QString &address)
{
    KProcess *proc = new KProcess;
    *proc << "evolution";
    *proc << address;
    if (!proc->start()) {
      KMessageBox::error(0,i18n("Could not start Evolution."));
      return;
    }
}

void SearchDlg::slotOpenThunderbird(const QString &address)
{
    KProcess *proc = new KProcess;
    *proc << "thunderbird";
    *proc << "-mail";
    *proc << address;
    if (!proc->start()) {
      KMessageBox::error(0,i18n("Could not start Thunderbird."));
      return;
    }
}

void SearchDlg::slotOpenKAddressBook(const QString &uid)
{
    KProcess *proc = new KProcess;
    *proc << "kaddressbook";
    *proc << "--uid" << uid;
    if (!proc->start()) {
      KMessageBox::error(0,i18n("Could not start KAddressBook."));
      return;
    }
}

void SearchDlg::slotOpenKNotes(const QString &noteid)
{
    if (ensureServiceRunning("knotes")) {
       QByteArray data;
       QDataStream arg(data, IO_WriteOnly);
       arg << noteid;

       kapp->dcopClient()->send("knotes","KNotesIface","showNote(QString)", data);
    }
}

void SearchDlg::slotOpenURL(const QString& url)
{
    (void) new KRun( url, parentWidget());
}

QString SearchDlg::formatBytes(QString& bytesStr) const
{
        static double kilobyte = 1024.0;
        static double megabyte = 1024.0*1024.0;
	double bytes = KGlobal::locale()->readNumber(bytesStr);
        if (bytes < megabyte)
                return KGlobal::locale()->formatNumber(bytes/kilobyte, 2) + "K";
        return KGlobal::locale()->formatNumber(bytes/megabyte, 2) + "M";
}

void SearchDlg::slotStartBeagle()
{
    beagleJustStarted = true;
    if (cb_beagleStart->isChecked()) {
      KConfig *config = KGlobal::config();
      config->setGroup("Beagle");
      config->writeEntry("AutoStart",true);
      config->sync();
    }

    KProcess *proc = new KProcess;
    *proc << "beagled";
    *proc << "--indexing-delay 2";
    if (!proc->start()) {
      KMessageBox::error(0,i18n("Could not start Beagle daemon."));
      return;
    }
    slotClear();
    QTimer::singleShot(5000, this, SLOT(search()));
}

void SearchDlg::searchLostOutput(BeagleSearch::VanishedURIList &items)
{
    bool rebuildDisplay = false;
    for (BeagleSearch::VanishedURIList::ConstIterator it = items.begin(); it != items.end(); ++it) {
      int i;
      for(i = 0; i < (int)displayed_results.count(); ++i)
      {
         BeagleSearch::beagle_result_struct *result = displayed_results.at(i);
         if (*(result->uri)==(*it)) {
           displayed_results.remove(i);
           if (displayed_results.count()==0)
              searchFinished();
           else if (displayAmount==1)
              tableHits->removeItem(i);
           else if (i<=displayOffset+displayAmount) {
             rebuildDisplay = true;
             if (displayOffset>=(int)displayed_results.count())
               displayOffset-=displayAmount;
           }
           break;
        }
      }
      for(i = 0; i < (int)results.count(); ++i)
      {
         BeagleSearch::beagle_result_struct *result = results.at(i);
         if (*(result->uri)==(*it)) {
           results.remove(i);
           break;
        }
      }
    }

    if (rebuildDisplay) {
      tableHits->setUpdatesEnabled(false);
      fillTableHits();
      tableHits->setUpdatesEnabled(true);
    }

    updateStatus();
}

void SearchDlg::searchFinished()
{
    m_searchPixmap->setPixmap( BarIcon( "find", 32 ) );

    still_searching = false;

    if (displayed_results.count()) {
       updateStatus();
       return;
    }

    if (editSearch->lineEdit()->text().isEmpty()) {
       showQuickTips();
       return;
    }

    tableHits->clear();
    HitWidget* item = new HitWidget(QString::null, QString::null);
    QLabel *headerLabel = new QLabel(item);
    headerLabel->setText(i18n("No results for \"%1\" were found.").arg(current_query.get()));
    item->insertHeaderWidget(0,headerLabel);

    item->setIcon("messagebox_warning");

    QString text = "<qt>";
    if (showMode)
	text += i18n("- A broader search scope might produce more results.")+"<br>";
    text += i18n("- You should check the spelling of your search words.");
    if (beagleJustStarted) {
        text += "<br>"+i18n("- The Beagle daemon was just started. Please be patient until it finished its indexing.");
        beagleJustStarted = false;
    }
    item->setDescriptionText(text+"</qt>");
    labelStatus->setText("");

    tableHits->insertItem(item);
}

void SearchDlg::searchError(const QString& error)
{
     kdDebug() << "SearchDlg::searchError() " << error << endl;
}

bool SearchDlg::canPreview( KFileItem* item )
{
    if ( !KGlobalSettings::showFilePreview( item->url() ) )
        return false;

    if ( pPreviewMimeTypes == 0L )
        updatePreviewMimeTypes();

    return mimeTypeMatch( item->mimetype(), *( pPreviewMimeTypes ) );
}

void SearchDlg::updatePreviewMimeTypes()
{
    if ( pPreviewMimeTypes == 0L )
        pPreviewMimeTypes = new QStringList;
    else
        pPreviewMimeTypes->clear();

    // Load the list of plugins to determine which mimetypes are supported
    KTrader::OfferList plugins = KTrader::self()->query("ThumbCreator");
    KTrader::OfferList::ConstIterator it;

    for ( it = plugins.begin(); it != plugins.end(); ++it ) {
        QStringList mimeTypes = (*it)->property("MimeTypes").toStringList();
        for (QStringList::ConstIterator mt = mimeTypes.begin(); mt != mimeTypes.end(); ++mt)
           pPreviewMimeTypes->append(*mt);
    }
}

bool SearchDlg::mimeTypeMatch( const QString& mimeType, const QStringList& mimeList ) const
{
    for (QStringList::ConstIterator mt = mimeList.begin(); mt != mimeList.end(); ++mt)
    {
        if ( mimeType == *mt )
            return true;
        // Support for *mt == "image/*"
        QString tmp( mimeType );
        if ( (*mt).endsWith("*") && tmp.replace(QRegExp("/.*"), "/*") == (*mt) )
            return true;
    }
    return false;
}

void SearchDlg::startPreview( const KFileItemList& items )
{
    stopPreview(); // just in case
    int iconSize = KGlobal::iconLoader()->currentSize( KIcon::Desktop );

    pPreviewJob = KIO::filePreview( items, KIcon::SizeHuge, KIcon::SizeHuge, iconSize,
        true /*m_pSettings->textPreviewIconTransparency()*/, true /* scale */,
        true /* save */, 0); //&(d->previewSettings) );
    connect( pPreviewJob, SIGNAL( gotPreview( const KFileItem *, const QPixmap & ) ),
             this, SLOT( slotPreview( const KFileItem *, const QPixmap & ) ) );
    connect( pPreviewJob, SIGNAL( result( KIO::Job * ) ),
             this, SLOT( slotPreviewResult() ) );
}


void SearchDlg::stopPreview()
{
    if (pPreviewJob)
    {
        pPreviewJob->kill();
        pPreviewJob = 0;
    }
}

void SearchDlg::slotPreview(const KFileItem *item, const QPixmap &pix)
{
  for(uint i=0; i < tableHits->count(); ++i)
  {
    HitWidget* w = static_cast<HitWidget*>(tableHits->item(i));
    if (w && KURL(w->uri()).prettyURL() == item->url().prettyURL() ) {
      w->icon->setPixmap(pix);
      break;
    }
  }
}

void SearchDlg::slotPreviewResult()
{
    pPreviewJob = 0;
    previewItems.clear();
}

void SearchDlg::keyPressEvent(QKeyEvent *e)
{
    if (e->key()==Key_PageDown && displayAmount!=1) {
      if (e->state()==ControlButton) {
        if (displayOffset+displayAmount>=(int)displayed_results.count())
          return;

        displayOffset=((displayed_results.count() -1) / displayAmount) * displayAmount;
        tableHits->setUpdatesEnabled(false);
        fillTableHits();
        tableHits->setUpdatesEnabled(true);
        updateStatus();
      }
      else
        slotNext();
    }
    else if (e->key()==Key_PageUp && displayAmount!=1) {
      if (e->state()==ControlButton) {
        if (displayOffset==0)
          return;

        displayOffset=0;
        tableHits->setUpdatesEnabled(false);
        fillTableHits();
        tableHits->setUpdatesEnabled(true);
        updateStatus();
      }
      else
        slotPrevious();
    }
    else
      HitsLayout::keyPressEvent(e);
}

void SearchDlg::showEvent(QShowEvent *e )
{
    HitsLayout::showEvent( e );
    if (pending_showQuickTips) {
      showQuickTips();
      pending_showQuickTips=false;
    }
}

void SearchDlg::slotContextMenu( int /*row*/, int /*col*/, const QPoint & pos )
{
    KPopupMenu *popup = new KPopupMenu(this);
    popup->insertTitle(i18n("Collapse") + " / " + i18n("Expand"));
    popup->insertItem(i18n("Collapse All"), 1);
    popup->insertItem(i18n("Expand All"), 2);
    int selected = popup->exec(pos);
    if (selected==1 || selected==2)
      for(uint i=0; i < tableHits->count(); ++i) {
         HitWidget* w = static_cast<HitWidget*>(tableHits->item(i));
         w->setCollapsed(selected==1);
      }
    delete popup;
}


void SearchDlg::showSearchDialog()
{
    show();
    KWin::setOnDesktop( winId(), KWin::currentDesktop() );
    kapp->updateUserTimestamp();
    KWin::forceActiveWindow( winId() );
    editSearch->setFocus();
    editSearch->lineEdit()->selectAll();
}

void SearchDlg::configChanged()
{
    KConfig *config = KGlobal::config();
    config->reparseConfiguration();
    config->setGroup("General");

    showBigTiles=config->readBoolEntry("ShowBigTiles",false);
    setDisplayAmount(config->readNumEntry("DisplayAmount", 20));
    setSortOrder(config->readNumEntry("DefaultSortOrder",0));
}

void SearchDlg::searchProgramList(QString relPath)
{
    KServiceGroup::Ptr group = KServiceGroup::group(relPath);
    if (!group || !group->isValid())
      return;

    KServiceGroup::List list = group->entries();
    if (list.isEmpty())
      return;

    KServiceGroup::List::ConstIterator it = list.begin();
    for(; it != list.end(); ++it) {
	KSycocaEntry *e = *it;

	if(e != 0) {
		if(e->isType(KST_KServiceGroup)) {
			KServiceGroup::Ptr g(static_cast<KServiceGroup *>(e));
			if(!g->noDisplay())
				searchProgramList(g->relPath());
		} else if(e->isType(KST_KService)) {
			KService::Ptr s(static_cast<KService *>(e));
			if(s->type() == "Application" && !s->noDisplay() && !checkUriInResults(locate("apps", s->desktopEntryPath()))) {
				if (!current_query.matches(s->name().lower()+' '+s->genericName()+' '+
                                    s->exec()+' '+s->keywords().join(",")+' '+s->comment()+' ' +
                                    group->caption()+' '+s->categories().join(",")))
					continue;

				BeagleSearch::beagle_result_struct *result = new BeagleSearch::beagle_result_struct;
				result->mime_type=new QString("application/x-desktop");
                                result->tilegroup = BeagleSearch::Application;
				result->uri=new QString("file://"+locate("apps", s->desktopEntryPath()));
				*(result->properties).append("fixme:Name="+s->name());
				result->tilegroup = BeagleSearch::Application;
				result->score = 0.0;
				result->show_expanded=showBigTiles;
				results.append(result);
				new_results.append(result);
			}
		}
	}
    }
}

bool SearchDlg::checkUriInResults(const QString& path) const
{
    for (BeagleSearch::BeagleResultList::ConstIterator it = results.begin(); it != results.end(); ++it) {
        BeagleSearch::beagle_result_struct *result = *it;
        if (*(result->uri) == path)
           return true;
    }
    return false;
}

void SearchDlg::searchBookmarks(KBookmarkGroup group)
{
	KBookmark bookmark = group.first();
	while(!bookmark.isNull()) {
		if (bookmark.isGroup()) {
			searchBookmarks(bookmark.toGroup());
		} else if (!bookmark.isSeparator() && !bookmark.isNull()) {
				if (!current_query.matches(bookmark.fullText()+' '+bookmark.url().url())) {
                                        bookmark = group.next(bookmark);
	    				continue;
                                }

                                if (checkUriInResults(bookmark.url().prettyURL())) {
                                        bookmark = group.next(bookmark);
                                        continue;
                                }

				BeagleSearch::beagle_result_struct *result = new BeagleSearch::beagle_result_struct;
				result->mime_type=new QString("text/html");
				result->uri=new QString(bookmark.url().prettyURL());
				*(result->properties).append("dc:title="+bookmark.fullText());
				result->tilegroup = BeagleSearch::Website;
				result->score = 0.0;
				result->show_expanded=showBigTiles;
				results.append(result);
				new_results.append(result);
		}
		bookmark = group.next(bookmark);
	}
}

void SearchDlg::searchAddressbook()
{
    if (!m_addressBook)
      m_addressBook = KABC::StdAddressBook::self( false );

    KABC::AddressBook::ConstIterator it = m_addressBook->begin();
    while (it!=m_addressBook->end()) {
        if (!current_query.matches((*it).assembledName()+' '+(*it).fullEmail())) {
            it++;
            continue;
        }

        if (checkUriInResults("kabc:///"+(*it).uid())) {
             it++;
             continue;
        }

        QString realName = (*it).realName();
        if (realName.isEmpty())
            realName=(*it).preferredEmail();

	BeagleSearch::beagle_result_struct *result = new BeagleSearch::beagle_result_struct;
	result->mime_type=new QString("text/html");
        result->uri=new QString("kabc:///"+(*it).uid());
	*(result->properties).append("vCard:FN="+realName);
        *(result->properties).append("vCard:EMAIL="+(*it).preferredEmail());
	result->tilegroup = BeagleSearch::Contact;
	result->score = 0.0;
	result->show_expanded=showBigTiles;
	results.append(result);
	new_results.append(result);

       it++;
    }
}


bool SearchDlg::eventFilter(QObject *obj, QEvent *e)
{
#define showSwitchesCount 9
static QLabel* showSwitches[showSwitchesCount] = {showEverything, showApplications, showContacts, showDocuments,
                                   showConversations, showImages, showMedia, showWebPages, showFilePathName};

#define sortSwitchesCount 4
static QLabel* sortSwitches[sortSwitchesCount] = {sortByType, sortByDate, sortByName, sortByRelevance};

#define dateSwitchesCount 6
static QLabel* dateSwitches[dateSwitchesCount] = {showAnyDate, showToday, showSinceYesterday, showThisWeek, 
                                  showThisMonth, showThisYear};

    if (e->type()==QEvent::Enter) {
       QApplication::setOverrideCursor( QCursor(Qt::PointingHandCursor) );
    } else if (e->type()==QEvent::Leave) {
       QApplication::restoreOverrideCursor();
    } else if (e->type()==QEvent::MouseButtonPress) {
       for (int i=0; i<showSwitchesCount; ++i)
	  if (obj ==showSwitches[i]) {
             if (i!=showMode) {
                QFont f= showSwitches[showMode]->font();
                f.setBold(false);
                showSwitches[showMode]->setFont(f);
                showMode = (ScopeType)i;
                f.setBold(true);
                showSwitches[showMode]->setFont(f);
                sortFilterResults();
             }
             return true;
          }

       for (int i=0; i<sortSwitchesCount; ++i)
	  if (obj ==sortSwitches[i]) {
             setSortOrder(i);
             return true;
          }

       for (int i=0; i<dateSwitchesCount; ++i)
	  if (obj ==dateSwitches[i]) {
             if (i!=dateRange) {
                QFont f= dateSwitches[dateRange]->font();
                f.setBold(false);
                dateSwitches[dateRange]->setFont(f);
                dateRange = (DateRange)i;
                f.setBold(true);
                dateSwitches[dateRange]->setFont(f);
                sortFilterResults();
             }
             return true;
          }
    }

    return false;
}

bool SearchDlg::ensureServiceRunning(const QString & name)
{
    QStringList URLs;
    QByteArray data, replyData;
    QCString replyType;
    QDataStream arg(data, IO_WriteOnly);
    arg << name << URLs;

    if ( !kapp->dcopClient()->call( "klauncher", "klauncher", "start_service_by_desktop_name(QString,QStringList)",
                      data, replyType, replyData) ) {
        qWarning( "call to klauncher failed.");
        return false;
    }
    QDataStream reply(replyData, IO_ReadOnly);

    if ( replyType != "serviceResult" )
    {
        qWarning( "unexpected result '%s' from klauncher.", replyType.data());
        return false;
    }
    int result;
    QCString dcopName;
    QString error;
    reply >> result >> dcopName >> error;
    if (result != 0)
    {
        qWarning("Error starting: %s", error.local8Bit().data());
        return false;
    }
    return true;
}

#include "searchdlg.moc"
