/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : Dialog to allow photo properties to be edited
 * 
 * Copyright 2005 by M. Asselstine

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */
 
#include "previewdlg.h"
#include "photopropsdlg.h"
#include "photolistview.h"

#include <qlabel.h>
#include <qimage.h>
#include <kglobal.h>
#include <qregexp.h>
#include <klocale.h>
#include <qcursor.h>
#include <qstring.h>
#include <qpixmap.h>
#include <kguiitem.h>
#include <qlistbox.h>
#include <qstrlist.h>
#include <qlineedit.h>
#include <qtextedit.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qnamespace.h>
#include <qvalidator.h>
#include <kpushbutton.h>
#include <qradiobutton.h>
#include <kapplication.h>
#include <qkeysequence.h>
#include <kstandarddirs.h>


PhotoPropsDlg::PhotoPropsDlg(QWidget *parent, const char *name)
    :PhotoEditDlg(parent, name),
    m_activePhoto(0L)
{
    // setup widgets
    m_public->setChecked( TRUE );
    updateChecks(m_family->isChecked());
    m_pixmap->setScaledContents(false);
    
    m_above->setAccel(Key_PageUp);
    m_above->setGuiItem(KGuiItem(i18n("OK"),"up"));
    m_below->setAccel(Key_PageDown);
    m_below->setGuiItem(KGuiItem(i18n("OK"),"down"));
    
    m_zoom->setGuiItem(KGuiItem("","viewmag"));
    m_rotate->setGuiItem(KGuiItem("","rotate"));
    
    // setup photo size combobox
    m_photoSize->insertItem(i18n("Square 75x75"));
    m_photoSize->insertItem(i18n("Thumb 75x100"));
    m_photoSize->insertItem(i18n("Small 180x240"));
    m_photoSize->insertItem(i18n("Medium 375x500"));
    m_photoSize->insertItem(i18n("Large 768x1024"));
    m_photoSize->insertItem(i18n("Original"));
    
    // disable photo resize if neither JPEG or PNG write support found
    QStrList fmts = QImage::outputFormats();
    if( !fmts.contains("JPEG") && !fmts.contains("PNG") )
        m_photoSize->setEnabled(false);
    
    // create connections
    connect(m_add, SIGNAL(clicked()), this, SLOT(addSelectedTag()));
    connect(m_rotate, SIGNAL(clicked()), this, SLOT(rotatePhoto()));
    connect(m_zoom, SIGNAL(clicked()), this, SLOT(showLargerPreview()));
    connect(m_remove, SIGNAL(clicked()), this, SLOT(removeSelectedTags()));
    connect(m_public, SIGNAL(toggled(bool)), this, SLOT(updatePrivate(bool)));
    connect(m_private, SIGNAL(toggled(bool)), this, SLOT(updatePublic(bool)));
    connect(m_private, SIGNAL(toggled(bool)), this, SLOT(updateChecks(bool)));
    connect(m_tags, SIGNAL(selectionChanged()), this, SLOT(updateRemoveBtn()));
    connect(m_availableTags, SIGNAL(activated(const QString&)), this, SLOT(insertNewTag(const QString&)));
    connect(m_availableTags, SIGNAL(textChanged(const QString&)), this, SLOT(updateAddBtn(const QString&)));
}


/*!
    \fn PhotoPropsDlg::~PhotoPropsDlg()
 */
PhotoPropsDlg::~PhotoPropsDlg()
{
}


/*!
    \fn PhotoPropsDlg::setPixmap(const QPixmap &pm)
 */
void PhotoPropsDlg::setPixmap(const QPixmap &pm)
{
    m_pixmap->setPixmap(pm);
}


/*!
    \fn PhotoPropsDlg::setFriends(bool friends)
 */
void PhotoPropsDlg::setFriends(bool friends)
{
    m_friends->setChecked(friends);
}


/*!
    \fn PhotoPropsDlg::setFamily(bool family)
 */
void PhotoPropsDlg::setFamily(bool family)
{
    m_family->setChecked(family);
}


/*!
    \fn PhotoPropsDlg::setPublic(bool pub)
 */
void PhotoPropsDlg::setPublic(bool pub)
{
    m_public->setChecked(pub);
    m_private->setChecked(!pub);
}


/*!
    \fn PhotoPropsDlg::setNeitherPublicOrPrivate()
 */
void PhotoPropsDlg::setNeitherPublicOrPrivate()
{
    // block signals temporarily to prevent the
    // automatic setting of the m_private toggle
    m_public->blockSignals(true);
    m_public->setChecked(false);
    m_public->blockSignals(false);
    
    m_private->setChecked(false);
}


/*!
    \fn PhotoPropsDlg::setDescription(const QString &)
 */
void PhotoPropsDlg::setDescription(const QString &str)
{
    m_desc->setText(str);
}


/*!
    \fn PhotoPropsDlg::setTitle(const QString &)
 */
void PhotoPropsDlg::setTitle(const QString &str)
{
    m_title->setText(str);
}


/*!
    \fn PhotoPropsDlg::setTags(const QStringList &lst)
 */
void PhotoPropsDlg::setTags(const QStringList &lst)
{
    QRegExp rx("^\".*\"$");
    QStringList::ConstIterator it;
    
    // clear the tags
    m_tags->clear();
    
    // Add each tag, remove surounding quotes if found
    for( it = lst.constBegin(); it != lst.constEnd(); ++it )
    {
        if( rx.search(*it) == 0 )
            m_tags->insertItem((*it).mid(1,(*it).length()-2));
        else
            m_tags->insertItem(*it);
    }
}


/*!
    \fn PhotoPropsDlg::setPhotoSize(const QString &)
 */
void PhotoPropsDlg::setPhotoSize(const QString &size)
{
    if( size.isEmpty() )
        m_photoSize->insertItem("");
        
    m_photoSize->setCurrentText(size);
}


/*!
    \fn PhotoPropsDlg::isFamily()
 */
bool PhotoPropsDlg::isFamily() const
{
    return m_family->isChecked();
}


/*!
    \fn PhotoPropsDlg::isFriends()
 */
bool PhotoPropsDlg::isFriends() const
{
    return m_friends->isChecked();
}


/*!
    \fn PhotoPropsDlg::isPublic()
 */
bool PhotoPropsDlg::isPublic() const
{
    return m_public->isChecked();
}


/*!
    \fn PhotoPropsDlg::isPrivate()
 */
bool PhotoPropsDlg::isPrivate() const
{
    return m_private->isChecked();
}


/*!
    \fn PhotoPropsDlg::description()
 */
QString PhotoPropsDlg::description() const
{
    return m_desc->text();
}


/*!
    \fn PhotoPropsDlg::title()
 */
QString PhotoPropsDlg::title() const
{
    return m_title->text();
}


/*!
    \fn PhotoPropsDlg::tags()
 */
QStringList PhotoPropsDlg::tags() const
{
    QStringList lst;
    
    
    // Add each tag from the tags list
    for( uint i = 0; i < m_tags->count(); ++i )
    {
        // quote any multi word tags
        if( m_tags->text(i).contains(QRegExp("\\s+")) )
            lst += QString("\"" + m_tags->text(i) + "\"");
        else
            lst += m_tags->text(i);
    }        
    return lst;
}


/*!
    \fn PhotoPropsDlg::photoSize()
 */
QString PhotoPropsDlg::photoSize() const
{
    return m_photoSize->currentText();
}


/*!
    \fn PhotoPropsDlg::updatePublic( bool )
 */
void PhotoPropsDlg::updatePublic( bool priv )
{
    m_public->setChecked( !priv );
}


/*!
    \fn PhotoPropsDlg::updatePrivate( bool )
 */
void PhotoPropsDlg::updatePrivate( bool pub )
{
    m_private->setChecked( !pub );
}


/*!
    \fn PhotoPropsDlg::updateChecks( bool )
 */
void PhotoPropsDlg::updateChecks( bool priv )
{
    m_family->setEnabled( priv );
    m_friends->setEnabled( priv );
}


/*!
    \fn PhotoPropsDlg::insertNewTag( const QString& )
 */
void PhotoPropsDlg::insertNewTag( const QString &str )
{
    m_tags->insertItem( str );
    m_availableTags->setCurrentText( "" );
}


/*!
    \fn PhotoPropsDlg::updateAddBtn( const QString & )
 */
void PhotoPropsDlg::updateAddBtn( const QString &str )
{
    // is there any text in the combo
    if( str.length( ) > 0 )
        return m_add->setEnabled( TRUE );   // yes, enable add button
    else
        return m_add->setEnabled( FALSE );  // no, disable it
}


/*!
    \fn PhotoPropsDlg::updateRemoveBtn( )
 */
void PhotoPropsDlg::updateRemoveBtn( )
{
    QListBoxItem *item = m_tags->firstItem( );
    
    // check if any items are selected
    while( item != 0L )
    {
        // is this one selected, we only need one
        if( item->isSelected( ) )
            return m_remove->setEnabled( TRUE );
            
        // check the next item
        item = item->next( );
    }
    // none selected
    return m_remove->setEnabled( FALSE );
}


/*!
    \fn PhotoPropsDlg::removeSelectedTags( )
 */
void PhotoPropsDlg::removeSelectedTags( )
{
    QListBoxItem *next;
    QListBoxItem *item = m_tags->firstItem( );
    
    // remove any selected tags
    while( item != 0L )
    {
        // store next item
        next = item->next( );
        
        // remove if selected
        if( item->isSelected( ) )
            delete item;
            
        // move on to next item
        item = next;
    }
}


/*!
    \fn PhotoPropsDlg::addSelectedTag()
 */
void PhotoPropsDlg::addSelectedTag()
{
    bool already;
    QString tagname = m_availableTags->currentText( );
    
    // add the tag to m_tags
    already = FALSE;
    for( int i = (int)m_tags->count( ) - 1; i >= 0; --i )
    {
        if( tagname == m_tags->text( i ) )
        {
            already = TRUE;
            break;
        }
    }
    if( !already )
        m_tags->insertItem( tagname );
    
    // add the tag to the available tags combobox
    already = FALSE;
    for( int i = m_availableTags->count( ) - 1; i >= 0; --i )
    {
        if( tagname == m_availableTags->text( i ) )
        {
            already = TRUE;
            break;
        }
    }
    if( !already )
        m_availableTags->insertItem( tagname );

    // clear out the combobox        
    m_availableTags->setCurrentText( "" );
}


/*!
    \fn PhotoPropsDlg::setAvailableTags( const QStringList & )
 */
void PhotoPropsDlg::setAvailableTags( const QStringList &lst )
{
    m_availableTags->insertStringList( lst );
    m_availableTags->setCurrentText( "" );
}


/*!
    \fn PhotoPropsDlg::persistentTags( ) const
 */
QStringList PhotoPropsDlg::availableTags( ) const
{
    QStringList lst;
   
    for( int i =  0; i < m_availableTags->count( ); ++i )
        lst += m_availableTags->text( i );
        
    return lst;
}


/*!
    \fn PhotoPropsDlg::editPhotoBatch()
 */
void PhotoPropsDlg::editPhotoBatch()
{
    KStandardDirs *dirs;
    
    // Setup the dialog for batch edit mode
    dirs = KApplication::kApplication()->dirs();
    setPixmap( QPixmap( dirs->findResource( "data", "kflickr/batchmode.png" ) ) );
    setPhotoSize("");
    setNeitherPublicOrPrivate();
    
    // disable next, previous, zoom and rotate buttons on the dialog
    m_above->setEnabled(false);
    m_below->setEnabled(false);
    m_zoom->setEnabled(false);
    m_rotate->setEnabled(false);
}


/*!
    \fn PhotoPropsDlg::updateActivePhotoInfo()
 */
void PhotoPropsDlg::updateActivePhotoInfo()
{
    // ensure valid photo item
    if( m_activePhoto == 0L )
        return;
        
    // set the photo's attributes to match dialog widgets
    m_activePhoto->setTitle(title());
    m_activePhoto->setDescription(description());
    m_activePhoto->setPublic(isPublic());
    m_activePhoto->setFamily(isFamily());
    m_activePhoto->setFriends(isFriends());
    m_activePhoto->setTags(tags());
    m_activePhoto->setSize(photoSize());    
}


/*!
    \fn PhotoPropsDlg::editSinglePhoto(PhotoListViewItem *photo)
 */
void PhotoPropsDlg::editSinglePhoto(PhotoListViewItem *photo)
{
    // store pointer to active photo item
    m_activePhoto = photo;
    if( !m_activePhoto->isSelected() )
    {
        m_activePhoto->setSelected(true);
        m_activePhoto->repaint();
    }
    
    // set dialog widgets to match the photo's attributes
    setTitle(photo->title());
    setPixmap(photo->previewImage());
    setDescription(photo->description());
    setPublic(photo->isPublic());
    setFamily(photo->isFamily());
    setFriends(photo->isFriends());
    setTags(photo->tags());
    setPhotoSize(photo->size());
    
    // enable/disable the previous and next buttons appropriately
    photo->itemAbove() == 0L ? m_above->setEnabled(false) : m_above->setEnabled(true);
    photo->itemBelow() == 0L ? m_below->setEnabled(false) : m_below->setEnabled(true);
}


/*!
    \fn PhotoPropsDlg::editPhotoAbove()
 */
void PhotoPropsDlg::editPhotoAbove()
{
    // Since this is also an 'OK' update the current photo then
    // move to the photo above the active one in the list
    updateActivePhotoInfo();
    m_activePhoto->setSelected(false);
    m_activePhoto->repaint();
    
    editSinglePhoto(static_cast<PhotoListViewItem*>(m_activePhoto->itemAbove()));
}


/*!
    \fn PhotoPropsDlg::editPhotoBelow()
 */
void PhotoPropsDlg::editPhotoBelow()
{
    // Since this is also an 'OK' update the current photo then
    // move to the photo above the active one in the list
    updateActivePhotoInfo();
    m_activePhoto->setSelected(false);
    m_activePhoto->repaint();
    
    editSinglePhoto(static_cast<PhotoListViewItem*>(m_activePhoto->itemBelow()));
}


/*!
    \fn PhotoPropsDlg::showLargerPreview()
 */
void PhotoPropsDlg::showLargerPreview()
{
    // Create large preview dialog
    PreviewDlg *dlg = new PreviewDlg(this);
    dlg->displayPhoto(m_activePhoto->url(), m_activePhoto->rotation());
    
    // Give visual feedback
    setCursor(ForbiddenCursor);
    dlg->exec();
    
    // Cleanup
    setCursor(ArrowCursor);
    delete dlg;    
}


/*!
    \fn PhotoPropsDlg::rotatePhoto()
 */
void PhotoPropsDlg::rotatePhoto()
{
    m_activePhoto->rotatePhoto();
    setPixmap(m_activePhoto->previewImage());
}


#include "photopropsdlg.moc"
