/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : Handle communications with flickr.com
 *
 * Copyright 2005,2007-2008 by M. Asselstine

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */
#include "flickrcomm.h"

#include <QUrl>
#include <QList>
#include <QFile>
#include <QImage>
#include <QDebug>
#include <QRegExp>
#include <QBuffer>
#include <QString>
#include <QWMatrix>
#include <QDomNode>
#include <QByteArray>
#include <QDomElement>
#include <QDataStream>
#include <QStringList>
#include <QDomDocument>
#include <QImageReader>

#include <krun.h>
#include <klocale.h>
#include <kcodecs.h>
#include <krandom.h>
#include <kimageio.h>
#include <kmimetype.h>
#include <kapplication.h>

#include "exif.h"
#include "previewmgr.h"

#include <algorithm>

namespace
{
  QVariant getRecordValue(const QSqlRecord& rec, const QString& field)
  {
    return rec.value(rec.indexOf(field));
  }
};

FlickrComm::FlickrComm(QObject *parent)
  : QObject(parent),
    m_APIKey("c0134cf226b1187e3d79e4e1be03d1bf"),
    m_publicKey("f92fb243a918a3d2"),
    m_userToken("")
{
  // variable inits
  m_requests.clear();
  m_incomingData.clear();
  m_MD5Context = new KMD5("");
}

FlickrComm::~FlickrComm()
{
  delete m_MD5Context;
}

QString FlickrComm::generateMD5(const ArgMap &args)
{
  QString str;
  ArgMap::ConstIterator it;

  // create the string from our arguments
  for( it = args.constBegin(); it != args.constEnd(); ++it )
  {
    str += it.key() + it.value();
 }

  // generate MD5, we have to make use of the QCString returned by the call
  // to QString::utf8 in order to properly account for unicode issues
  m_MD5Context->reset();
  m_MD5Context->update(m_publicKey);
  m_MD5Context->update(str.toUtf8());

  return m_MD5Context->hexDigest().data();
}

QString FlickrComm::assembleArgs(const ArgMap &args)
{
  QString result;
  ArgMap::ConstIterator it;

  // create the request URL  XXX=YYY&ZZZ=AAA...
  for( it = args.constBegin(); it != args.constEnd(); ++it )
  {
    if( !result.isEmpty() )
    {
      result += "&";
    }
    result += it.key() + "=" + it.value();
  }
  return result;
}

QString FlickrComm::validateHTTPResponse(const QString &str)
{
  QString err;
  QDomNode node;
  QDomElement root;
  QDomDocument doc("response");

  // pump str into the XML document
  if( !doc.setContent(str) )
  {
    return i18n("Unrecognizable response from Flickr.com");
  }

  // start at document root, first node
  root = doc.documentElement();
  node = root.firstChild();

  // check for fail response
  if( root.attribute("stat", "fail") == "fail")
  {
    // go through each node, should only be one
    while( !node.isNull() )
    {
      // we got our frob, good to go
      if( node.isElement() && node.nodeName() == "err" )
      {
	// get error text
	QDomElement elem = node.toElement();
	err = elem.attribute("msg", i18n("Unknown"));
      }
      node = node.nextSibling();
    }
  }
  return err;
}

void FlickrComm::setUserToken(const QString &str)
{
  m_userToken = str;
}

KIO::TransferJob* FlickrComm::sendRequest(ArgMap &args)
{
  QString url = "http://www.flickr.com/services/rest/?";

  // setup request specific arguments
  args["api_key"] = m_APIKey;
  args.insert("api_sig", generateMD5(args));

  // formulate complete URL
  url += assembleArgs(args);

  // send request
  KIO::TransferJob *job = KIO::http_post(url, QByteArray(0), KIO::HideProgressInfo);
  job->addMetaData("content-type", "Content-Type: application/x-www-form-urlencoded");

  connect(job,SIGNAL(result(KJob*)),this,SLOT(jobResult(KJob*)));
  connect(job,SIGNAL(data(KIO::Job*,const QByteArray&)),SLOT(jobData(KIO::Job*,const QByteArray&)));

  return job;
}

void FlickrComm::sendFROBRequest()
{
  ArgMap args;

  // setup args
  args["method"] = "flickr.auth.getFrob";

  // remember request type and make request
  KIO::TransferJob* job = sendRequest(args);
  m_requests[job] = FROB_REQ;
}

void FlickrComm::sendTokenRequest(const QString &frob)
{
  ArgMap args;

  // setup args
  args["method"]  = "flickr.auth.getToken";
  args["frob"]    = frob;

  // remember request type and make request
  KIO::TransferJob* job = sendRequest(args);
  m_requests[job] = TOKEN_REQ;
}

void FlickrComm::sendPhotosetsRequest(const QString &token, const QString &user)
{
  ArgMap args;

  // setup args
  args["method"]      = "flickr.photosets.getList";
  args["user_id"]     = user;
  args["auth_token"]  = token;

  // remember request type and make request
  KIO::TransferJob* job = sendRequest(args);
  m_requests[job] = PHOTOSET_REQ;
}

void FlickrComm::sendLicensesRequest()
{
  ArgMap args;

  // setup args
  args["method"]      = "flickr.photos.licenses.getInfo";

  // remember request type and make request
  KIO::TransferJob* job = sendRequest(args);
  m_requests[job] = LICENSES_REQ;
}

void FlickrComm::sendTagsRequest(const QString &token, const QString &user)
{
  ArgMap args;

  // setup args
  args["method"]      = "flickr.tags.getListUser";
  args["user_id"]     = user;
  args["auth_token"]  = token;

  // remember request type and make request
  KIO::TransferJob* job = sendRequest(args);
  m_requests[job] = TAGS_REQ;
}

void FlickrComm::sendUpStatusRequest( const QString &token )
{
  ArgMap args;

  // setup args
  args["method"]        = "flickr.people.getUploadStatus";
  args["auth_token"]    = token;

  // remember request type and make request
  KIO::TransferJob* job = sendRequest(args);
  m_requests[job] = STATUS_REQ;
}

void FlickrComm::doWebAuthentication(const QString &frob)
{
  ArgMap args;
  QString addr;

  // setup browser address string
  addr = "http://flickr.com/services/auth/";
  args["api_key"] = "c0134cf226b1187e3d79e4e1be03d1bf";
  args["perms"]   = "write";
  args["frob"]    = frob;
  args.insert("api_sig", generateMD5(args));

  addr += "?" + assembleArgs(args);

  // Open the default browser with the address
  new KRun(KUrl(addr), kapp->activeWindow());
}

KIO::TransferJob* FlickrComm::sendPhoto(const QString &token, const QSqlRecord& rec)
{
  ArgMap args;
  QByteArray array;
  QString dd = "--";
  QString crlf = "\r\n";
  QBuffer buffer(&array);
  QString url = "http://www.flickr.com/services/upload/?";
  QString boundary = QString("-----") + KRandom::randomString(20);


  // setup form
  buffer.open(QIODevice::WriteOnly);
  QTextStream textStream(&buffer);
  textStream.setCodec("UTF-8");

  // setup request specific arguments
  args["api_key"] = m_APIKey;
  textStream << dd << boundary << crlf;
  textStream << "Content-Disposition: form-data; name=\"api_key\"" << crlf << crlf;
  textStream << m_APIKey << crlf;

  args["auth_token"] = token;
  textStream << dd << boundary << crlf;
  textStream << "Content-Disposition: form-data; name=\"auth_token\"" << crlf << crlf;
  textStream << token << crlf;

  if( !getRecordValue(rec, "title").toString().isEmpty() )
  {
    args["title"] = getRecordValue(rec, "title").toString();
    textStream << dd << boundary << crlf;
    textStream << "Content-Disposition: form-data; name=\"title\"" << crlf << crlf;
    textStream << getRecordValue(rec, "title").toString() << crlf;
  }

  if( !getRecordValue(rec, "description").toString().isEmpty() )
  {
    args["description"] = getRecordValue(rec, "description").toString();
    textStream << dd << boundary << crlf;
    textStream << "Content-Disposition: form-data; name=\"description\"" << crlf << crlf;
    textStream << getRecordValue(rec, "description").toString() << crlf;
  }

  if( getRecordValue(rec, "tags").toString().split(",", QString::SkipEmptyParts).size() > 0 )
  {
    args["tags"] = getRecordValue(rec,"tags").toString().replace(QChar(','), QChar(' '));
    textStream << dd << boundary << crlf;
    textStream << "Content-Disposition: form-data; name=\"tags\"" << crlf << crlf;
    textStream << getRecordValue(rec,"tags").toString().replace(QChar(','), QChar(' ')) << crlf;
  }

  args["is_public"] = (getRecordValue(rec, "exposed").toBool() ? "1" : "0");
  textStream << dd << boundary << crlf;
  textStream << "Content-Disposition: form-data; name=\"is_public\"" << crlf << crlf;
  textStream << (getRecordValue(rec, "exposed").toBool() ? "1" : "0") << crlf;

  args["is_family"] = (getRecordValue(rec, "family").toBool() ? "1" : "0");
  textStream << dd << boundary << crlf;
  textStream << "Content-Disposition: form-data; name=\"is_family\"" << crlf << crlf;
  textStream << (getRecordValue(rec, "family").toBool() ? "1" : "0") << crlf;

  args["is_friend"] = (getRecordValue(rec, "friends").toBool() ? "1" : "0");
  textStream << dd << boundary << crlf;
  textStream << "Content-Disposition: form-data; name=\"is_friend\"" << crlf << crlf;
  textStream << (getRecordValue(rec, "friends").toBool() ? "1" : "0") << crlf;

  // We can now generate our MD5
  QString md5 = generateMD5(args);

  textStream << dd << boundary << crlf;
  textStream << "Content-Disposition: form-data; name=\"api_sig\"" << crlf << crlf;
  textStream << md5 << crlf;

  textStream << dd << boundary << crlf;
  textStream << "Content-Disposition: form-data; name=\"photo\"; filename=\"";
  textStream << getRecordValue(rec, "filename").toString() << "\"" << crlf;
  textStream << "Content-Type: ";
  textStream << KMimeType::findByUrl(getRecordValue(rec,"filename").toString())->name();
  textStream << crlf << crlf;

  // There has to be a safer way to deal with the raw data of the file
  // for now this seems to work however the idea of two streams operating
  // on the same IO Device is a bit scary to me.
  textStream.flush();

  if( getRecordValue(rec, "size").toString() == i18n("Original")
      && getRecordValue(rec, "rotation").toInt() == 0 )
  {
    // put the file data into the array
    QFile photoFile(getRecordValue(rec, "filename").toString());
    if ( !photoFile.open(QIODevice::ReadOnly) )
      return 0L;

    QDataStream dataStream(&buffer);
    dataStream.writeRawData(photoFile.readAll().data(), photoFile.size());
    photoFile.close();
  }
  else
  {
    QByteArray EXIFData;
    QByteArray fmt = QImageReader::imageFormat(getRecordValue(rec,"filename").toString());
    QList<QByteArray> fmts = QImageReader::supportedImageFormats();

    // copy the EXIF data header from the photo file
    if( (fmt == "JPEG" || fmt == "jpeg") && (fmts.contains("JPEG") || fmts.contains("jpeg")) )
    {
      EXIFData = EXIF(getRecordValue(rec,"filename").toString()).rawData();
    }

    // open the image
    QImage photoImage(getRecordValue(rec,"filename").toString());

    // size the image
    QString size = getRecordValue(rec,"size").toString();
    if( size != i18n("Original") )
    {
      // These are the values for width and height only if the size is defined as Custom
      int width(getRecordValue(rec,"width").toInt());
      int height(getRecordValue(rec,"height").toInt());

      // Otherwise we fall to default sizings. These have been set to match the
      // values used by flickr (look at "All Sizes" for any picture.
      if( size == i18n("Square") )
      {
	width = 75;
	height = 75;
      }
      else if( size == i18n("Thumb") )
      {
	width = 100;
	height = 75;
      }
      else if( size == i18n("Small") )
      {
	width = 240;
	height = 180;
      }
      else if( size == i18n("Medium") )
      {
	width = 500;
	height = 375;
      }
      else if( size == i18n("Large") )
      {
	width = 1024;
	height = 768;
      }

      // Swap width and height if the photo orientation is portrait, taking into account
      // any rotation that will be applied to the photograph as well.
      if( (PreviewMgr::instance()->orientation(getRecordValue(rec,"id").toInt())
	   == PreviewMgr::Portrait) && (getRecordValue(rec,"rotation").toInt() == 0 ||
					getRecordValue(rec,"rotation").toInt() == 180) )
      {
	int tmp = width;
	width = height;
	height = tmp;
      }
      
      // Preserve width or height if they are "74" (our indication to use original values)
      if( width == 74 )
      {
	width = photoImage.width();
      }
      if( height == 74 )
      {
	height= photoImage.height();
      }

      if( width <= 100 && height <= 100 )
      {
	photoImage = photoImage.scaled(width,height,Qt::IgnoreAspectRatio,Qt::SmoothTransformation);
      }
      else
      {
	photoImage = photoImage.scaled(width, height, Qt::KeepAspectRatio,Qt::SmoothTransformation);
      }
    }

    // rotate the image
    int rotation = getRecordValue(rec,"rotation").toInt();
    if( rotation != 0 )
    {
      QMatrix matrix;
      matrix = matrix.rotate(rotation);
      photoImage = photoImage.transformed(matrix, Qt::SmoothTransformation);
    }

    // add image data to our form data, maintain format if possible, otherwise try
    // to save as either jpeg or png. They used to use upper case, looks like lowercase
    // now, in either case we can now do either or.
    if( fmts.contains(fmt) )
    {
      if( !EXIFData.isEmpty() )
	writePhotoWithEXIF(buffer, photoImage, EXIFData);
      else
	photoImage.save(&buffer, fmt);
    }
    else if( fmts.contains("JPEG") )
    {
      photoImage.save(&buffer, "JPEG");
    }
    else if( fmts.contains("jpeg") )
    {
      photoImage.save(&buffer, "jpeg");
    }
    else if( fmts.contains("PNG") )
    {
      photoImage.save(&buffer, "PNG");
    }
    else if( fmts.contains("png") )
    {
      photoImage.save(&buffer, "png");
    }
  }

  // end form
  textStream << crlf << dd << boundary << dd << crlf << "\0";
  buffer.close();

  // send request
  KIO::TransferJob *job = KIO::http_post(url, array, KIO::HideProgressInfo);
  job->addMetaData("content-type", "Content-Type: multipart/form-data; boundary=" + boundary);

  connect(job,SIGNAL(result(KJob*)),SLOT(jobResult(KJob*)));
  connect(job,SIGNAL(data(KIO::Job*,const QByteArray&)),SLOT(jobData(KIO::Job*,const QByteArray&)));

  m_requests[job] = FILE_UPLOAD;
  return job;
}

void FlickrComm::jobResult(KJob *job)
{
  QString err;
  KIO::TransferJob *transferJob = dynamic_cast<KIO::TransferJob*>(job);

  // Get associated request type
  if( transferJob )
  {
    // Check that the operation did not error
    if( job->error() )
    {
      emit commError(i18n("HTTP request failed. (error: %1)", job->errorString()));
      m_requests.remove(transferJob);
      m_incomingData.remove(transferJob);
      return;
    }

    // notify user of specific error
    if( m_requests[transferJob] != NO_REQ
	&& (err = validateHTTPResponse(m_incomingData[transferJob])) != "" )
    {
      emit commError(i18n("HTTP request to Flickr.com failed. (msg: %1)", err));
      m_requests.remove(transferJob);
      m_incomingData.remove(transferJob);
      return;
    }

    // process the good response
    switch( m_requests[transferJob] )
    {
    case FROB_REQ:
      handleFrobResponse(m_incomingData[transferJob]);
      break;
    case TOKEN_REQ:
      handleTokenResponse(m_incomingData[transferJob]);
      break;
    case TAGS_REQ:
      handleTagsResponse(m_incomingData[transferJob]);
      break;
    case STATUS_REQ:
      handleStatusResponse(m_incomingData[transferJob]);
      break;
    case FILE_UPLOAD:
      handleUploadResponse(m_incomingData[transferJob]);
      break;
    case PHOTOSET_REQ:
      handlePhotosetResponse(m_incomingData[transferJob]);
      break;
    case CREATE_PHOTOSET_REQ:
      hanldeCreatePhotosetResponse(m_incomingData[transferJob]);
      break;
    case LICENSES_REQ:
      handleLicensesResponse(m_incomingData[transferJob]);
      break;
    default:
      break;
    };
    m_requests.remove(transferJob);
    m_incomingData.remove(transferJob);
  }
}

void FlickrComm::jobData(KIO::Job *job, const QByteArray &data)
{
  KIO::TransferJob *transferJob = dynamic_cast<KIO::TransferJob*>(job);

  if( transferJob && !data.isEmpty() )
  {
    // Add new data to end of the associated buffer
    m_incomingData[transferJob].append(QString::fromUtf8(data, data.size()));
  }
}

void FlickrComm::handleFrobResponse(const QString &str)
{
  QString frob = "";
  QDomNode node;
  QDomElement root;
  QDomDocument doc("frobresponse");

  // pump str into the XML document
  if( !doc.setContent(str) )
  {
    emit commError(i18n("Invalid response received from Flickr.com."));
    return;
  }

  // start at document root, first node
  root = doc.documentElement();
  node = root.firstChild();

  // go through each node, should only be one
  while( !node.isNull() )
  {
    // we got our frob, good to go
    if( node.isElement() && node.nodeName() == "frob" )
    {
      QDomElement elem = node.toElement();
      frob = elem.text();
    }
    node = node.nextSibling();
  }

  // move on to next step if we have our frob
  if( !frob.isEmpty() )
  {
    emit returnedFrob(frob);
  }
  else
  {
    emit commError(i18n("Flickr.com returned empty 'frob'"));
  }
}

void FlickrComm::handleTokenResponse(const QString &str)
{
  QString nsid;
  QString token;
  QString perms;
  QDomNode node;
  QDomElement root;
  QString username = i18n("Unknown");
  QDomDocument doc("tokenresponse");

  // pump str into the XML document
  if( !doc.setContent(str) )
  {
    emit commError(i18n("Invalid response received from Flickr.com."));
    return;
  }

  // start at document root, first node
  root = doc.documentElement();
  node = root.firstChild();

  // go through each node, should only be one
  while( !node.isNull() )
  {
    // we got our token
    if( node.isElement() && node.nodeName() == "token" )
    {
      QDomElement elem = node.toElement();
      token = elem.text();
    }

    // we got our permission
    if( node.isElement() && node.nodeName() == "perms" )
    {
      QDomElement elem = node.toElement();
      perms = elem.text();
    }

    // for the user
    if( node.isElement() && node.nodeName() == "user" )
    {
      QDomElement elem = node.toElement();
      username = elem.attribute("username", i18n("Unknown"));
      nsid = elem.attribute("nsid", "");
    }

    // step down the node tree
    if( node.isElement() && node.nodeName() == "auth" )
      node = node.firstChild();
    else
      node = node.nextSibling();
  }

  // add the user and make active
  emit returnedToken(username, token, nsid);
}

void FlickrComm::handleTagsResponse(const QString &str)
{
  QDomNode node;
  QStringList tags;
  QDomElement root;
  QDomDocument doc("tagsresponse");


  // pump str into the XML document
  if( !doc.setContent(str) )
  {
    emit commError(i18n("Invalid response received from Flickr.com."));
    return;
  }

  // start at document root, first node
  root = doc.documentElement();
  node = root.firstChild();

  // go through each node, should only be one
  while( !node.isNull() )
  {
    // we got a tag
    if( node.isElement() && node.nodeName() == "tag" )
    {
      QDomElement elem = node.toElement();

      // put quotes around multi word tags
      if( elem.text().contains(QRegExp("\\s+")) )
	tags += QString("\"" + elem.text() + "\"");
      else
	tags += elem.text();
    }

    // step down the node tree
    if( node.isElement() && (node.nodeName() == "who" || node.nodeName() == "tags") )
      node = node.firstChild();
    else
      node = node.nextSibling();
  }

  // add the user and make active
  emit returnedTags(tags);
}

void FlickrComm::handleStatusResponse(const QString &str)
{
  QDomNode node;
  QDomElement root;
  QString remaining;
  QString remainingkb;
  QDomDocument doc("statusresponse");

  // pump str into the XML document
  if( !doc.setContent(str) )
  {
    emit commError(i18n("Invalid response received from Flickr.com."));
    return;
  }

  // start at document root, first node
  root = doc.documentElement();
  node = root.firstChild();

  // go through each node, should only be one
  while( !node.isNull() )
  {
    // we got a tag
    if( node.isElement() && node.nodeName() == "bandwidth" )
    {
      QDomElement elem = node.toElement();
      remainingkb = elem.attribute("remainingkb", "");
    }

    // step down the node tree
    if( node.isElement() && node.nodeName() == "user" )
      node = node.firstChild();
    else
      node = node.nextSibling();
  }

  // calculate remaining bandwidth
  if( remainingkb != "" )
  {
    float value;

    // remaining in Bytes
    value = remainingkb.toFloat();

    // make a nice string
    if( value > (1024.0 * 1024.0) )
    {
      remaining.sprintf("%.1f", (value / (1024.0 * 1024.0)));
      remaining += "GB";
    }
    else if( value > 1024.0 )
    {
      remaining.sprintf("%.1f", (value / (1024.0)));
      remaining += "MB";
    }
    else
    {
      remaining.sprintf("%.1f", value);
      remaining += "KB";
    }
  }
  else
  {
    remaining = i18n("Unknown");
  }

  // emit the users upload remaining
  emit returnedUploadStatus(remaining);
}

void FlickrComm::handleUploadResponse(const QString &str)
{
  QString id;
  QDomNode node;
  QDomElement root;
  QDomDocument doc("uploadresponse");

  // pump str into the XML document
  if( !doc.setContent(str) )
  {
    emit commError(i18n("Invalid response received from Flickr.com."));
    return;
  }

  // start at document root, first node
  root = doc.documentElement();
  node = root.firstChild();

  while( !node.isNull( ) )
  {
    // we got our id, good to go
    if( node.isElement( ) && node.nodeName( ) == "photoid" )
    {
      QDomElement elem = node.toElement();
      id = elem.text( );
    }
    node = node.nextSibling();
  }

  // notify that upload was OK
  emit returnedUploadedOK(id);
}

void FlickrComm::handlePhotosetResponse(const QString &str)
{
  QString id;
  QDomNode node;
  QDomElement root;
  QStringList setTitles;
  QDomDocument doc("photosetsresponse");


  // pump str into the XML document
  if( !doc.setContent(str) )
  {
    emit commError(i18n("Invalid response received from Flickr.com."));
    return;
  }

  // Clear the current photoset data
  m_photoSets.clear();

  // start at document root, first node
  root = doc.documentElement();
  node = root.firstChild();

  // go through each node, should only be one
  while( !node.isNull() )
  {
    // we got a set, store the id and fish out the title
    if( node.isElement() && node.nodeName() == "photoset" )
    {
      QDomElement elem = node.toElement();
      id = elem.attribute("id");

      elem = elem.elementsByTagName("title").item(0).toElement();
      if( id != QString::null )
      {
	setTitles.append(elem.text());
	m_photoSets.insert(elem.text(), id);
      }

    }

    // step down the node tree
    if( node.isElement() && node.nodeName() == "photosets" )
      node = node.firstChild();
    else
      node = node.nextSibling();
  }

  // sort the title alphabetically
  //setTitles.sort();

  // notify interested widgets that new set titles are available
  emit returnedPhotosets(setTitles, QString::null);
}

void FlickrComm::hanldeCreatePhotosetResponse(const QString &str)
{
  QString id;
  QDomNode node;
  QString newTitle;
  QDomElement root;
  SetData::Iterator it;
  QStringList setTitles;
  QDomDocument doc("photosetsresponse");

  // pump str into the XML document
  if( !doc.setContent(str) )
  {
    emit commError(i18n("Invalid response received from Flickr.com."));
    return;
  }

  // start at document root, first node
  root = doc.documentElement();
  node = root.firstChild();

  // extract the id from the XML document
  while( !node.isNull() )
  {
    // we got our photoset ID, good to go
    if( node.isElement() && node.nodeName() == "photoset" )
    {
      QDomElement elem = node.toElement();
      id = elem.attribute("id");
    }
    node = node.nextSibling();
  }

  // Do two things at once; create the string list with all the photoset titles and
  // find the SetData entry with no ID, this will be the newly created set. Since we
  // are using a map the title will already be sorted as well so no need to sort.
  for( it = m_photoSets.begin(); it != m_photoSets.end(); ++it )
  {
    setTitles.append(it.key());

    // Is this the set we are creating
    if( it.value() == QString::null )
    {
      newTitle = it.key();         // store as active set
      m_photoSets[newTitle] = id;  // store the id
    }
  }

  // notify interested widgets that new set titles are available
  emit returnedPhotosets(setTitles, newTitle);
}

void FlickrComm::handleLicensesResponse(const QString& str)
{
  QString id;
  QString name;
  QDomNode node;
  QDomElement root;
  QStringList licenseNames;
  QDomDocument doc("licensesresponse");

  // pump str into the XML document
  if( !doc.setContent(str) )
  {
    emit commError(i18n("Invalid response received from Flickr.com."));
    return;
  }

  // Clear the current licenses data
  m_licenseIDs.clear();

  // start at document root, first node
  root = doc.documentElement();
  node = root.firstChild();

  // go through each node, should only be one
  while( !node.isNull() )
  {
    // we got a license, store the id and fish out the name
    if( node.isElement() && node.nodeName() == "license" )
    {
      QDomElement elem = node.toElement();
      id = elem.attribute("id");
      name = elem.attribute("name");

      if( id != QString::null && name != QString::null )
      {
	licenseNames.append(name);
	m_licenseIDs.insert(name, id);
      }

    }

    // step down the node tree
    if( node.isElement() && node.nodeName() == "licenses" )
      node = node.firstChild();
    else
      node = node.nextSibling();
  }

  // notify any interested objects of the set of available licenses
  emit returnedLicenses(licenseNames);
}

void FlickrComm::createPhotoset(const QString &token, const QString &name, const QString &photoID)
{
  ArgMap args;

  // Setup the flickr call
  args["method"]              = "flickr.photosets.create";
  args["title"]               = name;
  args["primary_photo_id"]    = photoID;
  args["auth_token"]          = token;

  // Add the set name to the SetData
  m_photoSets[name] = QString::null;

  // Send the request
  KIO::TransferJob* job = sendRequest(args);
  m_requests[job] = CREATE_PHOTOSET_REQ;
}

void FlickrComm::addPhoto2Photoset(const QString &token, const QString &photoset, const QString &photoID)
{
  // check if the set has to be created
  if( !m_photoSets.contains(photoset) )
  {
    createPhotoset(token,photoset,photoID);
  }
  else    // otherwise just add the photo to the set
  {
    ArgMap args;

    // Setup the flickr call
    args["method"]      = "flickr.photosets.addPhoto";
    args["photoset_id"] = m_photoSets[photoset];
    args["photo_id"]    = photoID;
    args["auth_token"]  = token;

    // Send the request
    KIO::TransferJob* job = sendRequest(args);
    m_requests[job] = ADD_PHOTO2SET_REQ;
  }
}

void FlickrComm::setPhotoLicense(const QString& token, const QString& license, const QString& photoID)
{
  ArgMap args;

  // Setup the flickr call
  args["method"]      = "flickr.photos.licenses.setLicense";
  args["license_id"] = m_licenseIDs[license];
  args["photo_id"]    = photoID;
  args["auth_token"]  = token;

  // Send the request
  KIO::TransferJob* job = sendRequest(args);
  m_requests[job] = SET_LICENSE_REQ;
}

void FlickrComm::abortCurrentRequest()
{
  QMap<KIO::TransferJob*,ResponseType>::iterator iter;

  for( iter = m_requests.begin(); iter != m_requests.end(); ++iter )
  {
    iter.key()->kill();
  }
  m_requests.clear();
  m_incomingData.clear();
}

void FlickrComm::writePhotoWithEXIF(QBuffer &out, QImage &in, QByteArray &exif)
{
  ulong arrayPos;
  ulong dataLength;
  QByteArray array;
  QBuffer buffer(&array);
  QDataStream dataStream(&out);

  // Save the image to the buffer
  buffer.open(QIODevice::WriteOnly);
  in.save(&buffer, "JPEG");
  buffer.close();

  // Write the JPEG standard first two bytes to the stream
  dataStream.writeRawData(array.data(), 2);
  arrayPos = 2;

  // Write JFIF data if present
  if( (uchar)array[2] == 0xFF && (uchar)array[3] == 0xE0 )
  {
    // Remember to add 2 bytes since the data length bytes are
    // included but not the marker bytes which also want included
    dataLength = ((uchar)array[4] * 256) + (uchar)array[5] + 2;
    dataStream.writeRawData(array.mid(2,dataLength).data(), dataLength);
    arrayPos += dataLength;
  }

  // Write our passed in EXIF data
  dataStream.writeRawData(exif.data(), exif.size());

  // Skip current EXIF data if present. 0xffe1 is EXIF marker followed by two
  // bytes which indicate EXIF size in big endian.
  if( (uchar)array.at(arrayPos) == 0xFF && (uchar)array.at(arrayPos+1) == 0xE1 )
  {
    // Remember to add 2 bytes since the data length bytes are
    // included but not the marker bytes which we also want included
    arrayPos += ((uchar)array.at(arrayPos+2) * 256) + (uchar)array.at(arrayPos+3) + 2;
  }

  // Write everything after the arrayPos bytes.
  dataStream.writeRawData(array.mid(arrayPos), array.size() - arrayPos);
}


#include "flickrcomm.moc"
