/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 08-11-2008
 * Description : Handle drawing the photo preview in listview
 *
 * Copyright 2008 by M. Asselstine

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include "photodelegate.h"

#include <QPen>
#include <QSize>
#include <QDebug>
#include <QBrush>
#include <QStyle>
#include <QPainter>
#include <QTransform>
#include <QModelIndex>
#include <QSqlTableModel>
#include <QStyleOptionViewItem>

#include <kicon.h>
#include <kglobal.h>
#include <kstandarddirs.h>

#include "previewmgr.h"


namespace
{
  static const QPixmap& getBorderImage()
  {
    static QPixmap border;

    if( border.isNull() )
    {
      border = QPixmap(KGlobal::dirs()->findResource("data", "kflickr/border.png"));
    }

    return border;
  }
};

PhotoDelegate::PhotoDelegate(QObject *parent)
  : QItemDelegate(parent)
{
}

void PhotoDelegate::paint(QPainter* painter, const QStyleOptionViewItem& option,
			  const QModelIndex& index) const
{
  QPalette::ColorGroup cg = option.state & QStyle::State_Enabled
    ? option.palette.currentColorGroup() : QPalette::Disabled;
  if (option.state & QStyle::State_Selected)
  {
    if (cg == QPalette::Normal && !(option.state & QStyle::State_Active))
      cg = QPalette::Inactive;
    painter->fillRect(option.rect, option.palette.brush(cg, QPalette::Highlight));
  }

  QSize size(option.rect.width(), option.rect.height());

  const QSqlTableModel* model = static_cast<const QSqlTableModel*>(index.model());
  int id = model->data(model->index(index.row(), model->fieldIndex("id"))).toInt();
  int rotation = model->data(model->index(index.row(), model->fieldIndex("rotation"))).toInt();

  painter->save();
  painter->translate(QPoint(option.rect.left(),option.rect.top()));
  painter->setRenderHint(QPainter::Antialiasing, true);
  if (option.state & QStyle::State_Selected)
    painter->setBrush(option.palette.highlightedText());
  else
    painter->setBrush(QBrush(Qt::black));

  // draw the photograph border
  painter->drawPixmap((size.width()-195)/2 , 4, getBorderImage());

  // draw the photo
  const QPixmap* pixmap = PreviewMgr::instance()->preview(id);    
  if( !pixmap )
  {
    pixmap = PreviewMgr::instance()->defaultPreview();
  }
  QPixmap pd = pixmap->transformed(QTransform().rotate(rotation));
  painter->drawPixmap((size.width() - pd.width())/2 , (size.height() - pd.height())/2, pd);

  painter->restore();
}

QSize PhotoDelegate::sizeHint(const QStyleOptionViewItem& option, const QModelIndex& index) const
{
  Q_UNUSED(option);
  Q_UNUSED(index);

  return QSize(200, 200);
}
