\ Adaptive integration using trapezoidal rule
\ with Richardson extrapolation
\ Integrate a real function from xa to xb

\ Forth Scientific Library Algorithm #19

CR .( ADAPTINT          V1.1b         17 April    2005   JVN, KM )

\ Usage:  use( fn.name xa xb err )integral
\ Examples:

\ use( FSQRT  0e  1e  1e-3 )integral  F. 0.666666  ok
\ use( FSQRT  0e  2e  1e-4 )integral  F. 1.88562  ok

\ : f1     FDUP FSQRT F*  ;  ok
\ use( f1  0e  1e  1e-3 )integral  F. 0.400001  ok
\ use( f1  0e  2e  1e-4 )integral  F. 2.26274  ok

\ Programmed by J.V. Noble (from "Scientific FORTH" by JVN)
\ ANS Standard Program  -- version of  10/5/1994
\ Revised 4/17/2005 by K. Myneni to use FSL array definitions and
\   vectoring; also modified for use with integrated stack Forths.
\
\ This is an ANS Forth program requiring:
\      The FLOAT and FLOAT EXT word sets
\ Environmental dependencies:
\ xxx   Assumes independent floating point stack   xxx  
\ This is a version for Forths which do not have a separate floating
\   point stack, e.g. kForth.
\
\     (c) Copyright 1994  Julian V. Noble.     Permission is granted
\     by the author to use this software for any application provided
\     the copyright notice is preserved.
\
\  Requires the following under kForth:
\
\    ans-words.4th
\    fsl-util.4th


\ Data structures
0 S>F  FCONSTANT F=0
4 S>F  3 S>F  F/  FCONSTANT F=4/3

20  float  array  x{
20  float  array  e{
20  float  array  f{
20  float  array  i{

0 VALUE  N

FVARIABLE  old.i
FVARIABLE  final.i

\ Begin definitions proper

: )int  ( n --)                           \ trapezoidal rule
\   F" ( F(N) + F(N-1) ) * ( X(N) - X(N-1) ) / 2  "
    >R
    x{ R@ } F@   x{ R@ 1- }  F@
    F-  f2/
    f{ R@ } F@   f{ R@ 1- }  F@
        F+  F*
    i{ R> 1- } F!  ;

v: dummy                                  \ dummy function name

: initialize  ( xt xa xb eps -- integral )  
     1 TO N
     e{ 0 } F!   x{ 1 } F!   x{ 0 } F!
     defines dummy
     x{ 0 } F@   dummy   f{ 0 } F!        \ F" f(0) = dummy( x(0) ) "
     x{ 1 } F@   dummy   f{ 1 } F!        \ F" f(1) = dummy( x(1) ) "
     1 )int
     F=0  final.i  F! ;

: check.N       N  19 >   ABORT" Too many subdivisions!"  ;
: e/2   e{ N  1- }  DUP   >R F@   F2/  R>  F! ;
: }down    ( adr n --)
        OVER cell- @  >R   }   DUP   R@ +   R>   MOVE  ;

: move.down    e{ N  1-       }down
               x{ N           }down
               f{ N           }down  ;

: x'   \  F" X(N) = ( X(N) + X(N-1) ) / 2 "
       \  F" F(N) = DUMMY(X(N)) "
       x{ N }  F@   x{ N 1- }  F@    F+  F2/
       FDUP  x{ N }  F!   dummy  f{ N }  F!   ;

: N+1   N 1+   TO N  ;
: N-2   N 2 -  TO N  ;

: subdivide    check.N     e/2   move.down
        i{ N 1- }  F@  old.i  F!
        x'   N )int   N 1+ )int    ;

: converged?   ( -- I[N]+I'[N-1]-I[N-1] flag )
\       F" I(N) + IP(N-1) - I(N-1) "
        i{ N } F@  i{ N 1- } F@   F+  old.i F@  F-
        FDUP   FABS
        e{ N 1- } F@   F2*  F<  ;

: interpolate  ( I[N]+I'[N-1]-I[N-1] -- )

\     F" FINAL.I = ( I(N)+I'(N-1) - OLD.I ) * (4/3) + OLD.I + FINAL.I "
      F=4/3   F*   old.i F@   F+
      final.i F@   F+                    \ accumulate
      final.i F!  ;                      \ store it


: )integral    ( A B ERR xt -- I[A,B] ) 
     initialize
     BEGIN   N 0>   WHILE
        subdivide
        converged?    N+1
        IF    interpolate  N-2
        ELSE  FDROP    THEN
     REPEAT   final.i  F@  ;

