/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#ifndef KFTPCLIENTTHREAD_H
#define KFTPCLIENTTHREAD_H

#include <qobject.h>
#include <qthread.h>
#include <qvaluelist.h>
#include <qstringlist.h>

#include <kurl.h>

#include "kftpextensions.h"
#include "socket.h"

#define THR_EVENT_ID 65100

enum CommandType {
    TCMD_NONE = -1,
    TCMD_GET = 0,
    TCMD_PUT = 1,
    TCMD_LIST = 2,
    TCMD_CONNECT = 3,
    TCMD_DISCONNECT = 4,
    TCMD_SYNC = 5,
    TCMD_REMOVE = 6,
    TCMD_RENAME = 7,
    TCMD_CHMOD = 8,
    TCMD_MKDIR = 9,
    TCMD_FXP = 10,
    TCMD_STAT = 11,
    TCMD_SCAN = 12,
    TCMD_RAW = 13
};

enum ParamType {
    PARAM_STRING,
    PARAM_INT,
    PARAM_URL,
    PARAM_CLIENT,
    PARAM_TRANSFER
};

enum EventType {
    EVENT_FINISHED,
    EVENT_LOG,
    EVENT_OFFSET,
    EVENT_LOGIN,
    EVENT_DISCONNECT,
    EVENT_RETRYOK,
    EVENT_ERRORHDL,
    EVENT_RAW_REPLY,
    EVENT_STATE_CH
};

/**
@author Jernej Kos
*/
class CommandParameter
{
public:
    CommandParameter();
    CommandParameter(const QString &string);
    CommandParameter(int integer);
    CommandParameter(const KURL &url);
    CommandParameter(KFTPNetwork::Socket *client);
    CommandParameter(KFTPQueue::Transfer *transfer);

    QString asString() { return m_string; }
    int asInt() { return m_int; }
    KURL asUrl() { return m_url; }
    KFTPNetwork::Socket *asClient() { return static_cast<KFTPNetwork::Socket*>(m_data); }
    KFTPQueue::Transfer *asTransfer() { return static_cast<KFTPQueue::Transfer*>(m_data); }
private:
    QString m_string;
    int m_int;
    KURL m_url;
    void *m_data;

    ParamType m_type;
};

typedef QValueList<CommandParameter> ParameterList;

/**
@author Jernej Kos
*/
class KFTPThreadCommand
{
friend class KFTPThreadDispatcher;
public:
    int execute(KFTPNetwork::Socket *client);
private:
    CommandType m_cmdType;
    ParameterList m_params;
};

typedef QValueList<KFTPThreadCommand> KFTPThreadCommandList;

/**
 * @author Jernej Kos
 *
 * @brief This class executes queued commands
 */
class KFTPThreadDispatcher : public QObject
{
Q_OBJECT
public:
    KFTPThreadDispatcher(QObject *parent = 0);

    void startDispatchLoop();
    void queueCommand(CommandType cmdType, ParameterList params);

    virtual void sleep(unsigned long msecs) = 0;
    CommandType getCurrentCommand() { return m_curCmd; }
protected:
    bool m_abortLoop;
    CommandType m_curCmd;

    KFTPThreadCommandList m_commandQueue;
    KFTPNetwork::SocketManager *m_ftpClient;
signals:
    void finished(CommandType cmdType);
};

/**
 * @author Jernej Kos
 *
 * @brief Object where the thread actually runs. The start method just runs the command dispatch loop
 */
class KFTPClientThread : public KFTPThreadDispatcher, public QThread
{
Q_OBJECT
public:
    KFTPClientThread();

    void setClient(KFTPNetwork::SocketManager *client);
    virtual void run();

    void sleep(unsigned long msecs);
};

class ThreadEvent : public QCustomEvent
{
public:
    ThreadEvent(void *data, EventType type)
      : QCustomEvent((QEvent::Type) THR_EVENT_ID, data), m_type(type) {}

    EventType type() const { return m_type; }
private:
    EventType m_type;
};

/**
 * @author Jernej Kos
 *
 * @brief Wrapper class for communication between the socket thread and the rest of the GUI
 */
class KFTPClientThr : public QObject
{
Q_OBJECT
public:
    KFTPClientThr(QObject *parent = 0, const char *name = 0);
    ~KFTPClientThr();

    void connectToHost(const KURL &url);
    void disconnect();

    void dirList(const KURL &url);
    void get(const KURL &source, const KURL &destination);
    void put(const KURL &source, const KURL &destination);
    void remove(const KURL &url);
    void rename(const KURL &source, const KURL &destination);
    void chmod(const KURL &url, int mode);
    void mkdir(const KURL &url);
    void fxpTransfer(const KURL &source, const KURL &destination, KFTPClientThr *client);
    void stat(const KURL &url);
    void scanDirectory(KFTPQueue::Transfer *transfer);
    void rawCommand(const QString &cmd);

    /* WARNING! This function is used to get the actual client - only for getting actions
       that return imediatly! For longer actions use the wraper functions above! */
    KFTPNetwork::Socket *getClient() { return m_client->socket(); }
    KFTPNetwork::SocketManager *getSocketManager() { return m_client; }
private:
    KFTPNetwork::SocketManager *m_client;
    KFTPClientThread *m_thread;

    QStringList m_lastLog;
    QStringList m_lastReply;

    FTPEntry m_lastEntry;

    void setupSignals(KFTPNetwork::Socket *socket);
protected:
    void customEvent(QCustomEvent *e);
private slots:
    void slotFinished(CommandType cmdType);

    void slotRawReply(const QString &reply);
    void slotLogUpdate(int type, const QString &text);
    void slotDisconnectDone();
    void slotLoginComplete(bool success);
    void slotResumedOffset(filesize_t bytes);
    void slotRetrySuccess();
    void slotErrorHandler(KFTPNetwork::Error error);

    void slotStateChanged(KFTPNetwork::SocketState state);
signals:
    void finished(CommandType cmdType);

    void rawReply(const QString &reply);
    void logUpdate(int type, const QString &text);
    void disconnectDone();
    void loginComplete(bool success);
    void resumedOffset(filesize_t bytes);
    void retrySuccess();
    void errorHandler(KFTPNetwork::Error error);
    void stateChanged(KFTPNetwork::SocketState state);
};

#endif
