/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "socket.h"
#include "ftpsocket.h"
#include "sftpsocket.h"

#include "misc/config.h"
#include "kftpqueue.h"
#include "errorhandler.h"

#include <qtimer.h>
#include <klocale.h>

namespace KFTPNetwork {

SocketStateInfo::SocketStateInfo(Socket *socket)
  : m_socket(socket),
    m_abortInProgress(false),
    m_compositeState(false),
    m_sockState(S_IDLE)
{
}

SocketState SocketStateInfo::getSocketState()
{
  return m_sockState;
}

void SocketStateInfo::abortFinished()
{
  if (m_compositeState)
    return;
    
  m_abortInProgress = false;
}

void SocketStateInfo::resetState()
{
  m_stateStack.clear();
  m_sockState = S_IDLE;
  
  emit m_socket->sigStateChanged(m_sockState);
}

void SocketStateInfo::enterSocketState(SocketState state, bool compositeState)
{
  // Clear the abort marker
  if (abortInProgress())
    abortFinished();
    
  if (state != S_IDLE) {
    m_stateStack.push(state);
  } else if (!m_stateStack.isEmpty()) {
    // Remove the item from the state list
    m_stateStack.pop();
    
    if (!m_stateStack.isEmpty())
      state = m_stateStack.pop();
  }
    
  // Reset some variables when beginning with transfer operations
  if (state == S_TRANSFER) {
    Socket *s = m_socket;
    
    s->m_processedOffset = 0;
    s->m_stallCount = 0;
    s->m_lastTime = 0;
    s->m_operationStart = time(0L);
    s->m_lastTime = -1;
    s->m_lastXfered = 0;
    s->m_processedSize = 0;
  }
  
  if (compositeState) {
    if (state == S_IDLE) {
      m_compositeState = false;
      abortFinished();
    } else {
      m_compositeState = true;
    }
  } else {
    // If we are in composite state, ignore all state changes
    if (m_compositeState)
      return;
  }
  
  // Change the state and signal the change
  m_sockState = state;
  
  emit m_socket->sigStateChanged(state);
}

void SpeedLimit::transferStart()
{
  // Get the transfer start time
  gettimeofday(&m_lastPoll, NULL);
}

int SpeedLimit::allowedBytes()
{
  if (m_speedLimit > 0) {
    long delta = SpeedLimit::_timeSince(&m_lastPoll);
    register double rate = (double) m_speedLimit / 1000.0;
    int bytes = 0;
    
    if (delta > rate) {
      bytes = int(rate * delta);
      gettimeofday(&m_lastPoll, NULL);
    }
    
    if (bytes == 0)
      ::usleep(1000);
    
    return bytes;
  }
  
  return 32768;
}

long SpeedLimit::_timeSince(struct timeval *then)
{
  struct timeval now;
  gettimeofday(&now, NULL);

  return (((now.tv_sec - then->tv_sec) * 1000L) +
    ((now.tv_usec - then->tv_usec) / 1000L));
}

Socket::Socket(QObject *parent, const QString &protocol)
  : QObject(parent),
    m_stateInfo(SocketStateInfo(this)),
    m_errorHandler(new ErrorHandler(this)),
    m_isLoggedIn(false),
    m_remoteEncoding(new KRemoteEncoding()),
    m_protocol(protocol)
{
  // Initialize configuration
  initConfig();
  
  m_stateInfo.enterSocketState(S_IDLE);
}

Socket::~Socket()
{
  delete m_remoteEncoding;
}

filesize_t Socket::getSpeed()
{
  /* If it is stalled - return 0 as speed */
  if (m_stallCount > 50) {
    // FIXME is 50 good enough ?
    return 0;
  }

  time_t secs = time(0L) - m_operationStart;
  filesize_t xferBytes = m_processedSize - m_processedOffset;

  if (secs - m_lastTime <= 0) {
    /* Too fast start, huh */
    return 0;
  }
  
  filesize_t curSpeed = (xferBytes - m_lastXfered)/(secs - m_lastTime);

  m_lastTime = secs;
  m_lastXfered = xferBytes;
  
  return curSpeed;
}

void Socket::processedSize(filesize_t size)
{
  if (size == 0) {
    /* No change in processed size -- is it stalled ? */
    m_stallCount++;
  } else {
    m_stallCount = 0;
  }

  m_processedSize += size;
}

void Socket::setOffset(filesize_t offset)
{
  m_processedOffset = offset;
  processedSize(offset);
}

void Socket::slotReconnect()
{
  m_reconnectTimeout = 1;
}

void Socket::setSiteUrl(KURL url)
{
  // Set the current url
  m_currentUrl = url;
  m_currentUrl.setPath("/");
}

void Socket::changeEncoding(const QString &encoding)
{
  // Alter encoding and change socket config
  m_remoteEncoding->setEncoding(encoding.ascii());
  setConfig("encoding", encoding);
}

void Socket::connect(KURL url)
{
  setSiteUrl(url);
  
  // Set remote encoding as specified by config
  m_remoteEncoding->setEncoding(getConfigStr("encoding").ascii());
  
  if (m_isLoggedIn) {
    // We are already logged in - disconnect
    Socket::disconnect();
  }
  
  m_stateInfo.enterSocketState(S_CONNECT);

  // Connect and login to server
  emit sigLogUpdate(4, i18n("Connecting to '%1:%2'...").arg(url.host()).arg(url.port()));
  int conn_result = protoConnect();
  if (conn_result < 0) {
    emit sigLogUpdate(4, i18n("Unable to connect to the server."));

    // Retry to connect, since server problems might be temporary
    goto retry_cycle;
  }

  // Retry if an error ocurred while logging in
  if (protoLogin() < 0) {
    // If the retry option is set, we should try to re-login until
    // retry_num is 0
    emit sigLogUpdate(4, i18n("Login failed."));
    Socket::disconnect();

retry_cycle:
    if (getConfig("do_retry") == 1) {
      int max_retries = getConfig("max_retries");
      int retry_num = max_retries;
      bool infinite = retry_num == 0;

      while (retry_num > 0 || infinite) {
        // Sleep for retry delay and retry
        emit sigLogUpdate(4, i18n("Waiting %1 seconds before reconnect...").arg(getConfig("retry_delay")));

        m_reconnectTimeout = 0;
        QTimer::singleShot(getConfig("retry_delay") * 1000, this, SLOT(slotReconnect()));
        while (m_reconnectTimeout == 0) {
          ::usleep(100000);

          if (m_stateInfo.abortInProgress()) {
            m_stateInfo.enterSocketState(S_IDLE);
            
            emit sigLogUpdate(4, i18n("Retry aborted."));
            goto done;
          }
        }

        if (!infinite)
          emit sigLogUpdate(4, i18n("Retrying connection (%1/%2)...").arg(max_retries-retry_num+1).arg(max_retries));
        else
          emit sigLogUpdate(4, i18n("Retrying connection..."));
          
        if (protoConnect() == 1) {
          if (protoLogin() == 1) {
            // Login success !
            break;
          } else {
            emit sigLogUpdate(4, i18n("Login failed."));
            Socket::disconnect();
          }
        } else {
          emit sigLogUpdate(4, i18n("Unable to connect."));
        }

        retry_num--;
      }

      m_stateInfo.enterSocketState(S_IDLE);

      if (!m_isLoggedIn) {
        // We are not logged in, so we have failed
        emit sigLogUpdate(4, i18n("Connection to '%1' has failed.").arg(url.host()));
      } else {
        // We have been retrying and we have successfully connected, emit a signal so a notification
        // might be shown (if user has set so).
        emit sigRetrySuccess();
      }
    }
  }
  
  m_stateInfo.enterSocketState(S_IDLE);

done:
  emit sigLoginComplete(m_isLoggedIn);
}

void Socket::disconnect()
{
  protoDisconnect();
  
  emit sigLogUpdate(3, i18n("Disconnected from server."));
  emit sigDisconnectDone();
}

FTPDirList *Socket::dirList(const KURL &url, bool ignoreCache)
{
  // Format the url correctly
  KURL p_url = url;
  if (!url.hasHost()) {
    KURL tmp = getClientInfoUrl();
    tmp.setPath(url.path());
    p_url = tmp;
  }

  // If enabled, first check the cache
  if (getConfig("use_cache") == 1 && FTPCache().listGetFromCache(p_url) != 0L && !ignoreCache) {
    /* We have it cached */
    FTPCacheItem *item = FTPCache().listGetFromCache(p_url);

    m_lastDirList = item->getDirList();
    return &m_lastDirList;
  }

  // Check if we are using this for browsing offline and return null if
  // url is not in cache
  if (!m_isLoggedIn)
    return 0L;

  m_stateInfo.enterSocketState(S_LIST);
  
  if (protoDirList(p_url)) {
    // Cache current dir listing
    FTPCache().listCache(p_url, m_lastDirList);
    m_stateInfo.enterSocketState(S_IDLE);
    
    return &m_lastDirList;
  }
  
  m_stateInfo.enterSocketState(S_IDLE);
  

  return 0L;
}

bool Socket::checkIsDir(const KURL &url)
{
  bool is_dir = false;
  FTPDirectoryItem *cachedItem = FTPCache().listGetFileFromCache(url);
  if (cachedItem != 0L && getConfig("use_cache") == 1) {
    // We have it cached
    is_dir = cachedItem->type() == 'd';

    // BUT, if the cached item type is a symlink, we can not be 100% that
    // it is not a directory (even with mimetype guessing), so ignore the
    // cache.
    if (cachedItem->type() == 'l')
      is_dir = protoCwd(url.path());
  } else {
    is_dir = protoCwd(url.path());
  }

  return is_dir;
}

void Socket::recursiveDelete(const KURL &url)
{
  if (m_stateInfo.abortInProgress()) return;
  
  // Temporarily disable the error handler
  m_errorHandler->setHandlerEnabled(false);
  
  // Get the directory listing
  FTPDirList *tmp = 0;
  
  try {
    tmp = dirList(url, true);
  } catch (...) {
    m_stateInfo.enterSocketState(S_IDLE);
  }
  
  // Reenable the error handler
  m_errorHandler->setHandlerEnabled(true);
  
  if (!tmp) return;
  
  FTPDirList p_list = *tmp;
  
  FTPDirList::iterator end( p_list.end() );
  for(FTPDirList::iterator i( p_list.begin() );i != end; ++i) {
    if (m_stateInfo.abortInProgress()) return;
    
    if ((*i).type() == 'd') {
      // Directory
      KURL next = url; next.cd((*i).name());

      // Call this function in recursion
      recursiveDelete(next);
      protoRmdir(next.path());
    } else {
      // File
      KURL next = url; next.addPath((*i).name());
      protoDelete(next.path());
    }
  }
}

void Socket::recursiveScan(KFTPQueue::Transfer *parent)
{
  m_stateInfo.enterSocketState(S_SCAN, true);
  recRecursiveScan(parent);
  m_stateInfo.enterSocketState(S_IDLE, true);
}

bool Socket::recRecursiveScan(KFTPQueue::Transfer *parent)
{
  // Scan the directory
  KURL url = parent->getSourceUrl();
  
  // Temporarily disable the error handler
  m_errorHandler->setHandlerEnabled(false);
  
  // Get the directory listing
  FTPDirList *tmp = 0;
  
  try {
    tmp = dirList(url, true);
  } catch (...) {
    m_stateInfo.enterSocketState(S_IDLE);
  }
  
  // Reenable the error handler
  m_errorHandler->setHandlerEnabled(true);
  
  if (!tmp)
    return false;
  
  // Sort the list by priority
  FTPDirList p_list = *tmp;
  qHeapSort(p_list);
  
  FTPDirList::iterator end( p_list.end() );
  for (FTPDirList::iterator i( p_list.begin() );i != end; ++i) {
    if (m_stateInfo.abortInProgress()) return false;
    
    KURL srcUrl = url;
    srcUrl.addPath((*i).name());
    
    KURL dstUrl = parent->getDestUrl();
    dstUrl.addPath((*i).name());

    if ((*i).type() == 'd') {
      // Directory
      KFTPQueue::TransferDir *transfer = new KFTPQueue::TransferDir(parent);
      transfer->setSourceUrl(srcUrl);
      transfer->setDestUrl(dstUrl);
      transfer->setTransferType(parent->getTransferType());
      transfer->setId(KFTPQueue::Manager::self()->m_lastQID++);
      
      qApp->lock();
      if (!m_stateInfo.abortInProgress())
        emit KFTPQueue::Manager::self()->newTransfer(transfer);
      qApp->unlock();

      // Call this function in recursion
      bool ok = recRecursiveScan(transfer);
      
      if (!ok || (KFTPCore::Config::skipEmptyDirs() && !transfer->children())) {
        qApp->lock();
        KFTPQueue::Manager::self()->removeTransfer(transfer);
        qApp->unlock();
      }
    } else {
      // Spawn a new child transfer
      if (KFTPCore::Config::enablePrioList() && KFTPCore::Config::skipNoQueue() && KFTPCore::Config::self()->getFilePriority(srcUrl.path()) == PRIO_SKIP)
        continue;
        
      if (KFTPCore::Config::skipEmptyFiles() && (*i).size() == 0)
        continue;
        
      KFTPQueue::TransferFile *transfer = new KFTPQueue::TransferFile(parent);
      transfer->setSourceUrl(srcUrl);
      transfer->setDestUrl(dstUrl);
      transfer->addSize((*i).size());
      transfer->setTransferType(parent->getTransferType());
      transfer->setId(KFTPQueue::Manager::self()->m_lastQID++);
      
      qApp->lock();
      if (!m_stateInfo.abortInProgress())
        emit KFTPQueue::Manager::self()->newTransfer(transfer);
      qApp->unlock();
    }
  }
  
  return true;
}

void Socket::rawCommand(const QString &cmd)
{
  if (getFeatures() & SF_RAW_COMMAND)
    protoSendRawCommand(cmd);
}

void Socket::initConfig()
{
  m_sockConfig.clear();
  
  setConfig("can_epsv", 1);
  setConfig("can_pasv", 1);
  setConfig("feat_resume", 1);
  setConfig("feat_mdtm", 1);
  
  m_busy[0] = false;
  m_busy[1] = false;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////// SOCKET MANAGER ///////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

SocketManager::SocketManager(QObject *parent)
  : QObject(parent), m_socket(0)
{
  // Add all the supported protocols
  addProtocol(new FtpSocket(this));
  addProtocol(new SftpSocket(this));
}

void SocketManager::addProtocol(Socket *socket)
{
  m_socketList.append(socket);
}

void SocketManager::setProtocol(KURL url)
{
  if (url.protocol() == m_protocol)
    return;
    
  m_protocol = url.protocol();
  
  Socket *socket;
  for (socket = m_socketList.first(); socket; socket = m_socketList.next()) {
    if (socket->getProtocol() == m_protocol) {
      m_socket = socket;
      return;
    }
  }
}

}

#include "socket.moc"
