/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2004 by the KFTPGrabber developers
 * Copyright (C) 2003-2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "kftpsession.h"
#include "kftpapi.h"
#include "browser/listview.h"
#include "browser/treeview.h"
#include "kftpbookmarks.h"
#include "misc.h"
#include "kftpsystemtray.h"
#include "errorhandler.h"

#include "misc/config.h"

#include <qdir.h>
#include <qobjectlist.h>

#include <kmessagebox.h>
#include <klocale.h>
#include <kdebug.h>
#include <kpassdlg.h>

using namespace KFTPGrabberBase;

KFTPSessionManager *FTPSessionManager;

//////////////////////////////////////////////////////////////////
//////////////////    KFTPSessionConnection     //////////////////
//////////////////////////////////////////////////////////////////

KFTPSessionConnection::KFTPSessionConnection(KFTPSession *session, bool core)
  : QObject(session), m_busy(false), m_aborting(false)
{
  // Create the actual connection client
  m_client = new KFTPClientThr(this);

 // If this is not a core session connection, connect
  if (!core) {
    // Connect signals
    connect(session->getClient(), SIGNAL(disconnectDone()), this, SLOT(slotSessionDisconnected()));

    // Connect to the server
    KURL url = session->getClient()->getClient()->getClientInfoUrl();

    FTPBookmarkManager->setupClient(session->getSite(), m_client);
    m_client->connectToHost(url);
  }
}

KFTPSessionConnection::~KFTPSessionConnection()
{
}

bool KFTPSessionConnection::isConnected()
{
  return !static_cast<KFTPSession*>(parent())->isRemote() || m_client->getClient()->isConnected();
}

void KFTPSessionConnection::acquire(KFTPQueue::Transfer *transfer)
{
  if (m_busy || !static_cast<KFTPSession*>(parent())->isRemote())
    return;

  m_curTransfer = transfer;
  m_busy = true;

  connect(transfer, SIGNAL(transferComplete(long)), this, SLOT(slotTransferCompleted()));
  connect(transfer, SIGNAL(transferAbort(long)), this, SLOT(slotTransferCompleted()));

  emit connectionAcquired();
}

void KFTPSessionConnection::remove()
{
  // Disconnect all signals from the transfer
  if (m_curTransfer)
    m_curTransfer->QObject::disconnect(this);

  m_curTransfer = 0L;
  m_busy = false;

  emit connectionRemoved();
  emit static_cast<KFTPSession*>(parent())->freeConnectionAvailable();
}

void KFTPSessionConnection::abort()
{
  if (m_aborting || !m_client->getClient()->isBusy()) return;

  // Emit the signal before aborting
  emit aborting();

  // Abort transfer
  m_aborting = true;
  m_client->getClient()->abort();
  m_aborting = false;
}

void KFTPSessionConnection::scanDirectory(KFTPQueue::Transfer *parent)
{
  // Lock the connection and the transfer
  acquire(parent);
  parent->lock();
  
  connect(m_client, SIGNAL(finished(CommandType)), this, SLOT(slotCommandCompleted(CommandType)));
  
  if (isConnected()) {
    m_client->scanDirectory(parent);
  } else {
    connect(m_client, SIGNAL(loginComplete(bool)), this, SLOT(slotSocketConnected(bool)));
  }
}

void KFTPSessionConnection::slotSocketConnected(bool success)
{
  m_client->QObject::disconnect(this, SLOT(slotSocketConnected(bool)));
  
  if (success) {
    m_client->scanDirectory(m_curTransfer);
  } else {
    // We were unable to establish a connection, scanning should be aborted
    m_curTransfer->unlock();
    remove();
    
    emit static_cast<KFTPSession*>(parent())->dirScanDone();
  }
}

void KFTPSessionConnection::slotCommandCompleted(CommandType type)
{
  if (type == TCMD_SCAN) {
    m_client->QObject::disconnect(this, SLOT(slotSocketConnected(bool)));
    m_client->QObject::disconnect(this, SLOT(slotCommandCompleted(CommandType)));
    
    m_curTransfer->unlock();
    remove();
    emit static_cast<KFTPSession*>(parent())->dirScanDone();
  }
}

void KFTPSessionConnection::slotSessionDisconnected()
{
  // Parent session has disconnected, we should disconnect as well
  m_client->KFTPClientThr::disconnect();
}

void KFTPSessionConnection::slotTransferCompleted()
{
  // Remove the lock
  remove();
}

////////////////////////////////////////////////////////
//////////////////    KFTPSession     //////////////////
////////////////////////////////////////////////////////

KFTPSession::KFTPSession(SessionSide side)
  : QObject(), m_side(side), m_remote(false), m_aborting(false), m_site(0), m_scannerThread(0)
{
  // Register this session
  FTPSessionManager->registerSession(this);

  connect(getClient(), SIGNAL(finished(CommandType)), this, SLOT(slotRemoteCommandDone(CommandType)));
}

KFTPSession::~KFTPSession()
{
  disconnect(getClient(), SIGNAL(finished(CommandType)), this, SLOT(slotRemoteCommandDone(CommandType)));
}

KFTPClientThr *KFTPSession::getClient()
{
  // Return the first (core) connection's client
  return m_connections.at(0)->getClient();
}

bool KFTPSession::isConnected()
{
  if (!m_remote)
    return true;

  return getClient()->getClient()->isConnected() || m_fileView->m_list->isOfflineMode();
}

void KFTPSession::slotClientConnected(bool success)
{
  if (success) {
    m_remote = true;
    m_aborting = false;
    m_lastUrl = getClient()->getClient()->getClientInfoUrl();
    
    QString siteName;
    if (m_site)
      siteName = m_site->getAttribute("name");
    else
      siteName = getClient()->getClient()->getClientInfoUrl().host();

    FTPSessionManager->getTabs(m_side)->changeTab(m_fileView, siteName);
    FTPSessionManager->getStatTabs()->changeTab(m_log, i18n("Log (%1)").arg(siteName));
    FTPSessionManager->getStatTabs()->showPage(m_log);
    FTPSessionManager->doEmitUpdate();

  }
}

void KFTPSession::slotClientDisconnected()
{
  m_remote = false;
  m_aborting = false;

  FTPSessionManager->getTabs(m_side)->changeTab(m_fileView, i18n("Local Session"));
  FTPSessionManager->getStatTabs()->changeTab(m_log, "[" + i18n("Log") + "]");
  FTPSessionManager->doEmitUpdate();
}

void KFTPSession::slotClientRetrySuccess()
{
  if (KFTPCore::Config::showRetrySuccessBalloon()) {
    s_sysTray->showBalloon(i18n("Connection with %1 has been successfully established.").arg(getClient()->getClient()->getClientInfoUrl().host()));
  }
}

void KFTPSession::scanDirectory(KFTPQueue::Transfer *parent)
{
  // Go trough all files in path and add them as transfers that have parent as their parent
  // transfer
  KURL path = parent->getSourceUrl();

  if (path.isLocalFile()) {
    // Lock the transfer
    parent->lock();
    
    // Create a new thread for local scans
    m_scannerThread = new KFTPQueue::QueueScannerThread(this, parent);
    m_scannerThread->start();
  } else if (m_remote) {
    // Check if we are in offline mode
    if (m_fileView->getListView()->isOfflineMode()) {
      emit dirScanDone();
      return;
    }

    // There should be a free connection or we are done :)
    if (!isFreeConnection()) {
      emit dirScanDone();
      return;
    }
    
    // Assign a new connection (it might be unconnected!)
    KFTPSessionConnection *conn = assignConnection();
    conn->scanDirectory(parent);
  }
}

void KFTPSession::customEvent(QCustomEvent *e)
{
  if (e->type() == QS_THR_EVENT_ID) {
    KFTPQueue::Transfer *transfer = static_cast<KFTPQueue::Transfer*>(e->data());
    
    // Unlock the transfer
    transfer->unlock();
    
    // Queue scanner thread has finished
    emit dirScanDone();
  }
}

void KFTPSession::slotRemoteCommandDone(CommandType type)
{
  // We should only do refreshes if the queue is not being processed
  bool shouldDoRefresh = KFTPQueue::Manager::self()->getNumRunning(true) == 0;
  
  // Refresh the view when certain commands are completed
  if (type == TCMD_PUT || type == TCMD_CHMOD || type == TCMD_MKDIR || type == TCMD_RENAME || type == TCMD_REMOVE ||
      type == TCMD_FXP && shouldDoRefresh) {
    m_fileView->getListView()->moveTo(KFTPWidgets::Browser::ListView::Reload);
  }
}

void KFTPSession::abort()
{
  if (m_aborting) return;

  if (m_scannerThread && m_scannerThread->running())
    m_scannerThread->terminate();

  m_aborting = true;

  emit aborting();

  // Abort all connections
  KFTPSessionConnection *conn;
  for (conn = m_connections.first(); conn; conn = m_connections.next()) {
    conn->abort();
  }

  m_aborting = false;
}

void KFTPSession::reconnect(const KURL &url)
{
  // Set the reconnect url
  m_reconnectUrl = url;
  
  if (m_remote && getClient()->getClient()->isConnected()) {
    abort();
    
    connect(getClient(), SIGNAL(disconnectDone()), this, SLOT(slotStartReconnect()));
    getClient()->KFTPClientThr::disconnect();
  } else {
    // The session is already disconnected, just call the slot
    slotStartReconnect();
  }
}

void KFTPSession::slotStartReconnect()
{
  disconnect(getClient(), SIGNAL(disconnectDone()), this, SLOT(slotStartReconnect()));
  
  if (!m_reconnectUrl.isLocalFile()) {
    // Reconnect only if this is a remote url
    if (m_site)
      m_fileView->getListView()->setHomeURL(remoteUrl(m_site->getProperty("defremotepath"), m_reconnectUrl));
    else
      m_fileView->getListView()->setHomeURL(KURL());
      
    FTPBookmarkManager->setupClient(m_site, getClient());
    getClient()->connectToHost(m_reconnectUrl);
  }
  
  // Invalidate the url
  m_reconnectUrl = KURL();
}

int KFTPSession::getMaxThreadCount()
{
  // First get the global thread count
  int t_count = KFTPCore::Config::threadCount();

  // Try to see if threads are disabled for this site
  if (t_count > 1 && isRemote()) {
    if (m_site && m_site->getIntProperty("disableThreads"))
      return 1;
  }

  return t_count;
}

bool KFTPSession::isFreeConnection()
{
  unsigned int c_max = getMaxThreadCount();
  unsigned int c_free = 0;

  if ((m_connections.count() < c_max && c_max > 1) || !isRemote())
    return true;

  KFTPSessionConnection *conn;
  for (conn = m_connections.first(); conn; conn = m_connections.next()) {
    if (!conn->isBusy())
      c_free++;
  }

  return c_free > 0;
}

KFTPSessionConnection *KFTPSession::assignConnection()
{
  unsigned int c_max = getMaxThreadCount();

  if (m_connections.count() == 0) {
    // We need a new core connection
    KFTPSessionConnection *conn = new KFTPSessionConnection(this, true);
    m_connections.append(conn);

    FTPSessionManager->doEmitUpdate();

    return conn;
  } else {
    // Find a free connection
    KFTPSessionConnection *conn;
    for (conn = m_connections.first(); conn; conn = m_connections.next()) {
      if (!conn->isBusy())
        return conn;
    }

    // No free connection has been found, but we may be able to create
    // another (if we are within limits)
    if (m_connections.count() < c_max) {
      conn = new KFTPSessionConnection(this);
      m_connections.append(conn);

      FTPSessionManager->doEmitUpdate();

      return conn;
    }
  }

  return 0;
}

void KFTPSession::disconnectAllConnections()
{
  KFTPSessionConnection *conn;

  for (conn = m_connections.first(); conn; conn = m_connections.next()) {
    if (conn->getClient()->getClient()->isConnected()) {
      conn->getClient()->KFTPClientThr::disconnect();
    }
  }
}

void KFTPSession::slotErrorHandler(KFTPNetwork::Error error)
{
  KFTPNetwork::ErrorHandler *handler = error.handler();

  // We are handling the error
  long id = handler->handlingError(error);

  switch (error.code()) {
    case KFTPNetwork::EC_TIMEOUT: {
      // The client connection has timed out. Reenable the directory
      // views, since they haven't been reenabled because the command
      // was interrupted.
      m_fileView->updateActions();
      break;
    }
    case KFTPNetwork::EC_SSH_PKEY_PASSWD: {
      QCString p_pass;
      int ret = KPasswordDialog::getPassword(p_pass, i18n("Please provide your private key decryption password."));
      
      // We've set up the password, so we should reconnect
      if (ret == KPasswordDialog::Accepted) {
        getClient()->getClient()->setConfig("pkey_pass", QString(p_pass));
        getClient()->connectToHost(getClient()->getClient()->getClientInfoUrl());
      }
      break;
    }
    default: {
      // We haven't handled the error
      handler->abandonHandler(id);
      break;
    }
  }
}

////////////////////////////////////////////////////////
////////////////// KFTPSessionManager //////////////////
////////////////////////////////////////////////////////

KFTPSessionManager::KFTPSessionManager(QObject *parent, QTabWidget *stat, KFTPTabWidget *left, KFTPTabWidget *right)
  : QObject(parent), m_statTabs(stat), m_leftTabs(left), m_rightTabs(right), m_currentSide(IGNORE_SIDE)
{
  // Init some stuff
  m_leftActive = 0L;
  m_rightActive = 0L;

  // Connect some signals
  connect(left, SIGNAL(currentChanged(QWidget*)), this, SLOT(slotActiveChanged(QWidget*)));
  connect(right, SIGNAL(currentChanged(QWidget*)), this, SLOT(slotActiveChanged(QWidget*)));

  connect(left, SIGNAL(closeRequest(QWidget*)), this, SLOT(slotSessionCloseRequest(QWidget*)));
  connect(right, SIGNAL(closeRequest(QWidget*)), this, SLOT(slotSessionCloseRequest(QWidget*)));

  connect(left, SIGNAL(mouseEvent(QString)), this, SLOT(slotSideChanged(QString)));
  connect(right, SIGNAL(mouseEvent(QString)), this, SLOT(slotSideChanged(QString)));
}

void KFTPSessionManager::registerSession(KFTPSession *session)
{
  // Create some new stuff and assign it to the session
  session->assignConnection();
  session->m_fileView = new KFTPWidgets::Browser::View(0L, "", session->getClient(), session);
  session->m_log = new KFTPWidgets::LogView();

  // KFTPFileDirListView - Item on the List was clicked
  connect(session->getFileView()->getListView(), SIGNAL(clicked(QListViewItem*)), this, SLOT(slotSideChanged(QListViewItem*)));

  // KFTPFileDirTreeView - Item on the list was clicked
  connect(session->getFileView()->getTreeView(), SIGNAL(clicked(QListViewItem*)), this, SLOT(slotSideChanged(QListViewItem*)));

  // KFTPFileDirView - First Toolbar clicked
  connect(session->getFileView()->m_toolBarFirst, SIGNAL(clicked(int)), this, SLOT(slotSideChanged(int)));

  // KFTPFileDirView - Second Toolbar clicked
  connect(session->getFileView()->m_toolBarSecond, SIGNAL(clicked(int)), this, SLOT(slotSideChanged(int)));

  // Connect some signals
  connect(session->getClient(), SIGNAL(logUpdate(int, const QString&)), session->m_log, SLOT(ftpLog(int, const QString&)));
  connect(session->getClient(), SIGNAL(loginComplete(bool)), session, SLOT(slotClientConnected(bool)));
  connect(session->getClient(), SIGNAL(disconnectDone()), session, SLOT(slotClientDisconnected()));
  connect(session->getClient(), SIGNAL(retrySuccess()), session, SLOT(slotClientRetrySuccess()));
  connect(session->getClient(), SIGNAL(errorHandler(KFTPNetwork::Error)), session, SLOT(slotErrorHandler(KFTPNetwork::Error)));

  // Assign GUI positions
  m_statTabs->addTab(session->m_log, "[" + i18n("Log") + "]");
  getTabs(session->m_side)->addTab(session->m_fileView, KFTPGrabberBase::loadSmallIcon("system"), i18n("Session"));

  // Actually add the session
  m_sessionList.append(session);
}

void KFTPSessionManager::unregisterSession(KFTPSession *session)
{
  // Destroy all objects related to the session and remove it
  getTabs(session->m_side)->removePage(session->m_fileView);
  m_statTabs->removePage(session->m_log);

  if (session->getClient()->getClient()->isConnected()) {
    session->abort();
    session->getClient()->KFTPClientThr::disconnect();
  }

  // Delete objects
  session->m_fileView->deleteLater();
  session->m_log->deleteLater();

  // Actually remove the session
  m_sessionList.remove(session);
  delete session;
  
  // Emit update
  emit update();
}

void KFTPSessionManager::doEmitUpdate()
{
  emit update();
}

KFTPSession *KFTPSessionManager::find(KFTPClientThr *client)
{
  KFTPSession *i;

  for (i = m_sessionList.first(); i; i = m_sessionList.next()) {
    if (i->getClient() == client)
      return i;
  }

  return 0L;
}

KFTPSession *KFTPSessionManager::find(KFTPWidgets::Browser::View *fileView)
{
  KFTPSession *i;

  for (i = m_sessionList.first(); i; i = m_sessionList.next()) {
    if (i->m_fileView == fileView)
      return i;
  }

  return 0L;
}

KFTPSession *KFTPSessionManager::find(KFTPWidgets::LogView *log)
{
  KFTPSession *i;

  for (i = m_sessionList.first(); i; i = m_sessionList.next()) {
    if (i->m_log == log)
      return i;
  }

  return 0L;
}

KFTPSession *KFTPSessionManager::find(const KURL &url, bool mustUnlock)
{
  if (url.isLocalFile())
    return find(true);

  KFTPSession *i;

  for (i = m_sessionList.first(); i; i = m_sessionList.next()) {
    KURL tmp = i->getClient()->getClient()->getClientInfoUrl();
    tmp.setPath(url.path());

    if (tmp == url && i->isRemote() && i->isConnected() && (!mustUnlock || i->isFreeConnection()))
      return i;
  }
  
  return 0L;
}

KFTPSession *KFTPSessionManager::find(bool local)
{
  KFTPSession *i;

  for (i = m_sessionList.first(); i; i = m_sessionList.next()) {
    if (i->m_remote != local)
      return i;
  }

  return 0L;
}

KFTPSession *KFTPSessionManager::findLast(const KURL &url, SessionSide side)
{
  if (url.isLocalFile())
    return find(true);

  KFTPSession *i;

  for (i = m_sessionList.first(); i; i = m_sessionList.next()) {
    KURL tmp = i->m_lastUrl;
    tmp.setPath(url.path());

    if (tmp == url && !i->isRemote() && (i->getSide() || side == IGNORE_SIDE))
      return i;
  }
  
  return 0L;
}

KFTPSession *KFTPSessionManager::spawnLocalSession(SessionSide side, bool forceNew)
{
  // Creates a new local session
  KFTPSession *session = 0L;

  if (forceNew || (session = find(true)) == 0L || (session->m_side != side && side != IGNORE_SIDE)) {
    side = side == IGNORE_SIDE ? LEFT_SIDE : side;

    session = new KFTPSession(side);
    session->m_remote = false;
    getTabs(side)->changeTab(session->m_fileView, i18n("Local Session"));
    getStatTabs()->changeTab(session->m_log, "[" + i18n("Log") + "]");
  }

  setActive(session);
  return session;
}

KFTPSession *KFTPSessionManager::spawnRemoteSession(SessionSide side, const KURL &remoteUrl, KFTPBookmarks::Site *site, bool mustUnlock)
{
  // Creates a new remote session and connects it to the correct server
  KFTPSession *session;

  if (remoteUrl.isLocalFile())
    return spawnLocalSession(side);

  if ((session = find(remoteUrl, mustUnlock)) == 0L || (session->m_side != side && side != IGNORE_SIDE)) {
    // Try to find the session that was last connected to this URL and
    // if that fails, spawn a new one.
    if ((session = findLast(remoteUrl, side)) == 0L) {
      side = side == IGNORE_SIDE ? RIGHT_SIDE : side;
      session = new KFTPSession(side);
    }
  
    // Try to find the site by url if it is not set
    if (!site)
      site = FTPBookmarkManager->findSite(remoteUrl);

    // Set properties
    session->m_remote = true;
    session->m_site = site;

    if (site) {
      session->m_fileView->getListView()->setHomeURL(KFTPGrabberBase::remoteUrl(site->getProperty("defremotepath"), remoteUrl));
      FTPBookmarkManager->setupClient(site, session->getClient());
    } else {
      session->m_fileView->getListView()->setHomeURL(KURL());
      FTPBookmarkManager->setupClient(0, session->getClient());
    }
    session->getClient()->connectToHost(remoteUrl);
  }

  return session;
}

void KFTPSessionManager::setActive(KFTPSession *session)
{
  // Make a session active on its own side ;)
  KFTPSession *oldActive = getActive(session->m_side);

  oldActive ? oldActive->m_active = false : 0;
  session->m_active = true;

  // Set dir view companions
  if (oldActive) oldActive->m_fileView->setCompanion(0L);

  if (getActive(OPPOSITE_SIDE(session->m_side))) {
    session->m_fileView->setCompanion(getActive(OPPOSITE_SIDE(session->m_side))->m_fileView);
    getActive(OPPOSITE_SIDE(session->m_side))->m_fileView->setCompanion(session->m_fileView);
  } else {
    session->m_fileView->setCompanion(0L);
  }

  switch (session->m_side) {
    case LEFT_SIDE: m_leftActive = session; break;
    case RIGHT_SIDE: m_rightActive = session; break;
    case IGNORE_SIDE: qDebug("Invalid side specified!"); return;
  }

  // Refresh the GUI
  getTabs(session->m_side)->showPage(session->m_fileView);
}

KFTPSession *KFTPSessionManager::getActive(SessionSide side)
{
  switch (side) {
    case LEFT_SIDE: return m_leftActive;
    case RIGHT_SIDE: return m_rightActive;
    case IGNORE_SIDE: qDebug("Invalid side specified!"); break;
  }

  return NULL;
}

KFTPSession *KFTPSessionManager::getActiveSide()
{
  if (m_currentSide != IGNORE_SIDE) {
    return getActive(m_currentSide);
  } else {
    // If there is a remote session in the current view, this might
    // be it (even if it is not active).
    if (getActive(LEFT_SIDE)->isRemote())
      return getActive(LEFT_SIDE);
    else if (getActive(RIGHT_SIDE)->isRemote())
      return getActive(RIGHT_SIDE);
  }

  qWarning("[ERROR] No session side is selected! Possible bug!");
  return NULL;
}

KFTPTabWidget *KFTPSessionManager::getTabs(SessionSide side)
{
  switch (side) {
    case LEFT_SIDE: return m_leftTabs;
    case RIGHT_SIDE: return m_rightTabs;
    case IGNORE_SIDE: qDebug("Invalid side specified!"); break;
  }

  return NULL;
}

void KFTPSessionManager::slotActiveChanged(QWidget *page)
{
  KFTPSession *session = find(static_cast<KFTPWidgets::Browser::View*>(page));
  setActive(session);
}

void KFTPSessionManager::slotSideChanged(QString name)
{
  if (name.contains("TabWidgetLeft"))
    m_currentSide = LEFT_SIDE;

  if (name.contains("TabWidgetRight"))
    m_currentSide = RIGHT_SIDE;
}

void KFTPSessionManager::slotSideChanged(QListViewItem*)
{
  QString name = QObject::sender()->parent()->parent()->parent()->parent()->name();
  this->slotSideChanged(name);
}

void KFTPSessionManager::slotSideChanged(int)
{
  QString name = QObject::sender()->parent()->parent()->parent()->name();
  this->slotSideChanged(name);
}

void KFTPSessionManager::slotSessionCloseRequest(QWidget *page)
{
  KFTPSession *session = find(static_cast<KFTPWidgets::Browser::View*>(page));

  if (getTabs(session->m_side)->count() == 1) {
    KMessageBox::error(0L, i18n("At least one session must remain open on each side."));
    return;
  }

  if ((session->m_remote && session->getClient()->getClient()->isBusy()) || !session->isFreeConnection()) {
    KMessageBox::error(0L, i18n("Please finish all transfers before closing the session."));
    return;
  } else {
    // Remove the session
    if (session->getClient()->getClient()->isConnected()) {
      if (KMessageBox::questionYesNo(0L, i18n("This session is currently connected. Are you sure you wish to disconnect?"), i18n("Close Session")) == KMessageBox::No)
        return;
    }

    unregisterSession(session);
  }
}

#include "kftpsession.moc"
