/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 *
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */
 
#include "configbase.h"
#include "config.h"
#include "kftpapi.h"

#include <qregexp.h>

#include <kapplication.h>
#include <kconfig.h>
#include <kemailsettings.h>
#include <klocale.h>

namespace KFTPCore {

BrowserColorizer::BrowserColorizer()
{
}

void BrowserColorizer::addFilePattern(const QString &pattern, const QColor &color)
{
  PatternStruct ps;
  ps.pattern = pattern;
  ps.color = color;
  
  m_filePatterns.append(ps);
}

QColor BrowserColorizer::getColor(const QString &name)
{
  if (!Config::enableHighlight()) return QColor();

  QRegExp e;
  e.setWildcard(true);
  for (QValueList<PatternStruct>::iterator i = m_filePatterns.begin(); i != m_filePatterns.end(); ++i) {
    e.setPattern((*i).pattern);
    
    if (e.exactMatch(name))
      return (*i).color;
  }

  return QColor();
}

ConfigBase::ConfigBase(const QString &fileName)
  : QObject(),
    KConfigSkeleton(fileName),
    m_colorizer(new BrowserColorizer()),
    m_priorityList(new QValueList<QPair<QString, int> >())
{
  m_fileExistsDownActions.setTypeText(i18n("Download"));
  m_fileExistsUpActions.setTypeText(i18n("Upload"));
  m_fileExistsFxpActions.setTypeText(i18n("FXP"));
  
  m_transMode = 'I';
}

void ConfigBase::postInit()
{
  refreshColorizer();
  
  // Restore the actions
  QString tmp = Config::downloadActions();
  tmp >> m_fileExistsDownActions;
  
  tmp = Config::uploadActions();
  tmp >> m_fileExistsUpActions;
  
  tmp = Config::fxpActions();
  tmp >> m_fileExistsFxpActions;
  
  // Load priority list
  loadPriorityList();
}

void ConfigBase::saveConfig()
{
  // Save actions before writing
  QString tmp;
  tmp << m_fileExistsDownActions;
  Config::setDownloadActions(tmp);
  
  tmp << m_fileExistsUpActions;
  Config::setUploadActions(tmp);
  
  tmp << m_fileExistsFxpActions;
  Config::setFxpActions(tmp);
  
  // Save priority list
  savePriorityList();
  
  // Save the window's position
  Config::setSize(KFTPAPI::getInstance()->mainWindow()->size());
  Config::setPosition(KFTPAPI::getInstance()->mainWindow()->pos());
  
  // Write the config
  writeConfig();
}

void ConfigBase::emitChange()
{
  emit configChanged();
}

void ConfigBase::refreshColorizer()
{
  colorizer()->m_filePatterns.clear();
  
  QStringList list = Config::filterList();
  QStringList::Iterator end(list.end());
  for (QStringList::Iterator i(list.begin()); i != end; ++i) {
    QString pat = (*i).section(':', 0, 0);
    QString col = (*i).section(':', 1, 1);

    colorizer()->addFilePattern(pat, QColor(col));
  }
}

char ConfigBase::ftpMode(const QString &filename)
{
  // Get FTP mode (binary/ascii)
  switch (m_transMode) {
    case 'A': return 'A'; break;
    case 'I': return 'I'; break;
    case 'X':
    default: {
      char mode = 'I';
      QRegExp e;
      e.setWildcard(true);
      
      QStringList list = Config::asciiList();
      QStringList::iterator end(list.end());
      for (QStringList::iterator i(list.begin()); i != end; ++i) {
        e.setPattern((*i));
        
        if (e.exactMatch(filename)) {
          mode = 'A';
          break;
        }
      }
    
      return mode;
    }
  }
}

int ConfigBase::getFilePriority(const QString &filename)
{
  QRegExp e;
  e.setWildcard(true);
  
  QValueList<QPair<QString, int> > list = *m_priorityList;
  QValueList<QPair<QString, int> >::iterator end(list.end());
  for (QValueList<QPair<QString, int> >::iterator i(list.begin()); i != end; ++i) {
    QString pattern = (*i).first;
    int priority = (*i).second;
    e.setPattern(pattern);
  
    if (e.exactMatch(filename))
      return priority;
  }

  return PRIO_NOT_FOUND;
}

void ConfigBase::loadPriorityList()
{
  m_priorityList->clear();
  
  QStringList list = Config::prioList();
  QStringList::Iterator end(list.end());
  for (QStringList::Iterator i(list.begin()); i != end; ++i) {
    QString pat = (*i).section(':', 0, 0);
    QString prio = (*i).section(':', 1, 1);
    
    m_priorityList->append(QPair<QString, int>(pat, prio.toInt()));
  }
}

void ConfigBase::savePriorityList()
{
  QStringList slist;
  
  QValueList<QPair<QString, int> > *list = m_priorityList;
  QValueList<QPair<QString, int> >::iterator end(list->end());
  for (QValueList<QPair<QString, int> >::iterator i(list->begin()); i != end; ++i) {
    slist.append(QString("%1:%2").arg((*i).first).arg((*i).second));
  }
  
  Config::setPrioList(slist);
}

QString ConfigBase::getGlobalMail()
{
  KEMailSettings kes;
  kes.setProfile(kes.defaultProfileName());
  return kes.getSetting(KEMailSettings::EmailAddress);
}

}

#include "configbase.moc"
