/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2004 by the KFTPGrabber developers
 * Copyright (C) 2004 Jernej Kos <kostko@jweb-network.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#include "browser/propsplugin.h"
#include "kftpsession.h"

#include <qlayout.h>
#include <qlabel.h>
#include <qstyle.h>
#include <qgroupbox.h>

#include <klocale.h>
#include <kiconloader.h>
#include <kseparator.h>
#include <ksqueezedtextlabel.h>

namespace KFTPWidgets {

namespace Browser {

PropsPlugin::PropsPlugin(KPropertiesDialog *_props, FTPDirectoryItem item, KURL url)
 : KPropsDlgPlugin(_props), m_dirObject(item), m_url(url)
{
  QFrame *frame = properties->addPage(i18n("&General"));
  frame->setMinimumWidth(320);
  frame->setMinimumHeight(300);
  
  KURL fileUrl = url;
  QString iconText;
  QString mimeComment;
  bool isDir = false;
  
  // Guess file type
  if (m_dirObject.type() == 'd') {
    iconText = "folder";
    isDir = true;
    mimeComment = i18n("Remote folder");
  } else if (m_dirObject.type() == 'l') {
    // We can't know if the sym-linked file is realy a directory, but most of
    // the time it is. So if we can't determine the MIME type, set it to directory.
    KMimeType::Ptr mimeType = KMimeType::findByURL(fileUrl, 0, false, true);
    if (mimeType->name() == KMimeType::defaultMimeType()) {
      iconText = "folder";
      isDir = true;
      mimeComment = i18n("Remote folder");
    } else {
      iconText = mimeType->icon(QString::null, false);
      mimeComment = mimeType->comment();
    }
  } else {
    KMimeType::Ptr mimeType = KMimeType::findByURL(fileUrl, 0, false, true);
    iconText = mimeType->icon(QString::null, false);
    mimeComment = mimeType->comment();
  }
  
  if (mimeComment.isEmpty()) {
    mimeComment = i18n("Unknown");
  }
  
  QVBoxLayout *vbl = new QVBoxLayout(frame, 0, KDialog::spacingHint(), "vbl");
  QGridLayout *grid = new QGridLayout(0, 3);
  grid->setColStretch(0, 0);
  grid->setColStretch(1, 0);
  grid->setColStretch(2, 1);
  grid->addColSpacing(1, KDialog::spacingHint());
  vbl->addLayout(grid);
  
  // BEGIN first part
  QLabel *iconLabel = new QLabel(frame);
  int bsize = 66 + 2 * iconLabel->style().pixelMetric(QStyle::PM_ButtonMargin);
  iconLabel->setFixedSize(bsize, bsize);
  iconLabel->setPixmap(DesktopIcon(iconText));
  grid->addWidget(iconLabel, 0, 0, AlignLeft);
  
  QLabel *nameLabel = new QLabel(frame);
  nameLabel->setText(item.name());
  grid->addWidget(nameLabel, 0, 2);
  
  KSeparator *sep = new KSeparator(KSeparator::HLine, frame);
  grid->addMultiCellWidget(sep, 2, 2, 0, 2);
  // END first part
  
  // BEGIN second part
  QLabel *l = new QLabel(i18n("Type:"), frame);
  grid->addWidget(l, 3, 0);
  
  l = new QLabel(mimeComment, frame);
  grid->addWidget(l, 3, 2);
  
  l = new QLabel(i18n("Location:"), frame);
  grid->addWidget(l, 4, 0);
  
  l = new KSqueezedTextLabel(frame);
  l->setText(fileUrl.directory());
  grid->addWidget(l, 4, 2);
  
  l = new QLabel(i18n("Size:"), frame);
  grid->addWidget(l, 5, 0);
  
  l = new QLabel(frame);
  grid->addWidget(l, 5, 2);
  
  l->setText(QString::fromLatin1("%1 (%2)").arg(KIO::convertSize(item.size()))
             .arg(KGlobal::locale()->formatNumber(item.size(), 0)));
             
  sep = new KSeparator(KSeparator::HLine, frame);
  grid->addMultiCellWidget(sep, 6, 6, 0, 2);
  // END second part
  
  // BEGIN third part
  l = new QLabel(i18n("Created:"), frame);
  grid->addWidget(l, 7, 0);
  
  QDateTime dt;
  dt.setTime_t(item.date());
  l = new QLabel(KGlobal::locale()->formatDateTime(dt), frame);
  grid->addWidget(l, 7, 2);
  // END third part
  
  vbl->addStretch(1);
}


PropsPlugin::~PropsPlugin()
{
}

void PropsPlugin::applyChanges()
{
}

PermissionsPropsPlugin::PermissionsPropsPlugin(KPropertiesDialog *_props, FTPDirectoryItem item, KFTPSession *session,
                                                               KURL url)
  : KPropsDlgPlugin(_props), m_dirObject(item), m_session(session), m_url(url)
{
  QFrame *frame = properties->addPage(i18n("&Permissions"));
  frame->setMinimumWidth(320);
  frame->setMinimumHeight(300);
  
  KURL fileUrl = url;
  bool isDir = false;
  
  // Guess file type
  if (m_dirObject.type() == 'd') {
    isDir = true;
  } else if (m_dirObject.type() == 'l') {
    // We can't know if the sym-linked file is realy a directory, but most of
    // the time it is. So if we can't determine the MIME type, set it to directory.
    KMimeType::Ptr mimeType = KMimeType::findByURL(fileUrl, 0, false, true);
    if (mimeType->name() == KMimeType::defaultMimeType()) {
      isDir = true;
    }
  }
  
  QBoxLayout *box = new QVBoxLayout(frame, 0, KDialog::spacingHint());
  
  QGroupBox *gb = new QGroupBox(0, Qt::Vertical, i18n("Access Permissions"), frame);
  gb->layout()->setSpacing(KDialog::spacingHint());
  gb->layout()->setMargin(KDialog::marginHint());
  box->addWidget(gb);
  
  QGridLayout *gl = new QGridLayout(gb->layout(), 6, 6, 15);
  
  QLabel *l = new QLabel(i18n("Class"), gb);
  gl->addWidget(l, 1, 0);
  
  if (isDir)
    l = new QLabel(i18n("Show\nEntries"), gb);
  else
    l = new QLabel(i18n("Read"), gb);
  gl->addWidget(l, 1, 1);
  
  if (isDir)
    l = new QLabel(i18n("Write\nEntries"), gb);
  else
    l = new QLabel(i18n("Write"), gb);
  gl->addWidget(l, 1, 2);
  
  if (isDir)
    l = new QLabel(i18n("Enter folder", "Enter"), gb);
  else
    l = new QLabel(i18n("Exec"), gb);
  QSize size = l->sizeHint();
  size.setWidth(size.width() + 15);
  l->setFixedSize(size);
  gl->addWidget(l, 1, 3);
  
  l = new QLabel(i18n("Special"), gb);
  gl->addMultiCellWidget(l, 1, 1, 4, 5);
  
  l = new QLabel(i18n("User"), gb);
  gl->addWidget(l, 2, 0);
  
  l = new QLabel(i18n("Group"), gb);
  gl->addWidget(l, 3, 0);
  
  l = new QLabel(i18n("Others"), gb);
  gl->addWidget(l, 4, 0);
  
  l = new QLabel(i18n("Set UID"), gb);
  gl->addWidget(l, 2, 5);
  
  l = new QLabel(i18n("Set GID"), gb);
  gl->addWidget(l, 3, 5);
  
  l = new QLabel(i18n("Sticky"), gb);
  gl->addWidget(l, 4, 5);
  
  m_perms[0] = item.permissions().mid(1, 3);
  m_perms[1] = item.permissions().mid(4, 3);
  m_perms[2] = item.permissions().mid(7, 3);
  
  // BEGIN checkboxes
  for (int row = 0; row < 3; ++row) {
    for (int col = 0; col < 4; ++col) {
      QCheckBox *cb = new QCheckBox(gb);
      connect(cb, SIGNAL(clicked()), this, SLOT(setDirty()));
      m_permsCheck[row][col] = cb;
      
      if (col < 2) {
        m_permsCheck[row][col]->setChecked(m_perms[row].at(col) != '-');
      } else if (col == 2) {
        // Execute bit (x/s/t)
        QChar m = m_perms[row].at(col);
        m_permsCheck[row][col]->setChecked(m == 'x' || m == 's' || m == 't');
      } else if (col == 3) {
        switch (row) {
          // SUID
          case 0: m_permsCheck[row][col]->setChecked(m_perms[0].lower().at(2) == 's'); break;
          
          // SGID
          case 1: m_permsCheck[row][col]->setChecked(m_perms[1].lower().at(2) == 's'); break;
          
          // STICKY
          case 2: m_permsCheck[row][col]->setChecked(m_perms[2].lower().at(2) == 't'); break; 
        }
      }
      
      gl->addWidget(cb, row+2, col+1);
    }
  }
  // END checkboxes  
  gl->setColStretch(6, 10);
  box->addStretch(10);
}

PermissionsPropsPlugin::~PermissionsPropsPlugin()
{
}

void PermissionsPropsPlugin::applyChanges()
{
  // Generate new permissions =)
  int newPerms[4] = {0,};
  
  for (int row = 0; row < 3; ++row) {
    for (int col = 0; col < 4; ++col) {
      if (!m_permsCheck[row][col]->isChecked()) continue;
      
      int x = col < 3 ? col : row;
      int c = 0;
      
      switch (x) {
        case 0: c = 4; break;
        case 1: c = 2; break;
        case 2: c = 1; break;
      }
      
      if (col < 3) {
        newPerms[row+1] += c;
      } else {
        newPerms[0] += c;
      }
    }
  }
  
  // Actually do a chmod
  int mode = newPerms[0] * 1000 + newPerms[1] * 100 + newPerms[2] * 10 + newPerms[3];
  m_session->getClient()->chmod(m_url, mode);
}

}

}

#include "propsplugin.moc"
