/*
 * This file is part of the KFTPGrabber project
 *
 * Copyright (C) 2003-2005 by the KFTPGrabber developers
 * Copyright (C) 2003-2005 Jernej Kos <kostko@jweb-network.net>
 * Copyright (C) 2004-2005 Max Howell <max.howell@methylblue.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Steet, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations
 * including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * file(s) with this exception, you may extend this exception to your
 * version of the file(s), but you are not obligated to do so.  If you
 * do not wish to do so, delete this exception statement from your
 * version.  If you delete this exception statement from all source
 * files in the program, then also delete it here.
 */

#ifndef KFTPWIDGETSSIDEBAR_H
#define KFTPWIDGETSSIDEBAR_H

#include <qwidget.h>
#include <qvaluevector.h>

class KURL;
class QSignalMapper;
class QVBox;

namespace KFTPWidgets {

typedef QValueVector<QWidget*> SidebarList;

class MultiTabBar;
class MultiTabBarTab;

/**
 * This class provides a sidebar. It uses a customised version of KMultiTabBar for
 * displaying the actual tabs. It uses a QSignalMapper to map signals from the
 * tabs to the actual sidebars.
 *
 * This class was copied from amaroK (BrowserBar) and modified to support bottom
 * sidebars as well.
 *
 * @author amaroK developers
 * @author Jernej Kos
 */
class Sidebar : public QWidget
{
Q_OBJECT
friend class Splitter;
public:
    enum Position {
      Left,
      Bottom
    };
    
    Sidebar(QWidget *parent, Position position);
    ~Sidebar();
    
    /**
     * Toggle visibility of this sidebar.
     *
     * @param visible True if the sidbar should be visible, false otherwise
     */
    void setVisible(bool visible);
    
    /**
     * Returns the content box of the sidebar. This should be where the widget, that
     * should be resized when the sidebar resizes, should be placed.
     *
     * @return A QVBox container widget
     */
    QVBox *content() { return m_content; }

    /**
     * Returns the sidebar by it's name.
     *
     * @param name The sidebar's name
     * @return A QWidget representing the sidebar or NULL if it is not found
     */
    QWidget *sidebar(const QString &name) const;
    
    /**
     * Returns the sidebar by it's index.
     *
     * @param index The sidebar's index
     * @return A QWidget representing the sidebar or NULL if the index is invalid
     */
    QWidget *sidebar(int index) const { if (index < 0) index = 0; return m_sidebars[index]; }
    
    /**
     * Returns the currently open sidebar.
     *
     * @return A QWidget representing the current sidebar
     */
    QWidget *currentSidebar() const { return sidebar(m_currentIndex); }

    /**
     * Adds a new sidebar widget.
     *
     * @param widget The widget to add
     * @param title The title that will be displayed on the tab
     * @param icon Name of the icon that will be displayed besides the title
     */
    void addSidebar(QWidget *widget, const QString &title, const QString &icon);
protected:
    virtual void polish();
    virtual bool event(QEvent *e);
public slots:
    void showSidebar(const QString &name) { showSidebar(indexForName(name)); }
    void showSidebar(int index) { if (index != m_currentIndex) showHideSidebar(index); }
    void showHideSidebar(int);
    void closeCurrentSidebar() { showHideSidebar(m_currentIndex); }
private:
    int indexForName(const QString&) const;
    void mouseMovedOverSplitter(QMouseEvent *e);

    void adjustWidgetSizes();
    uint maxSidebarWidth() const { return width() / 2; }
    uint maxSidebarHeight() const { return height() / 2; }

    Position m_position;
    uint m_pos;
    
    QWidget *m_divider;
    QVBox *m_content;
    MultiTabBar *m_tabBar;
    QWidget *m_sidebarBox;
    
    SidebarList m_sidebars;
    int m_currentIndex;
    int m_lastIndex;
    
    QSignalMapper *m_mapper;
};

}

#endif
