/***************************************************************************
 *   Copyright (C) 2007, 2008 by Marcel Hasler                             *
 *   mahasler@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "kgmailnotifier.h"
#include "iconpaths.h"
#include "configuration.h"

#include <kuniqueapplication.h>
#include <klocale.h>
#include <kprocess.h>
#include <kglobal.h>
#include <kmessagebox.h>

#include <iostream>
#include <cstdlib>
using namespace std;

KGmailNotifier::KGmailNotifier(AboutData *aboutData) : KUniqueApplication()
{
	cout << endl;
	cout << "Welcome to KGmailNotifier." << endl;
	cout << "(c) 2007, 2008 Marcel Hasler (mahasler@gmail.com)" << endl << endl;
	cout << "This software is released under the terms of the GNU General Public License Version 2." << endl << endl;
	cout << "Enjoy!" << endl << endl;

	//////////////////////////
	// Set application icon //
	//////////////////////////
	aboutData->setLogo(ICON_RED);


	//////////////////////////
	// Set up config window //
	//////////////////////////
	m_configWindow = 0; // Constructed dynamically in slotShowConfigWindow()


	/////////////////////////
	// Set up popup window //
	/////////////////////////
	m_popupWindow = new PopupWindow();
	connect(m_popupWindow, SIGNAL(urlClicked()), this, SLOT(slotLaunchBrowser()));


	////////////////////////////////////
	// Set up timer for checking mail //
	////////////////////////////////////
	m_checkMailTimer = new QTimer();
	connect(m_checkMailTimer, SIGNAL(timeout()), this, SLOT(slotCheckNewMail()));


	///////////////////////////////
	// Set up notification tools //
	///////////////////////////////
	m_audioPlayer = new AudioPlayer();
	m_ledHandler = new LEDHandler();


	//////////////////////
	// Set up tray icon //
	//////////////////////
	m_trayIcon = new TrayIcon("KGmailNotifier", ICON_SMALL, ICON_RED);

	m_trayIcon->addItemToMenu(ICON_CHECK, i18n("Check mail now"), this, SLOT(slotCheckNewMail()));
	m_trayIcon->addItemToMenu(ICON_INBOX, i18n("Go to inbox"), this, SLOT(slotLaunchBrowser()));
	m_trayIcon->addSeparatorToMenu();

	m_trayIcon->addItemToMenu(ICON_CONFIGURE, i18n("Configure"), this, SLOT(slotShowConfigWindow()));
	m_trayIcon->addItemToMenu(ICON_ABOUT, i18n("About KGmailNotifier"), this, SLOT(slotShowAboutWindow()));
	m_trayIcon->addSeparatorToMenu();

	m_silentNotificationMenuID = m_trayIcon->addItemToMenu(i18n("Silent notification"), this, SLOT(slotSilentNotificationSelected()));
	if (Configuration().readBoolEntry("SilentNotification"))
		m_trayIcon->setItemChecked(m_silentNotificationMenuID);

	m_trayIcon->addItemToMenu(ICON_QUIT, i18n("Quit"), this, SLOT(slotQuit()));

	m_trayIcon->setToolTip(i18n("No new messages"));
	connect(m_trayIcon, SIGNAL(trayIconClicked()), this, SLOT(slotShowPopupWindow()));


	/////////////////////////
	// Set up about window //
	/////////////////////////
	m_aboutWindow = new KAboutApplication(aboutData->data(), 0, "aboutWindow", false);


	////////////////////////////
	// Set up account manager //
	////////////////////////////
	m_accountManager = new AccountManager();
	connect(m_accountManager, SIGNAL(connectionSuccessful()), this, SLOT(slotProcessMailInfo()));
	connect(m_accountManager, SIGNAL(connectionError(const QString)), this, SLOT(slotProcessConnectionError(const QString)));

	if (!m_accountManager->isValid())
		this->slotShowConfigWindow();


	/////////////////////////////////
	// Done setting up. Check mail //
	/////////////////////////////////
	slotCheckNewMail();
}


KGmailNotifier::~KGmailNotifier()
{
	if (m_trayIcon)
		delete m_trayIcon;

	if (m_aboutWindow)
		delete m_aboutWindow;

	if (m_configWindow)
		delete m_configWindow;

	if (m_popupWindow)
		delete m_popupWindow;

	if (m_accountManager)
		delete m_accountManager;

	if (m_checkMailTimer)
		delete m_checkMailTimer;

	if (m_audioPlayer)
		delete m_audioPlayer;

	if (m_ledHandler)
		delete m_ledHandler;
}


///////////////////
// Private slots //
///////////////////
void KGmailNotifier::slotSilentNotificationSelected()
{
	m_trayIcon->setItemChecked(m_silentNotificationMenuID);
	Configuration().writeEntry("SilentNotification", m_trayIcon->itemChecked(m_silentNotificationMenuID));

	// Save configuration
	Configuration().writeToDisk();
}


void KGmailNotifier::slotShowPopupWindow()
{
	m_popupWindow->popup();
}


void KGmailNotifier::slotCheckNewMail()
{
	m_checkMailTimer->stop();
	m_trayIcon->setToolTip(i18n("Connecting to server..."));
	m_accountManager->checkMail();
}


void KGmailNotifier::slotProcessMailInfo()
{
	unsigned int msgCount = m_accountManager->messageCount();

	if (msgCount > 0)
	{
		m_trayIcon->setIcon(ICON_BLUE, Configuration().readBoolEntry("NumberOnSystray"), msgCount);
		m_trayIcon->setToolTip(i18n("%1 new message(s)").arg(msgCount));
		m_popupWindow->setMessageCount(msgCount);
		m_popupWindow->setSender(m_accountManager->lastSender());
		m_popupWindow->setSubject(m_accountManager->lastSubject());

		if (m_accountManager->hasNewMail())
		{
			if (!m_trayIcon->itemChecked(m_silentNotificationMenuID))
			{
				m_audioPlayer->play();
				if (Configuration().readBoolEntry("ShowPopup"))
					slotShowPopupWindow();
			}

			// Turn on LED
			m_ledHandler->setLedOn(true);
		}
	}

	else
	{
		m_trayIcon->setIcon(ICON_RED, false, msgCount);
		m_trayIcon->setToolTip(i18n("No new messages"));
		m_popupWindow->setMessageCount(0);

		// Turn off LED
		m_ledHandler->setLedOn(false);
	}

	// Restart timer
	m_checkMailTimer->start(Configuration().readNumEntry("CheckInterval") * 1000, true);
}


void KGmailNotifier::slotProcessConnectionError(const QString errorType)
{
	QString errorMessage;
	if (errorType == "timeout")
		errorMessage = i18n("Connection timeout!");
	else
		errorMessage = i18n("Connection failed!");

	m_trayIcon->setIcon(ICON_ALERT, false, 0);
	m_trayIcon->setToolTip(errorMessage);
	m_popupWindow->setConnectionFailed();
	
	// Restart timer
	m_checkMailTimer->start(Configuration().readNumEntry("CheckInterval") * 1000, true);
}


void KGmailNotifier::slotLaunchBrowser()
{
	KProcess browserProc;
	browserProc << Configuration().readEntry("BrowserPath");
	browserProc << "https://mail.google.com/mail";
	browserProc.start(KProcess::DontCare);
}


void KGmailNotifier::slotShowConfigWindow()
{
	m_configWindow = new ConfigurationDialog(0, "Configuration");
	connect(m_configWindow, SIGNAL(configurationUpdated()), this, SLOT(slotApplyConfiguration()));
	connect(m_configWindow, SIGNAL(finished()), this, SLOT(slotDeleteConfigWindow()));

	m_configWindow->show();
}


void KGmailNotifier::slotDeleteConfigWindow()
{
	m_configWindow->delayedDestruct();
}


void KGmailNotifier::slotApplyConfiguration()
{
	// Apply what's necessary when configuration has changed
	m_popupWindow->updateConfiguration();
	m_ledHandler->updateConfiguration();
	m_audioPlayer->updateConfiguration();
	m_accountManager->updateConfiguration();

	m_checkMailTimer->changeInterval(Configuration().readNumEntry("CheckInterval") * 1000);
}


void KGmailNotifier::slotShowAboutWindow()
{
	m_aboutWindow->show();
}


void KGmailNotifier::slotQuit()
{
	KConfig *config = KGlobal::config();
	config->setGroup("Notification Messages");

	if (config->readEntry("QuitDontAskAgain").isEmpty())
	{
		int result = KMessageBox::questionYesNo (0, i18n ("Are you sure you want to quit KGmailNotifier?"), i18n ("Question"),
			i18n ("Quit"), i18n ("Cancel"), "QuitDontAskAgain");
		Configuration().writeEntry("QuitDontAskAgain", config->readEntry("QuitDontAskAgain"));

		if (result == KMessageBox::No)
			return;
	}

	if (config->readEntry("AutostartDontAskAgain").isEmpty())
	{
		int result = KMessageBox::questionYesNo (0, i18n ("Start KGmailNotifier automatically when you log in?"), i18n ("Question"),
			i18n ("Start Automatically"), i18n ("Do Not Start"), "AutostartDontAskAgain");
		Configuration().writeEntry("Autostart", result == KMessageBox::Yes);
	}

	Configuration().writeToDisk();
	exit(0);
}

#include "kgmailnotifier.moc"
