/***************************************************************************
 *   Copyright (C) 2007, 2008 by Marcel Hasler                             *
 *   mahasler@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "popupwindow.h"
#include "configuration.h"
#include "iconpaths.h"

#include <qdesktopwidget.h>
#include <qapplication.h>

#include <klocale.h>

PopupWindow::PopupWindow()
	: QWidget(0, "KGmailNotifier", Qt::WType_TopLevel | Qt::WDestructiveClose | Qt::WX11BypassWM | Qt::WStyle_NoBorder)
{
	// Set up the window
	m_width = 200;
	m_height = 150;

	m_font = new QFont();
	m_font->setPixelSize(11);

	// Add the background image
	m_image = new QLabel(this, "gmailLabel");
	m_image->setPaletteBackgroundPixmap(PIC_BACKGROUND);
	m_image->setPixmap(PIC_BACKGROUND);
	m_image->setGeometry(0, 0, m_width, m_height);
	m_image->show();

	// Add close button
	m_closeButton = new QPushButton(m_image, "closeButton");
	m_closeButton->setIconSet(QIconSet(ICON_CLOSE));
	m_closeButton->setGeometry(QRect(178,10,12,12));
	m_closeButton->setFlat(true);
	m_closeButton->setAutoDefault(false);
	m_closeButton->show();
	connect(m_closeButton, SIGNAL(clicked()), this, SLOT(slotClose()));

	// Define new mail label
	m_newMailLabel = new KURLLabel(m_image, "newMailLabel");
	m_newMailLabel->setGeometry(10, 50, 180, 20);
	m_newMailLabel->setBackgroundOrigin(QWidget::ParentOrigin);
	m_newMailLabel->setAlignment(QLabel::SingleLine | QLabel::AlignCenter);
	m_newMailLabel->setFont(*m_font);
	m_newMailLabel->setSelectedColor(QColor("blue"));
	m_newMailLabel->setText(i18n("No new mail"));
	m_newMailLabel->show();
	connect(m_newMailLabel, SIGNAL(leftClickedURL(const QString&)), this, SLOT(slotProcessURLClicked()));

	// Define sender label
	m_senderLabel = new QLabel(m_image, "senderLabel");
	m_senderLabel->setGeometry(15, 70, 170, 35);
	m_senderLabel->setBackgroundOrigin(QWidget::ParentOrigin);
	m_senderLabel->setAlignment(QLabel::SingleLine | QLabel::AlignLeft | QLabel::AlignTop);
	m_senderLabel->setFont(*m_font);
	m_senderLabel->setText("");
	m_senderLabel->hide();

	// Define message title label
	m_subjectLabel = new QLabel(m_image, "subjectLabel");
	m_subjectLabel->setGeometry(15, 105, 170, 35);
	m_subjectLabel->setBackgroundOrigin(QWidget::ParentOrigin);
	m_subjectLabel->setAlignment(QLabel::SingleLine | QLabel::AlignLeft | QLabel::AlignTop);
	m_subjectLabel->setFont(*m_font);
	m_subjectLabel->setText("");
	m_subjectLabel->hide();

	// Define connection failed label
	m_connectionFailedLabel = new QLabel(m_image, "connectionFailedLabel");
	m_connectionFailedLabel->setGeometry(10, 50, 180, 20);
	m_connectionFailedLabel->setBackgroundOrigin(QWidget::ParentOrigin);
	m_connectionFailedLabel->setAlignment(QLabel::SingleLine | QLabel::AlignCenter);
	m_connectionFailedLabel->setFont(*m_font);
	m_connectionFailedLabel->setText(i18n("Connection failed!"));
	m_connectionFailedLabel->hide();

	// Set up popup timer
	m_closeTimer = new QTimer();
	connect(m_closeTimer, SIGNAL(timeout()), this, SLOT(slotTimeout()));

	// Set up animation timer
	m_animationTimer = new QTimer();
	connect(m_animationTimer, SIGNAL(timeout()), this, SLOT(slotProcessAnimation()));

	// Set states
	m_isAnimated = false;
	m_isUp = false;
	m_isOpening = false;

	this->updateConfiguration();
}


PopupWindow::~PopupWindow()
{
	if (m_font)
		delete m_font;
	if (m_image)
		delete m_image;
	if (m_closeButton)
		delete m_closeButton;
	if (m_newMailLabel)
		delete m_newMailLabel;
	if (m_senderLabel)
		delete m_senderLabel;
	if (m_subjectLabel)
		delete m_subjectLabel;
	if (m_connectionFailedLabel)
		delete m_connectionFailedLabel;

	if (m_closeTimer)
		delete m_closeTimer;
	if (m_animationTimer)
		delete m_animationTimer;
}


void PopupWindow::popup()
{
	this->slotOpen();
}


void PopupWindow::updateConfiguration()
{
	m_closeTimeout = Configuration().readNumEntry("PopupTimeout");
	m_animationDelay = Configuration().readNumEntry("AnimationDelay");
	m_popupDirection = Configuration().readEntry("PopupDirection");
	m_isAnimated = Configuration().readBoolEntry("PopupAnimated");
	m_hOffset = Configuration().readNumEntry("HorizontalOffset");
	m_vOffset = Configuration().readNumEntry("VerticalOffset");
	m_lockOnPrimaryScreen = Configuration().readBoolEntry("LockOnPrimaryScreen");

	this->updatePosition();
}


void PopupWindow::updatePosition()
{
	QDesktopWidget *d = QApplication::desktop();

	if (m_lockOnPrimaryScreen)
	{
		QRect rect = d->screenGeometry(d->primaryScreen());
		m_xPos = rect.width() - m_hOffset - m_width;
		m_yPos = rect.height() - m_vOffset - m_height;
	}
	else
	{
		m_xPos = d->width() - m_hOffset - m_width;
		m_yPos = d->height() - m_vOffset - m_height;
	}
}


void PopupWindow::setMessageCount(unsigned int number)
{
	if (number == 0)
	{
		m_connectionFailedLabel->hide();
		m_senderLabel->hide();
		m_subjectLabel->hide();

		m_newMailLabel->setAlignment(QLabel::SingleLine | QLabel::AlignCenter);
		m_newMailLabel->setText(i18n("No new mail"));
		m_newMailLabel->show();
	}
	else
	{
		m_connectionFailedLabel->hide();

		m_newMailLabel->setAlignment(QLabel::SingleLine | QLabel::AlignLeft | QLabel::AlignTop);
		m_newMailLabel->setText(i18n("You have new mail") + QString(" (%1)").arg(number));
		m_newMailLabel->show();
	}
}


void PopupWindow::setSender(const QString &sender)
{
	QString s = sender;
	if (s == "")
		s = i18n("(No sender)");

	m_senderLabel->setText("<b>" + i18n("From:") + "</b><br>" + s);
	m_senderLabel->show();
}


void PopupWindow::setSubject(const QString &subject)
{
	QString s = subject;
	if (s == "")
		s = i18n("(No subject)");
	
	m_subjectLabel->setText("<b>" + i18n("Subject:") + "</b><br>" + s);
	m_subjectLabel->show();
}


void PopupWindow::setConnectionFailed()
{
	m_newMailLabel->hide();
	m_senderLabel->hide();
	m_subjectLabel->hide();

	m_connectionFailedLabel->show();
}


///////////////////
// Private slots //
///////////////////
void PopupWindow::slotOpen()
{
	if (m_isUp)
		m_closeTimer->start(m_closeTimeout*1000, true);
	else
	{
		this->updatePosition();

		if (m_isAnimated)
		{
			m_isOpening = true;
			m_currentHeight = 0;
			m_currentImagePos = 0;
			this->setGeometry(m_xPos, m_yPos + m_height - m_currentHeight, m_width, m_currentHeight);
			if (m_popupDirection == "Down")
			{
				m_currentImagePos = -m_image->height();
				m_image->move(0, m_currentImagePos);
			}
			else
				m_image->move(0,0);

			m_animationTimer->start(m_animationDelay, false);
		}
		else
		{
			setGeometry(m_xPos, m_yPos, m_width, m_height);
			m_image->move(0, 0);
			this->show();
			m_closeTimer->start(m_closeTimeout*1000, true);
			m_isUp = true;
		}
	}
}


void PopupWindow::slotTimeout()
{
	// Keep the popup window open if it has mouse focus
	if (this->hasMouse())
	{
		this->m_closeTimer->stop();
		this->m_closeTimer->start(1000, true);
	}
	else
		this->slotClose();
}


void PopupWindow::slotClose()
{
	m_closeTimer->stop();

	m_isUp = false;
	if (m_isAnimated)
	{
		m_isOpening = false;
		m_animationTimer->start(m_animationDelay, false);
	}
	else
		this->hide();
}


void PopupWindow::slotProcessAnimation()
{
	if(m_isOpening)
	{
		if (m_popupDirection == "Up")
		{
			m_currentHeight += 2;
			setGeometry(m_xPos, m_yPos + m_height - m_currentHeight, m_width, m_currentHeight);
			this->show();
		}
		else
		{
			m_currentHeight += 2;
			setGeometry(m_xPos, m_yPos, m_width, m_currentHeight);
			m_currentImagePos += 2;
			m_image->move(0, m_currentImagePos);
			this->show();
		}

		if (m_currentHeight >= m_height)
		{
			m_animationTimer->stop();
			m_closeTimer->start(m_closeTimeout*1000, true);
			m_isUp = true;
		}
	}

	else
	{
		if (m_popupDirection == "Up")
		{
			m_currentHeight -= 2;
			setGeometry(m_xPos, m_yPos + m_height - m_currentHeight, m_width, m_currentHeight);
			this->show();
		}
		else
		{
			m_currentHeight -= 2;
			setGeometry(m_xPos, m_yPos, m_width, m_currentHeight);
			m_currentImagePos -= 2;
			m_image->move(0, m_currentImagePos);
			this->show();
		}

		if (m_currentHeight <= 0)
		{
			m_animationTimer->stop();
			this->hide();
		}
	}
}


void PopupWindow::slotProcessURLClicked()
{
	emit urlClicked();
	this->slotClose();
}

#include "popupwindow.moc"
