/*
 *  Wrapper base class for GstCaps
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *  API Documentation
 *  Copyright (C) 1999,2000 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000 Wim Taymans <wtay@chello.be>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_CAPS_H
#define KDE_GST_CAPS_H

#include <kde/gst//simplewrapper.h>

namespace KDE {
namespace GST {
	class Props;

/**
 * Caps is used to attach capabilities to a pad. Capabilities are made of
 * a mime-type and a set of properties. Caps can be named and chained into
 * a list, which is then a Caps on its own.
 * Caps are created with the constructor, which takes a name, a mime type and
 * a pointer to a @ref Props. The following example shows how
 * to create a GstCaps.
 * <pre>
 *    Caps *caps;
 *    Props *props;
 *
 *    props = new Props(); // properties 
 *    props->set("format", "float");
 *    props->set("layout", 5);
 *    caps = new Caps("my_caps",		// capability name
 *	              "audio/raw",		// mime type
 *		      props);
 * </pre>
 *
 * Caps are refcounted with @ref #ref() and @ref #unref().
 * Caps can be chained with the @ref #append(), @ref #prepend() and
 * @ref #chain() functions. Use @ref #getByName() to get a named caps
 * structure from a chained list.
 *
 * Before modifying a Caps, it is a good idea to make a copy if it first with
 * @ref #copyOnWrite(). This will copy the Caps if the refcount is >1.
 *
 * @short Capabilities of pads
 * @see Props
 * @see Pad
 */
	class Caps : public SimpleWrapper {
	private:
		void *reserved;
			
	public:
/**
 * Creates a new Caps that wrapps the given GstCaps.
 * You must not create a object of this class on the stack, always 
 * use new.
 * @param real the GstCaps to be wrapped.
 */
	        Caps(void *real);
	        virtual ~Caps();

/**
 * Creates a new Caps that wrapps the given object. Unlike the
 * constructor this re-used already existing objects.
 * @param real the object to be wrapped
 * @return the Caps
 */
		static Caps *wrap(void *real);

/**
 * Create a new capability with the given mime type and properties.
 * You must not create a object of this class on the stack, always 
 * use new.
 *
 * @param name the name of this capability
 * @param mime the mime type to attach to the capability
 * @param props the properties to add to this capability
 */
		Caps(const QString &name, 
		     const QString &mime, 
		     Props *props);

/**
 * Create a new capability with the given mime typeid and properties.
 * You must not create a object of this class on the stack, always 
 * use new.
 *
 * @param name the name of this capability
 * @param id the id of the mime type 
 * @param props the properties to add to this capability
 */
		Caps(const QString &name, 
		     const unsigned short id, 
		     Props *props);

/**
 * Increase the refcount of this caps structure
 *
 * @return the caps with the refcount incremented
 */
		Caps* ref();

/**
 * Decrease the refcount of this caps structure, 
 * destroying it when the refcount is 0
 *
 * @return caps or NULL if the refcount reached 0
 */
		Caps* unref();

/**
 * Copies the caps.
 *
 * @return a copy of the Caps structure.
 */
		Caps* copy();

/**
 * Copies the caps, not copying any chained caps.
 *
 * @return a copy of the Caps structure.
 */
		Caps* copy1();

/**
 * Chains the given capabilities.
 *
 * @param caps1 a capabilty
 * @param caps2 more capabilities (can be 0)
 * @param caps3 more capabilities (can be 0)
 * @param caps4 more capabilities (can be 0)
 * @param caps5 more capabilities (can be 0)
 *
 * @return the new capability
 */
		Caps* chain(Caps *caps1,
			    Caps *caps2 = 0,
			    Caps *caps3 = 0,
			    Caps *caps4 = 0,
			    Caps *caps5 = 0);

/**
 * Copies the caps if the refcount is greater than 1
 *
 * @return a pointer to a Caps strcuture that can
 * be safely written to
 */
		Caps* copyOnWrite();

/**
 * Appends a capability to the existing capability.
 *
 * @param capstoadd the capability to append
 * @return the new capability
 */
		Caps* append(Caps *capstoadd);

/**
 * Prepend the capability to the list of capabilities.
 *
 * @param capstoadd the capability to append
 * @return the new capability
 */
		Caps* prepend(Caps *capstoadd);

/**
 * Set the name of a caps.
 *
 * @param name the name to set
 */
		void setName(const QString &name);

/**
 * Get the name of a Caps structure.
 *
 * @return the name of the caps
 */
		QString getName();

/**
 * Set the type id of the caps.
 *
 * @param typeId the type id to set
 */
		void setTypeId(unsigned short typeId);

/**
 * Get the type id of the caps.
 *
 * @return the type id of the caps
 */
		unsigned short getTypeId();

/**
 * Set the mime type of the caps as a string.
 *
 * @param mime the mime type to attach to the caps
 */
		void setMime(const QString &mime);

/**
 * Get the mime type of the caps as a string.
 *
 * @return the mime type of the caps
 */
		QString getMime();

/**
 * Set the properties to the given caps.
 *
 * @param props the properties to attach
 * @return the new caps structure
 */
		Caps* setProps(Props *props);

/**
 * Get the properties of the given caps.
 *
 * @return the properties of the caps
 */
		Props* props();

/**
 * Make the normalisation of the caps. This will return a new caps
 * that is equivalent to the input caps with the exception that all
 * lists are unrolled. This function is useful when you want to iterate
 * the caps.
 *
 * @eturn The normalisation of the caps.
 */
		Caps* normalize();

/**
 * Make the intersection between two caps.
 *
 * @param caps2 a capabilty
 * @return The intersection of the two caps or NULL if the intersection
 * is empty.
 */
		Caps* intersect(Caps *caps2);

/**
 * Get the capability with the given name from this
 * chain of capabilities.
 *
 * @param name the name of the capability to get
 * @return the first capability in the chain with the 
 *         given name
 */
		Caps* getByName(const QString &name);
	};
}
}

#endif
