/*
 *  Interface/wrapper for GStreamer GstElement
 *  Copyright (C) 2002-2003 Tim Jansen <tim@tjansen.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "element.h"
#include "elementfactory.h"
#include "padtemplate.h"
#include "caps.h"
#include "bin.h"
#include "event.h"
#include "helper.h"
#include "callbackmanager.h"

#include <qevent.h>
#include <qapplication.h>

extern "C" {
#include <gst/gst.h>
}

using namespace KDE::GST;

#define r() GST_ELEMENT(m_real)

const int StateChangeEventType = 11001;
class StateChangeEvent : public QCustomEvent
{
	Element::State m_oldState, m_newState;
public:
	StateChangeEvent(Element::State oldState, 
			 Element::State newState) : 
		QCustomEvent(StateChangeEventType),
		m_oldState(oldState),
		m_newState(newState)
	{};
	Element::State newState() { return m_newState; }
	Element::State oldState() { return m_oldState; }
};

const int EosEventType = 11002;
class EosEvent : public QCustomEvent
{
public:
	EosEvent() : 
		QCustomEvent(EosEventType)
	{};
};

const int ErrorEventType = 11003;
class ErrorEvent : public QCustomEvent
{
	QString m_msg;
public:
	ErrorEvent(const char *msg) : 
		QCustomEvent(ErrorEventType),
		m_msg(msg)
	{};
	QString msg() { return m_msg; }
};


// helper factory function for classMap. See used in gstreamer.cpp
Wrapper* Element::wrapperFactory(void *real) {
	return new Element(real);
}

Element::Element(void *real) :
	Object(real) {
}
Element::~Element() {
}

Element* Element::wrap(void *real) {
	return (Element*) Object::wrap(real);
}

ElementFactory *Element::getFactory() {
	return ElementFactory::wrap(gst_element_get_factory(r()));
}

void Element::addPad(Pad *pad) {
	gst_element_add_pad(r(), GST_PAD(pad->realObject()));
}

void Element::removePad(Pad *pad) {
	gst_element_remove_pad(r(), GST_PAD(pad->realObject()));
}

Pad *Element::addGhostPad(Pad *pad, const QString &name) {
	return Pad::wrap(gst_element_add_ghost_pad(r(), 
						   GST_PAD(pad->realObject()),
						   (gchar*)name.latin1()));
}

void Element::removeGhostPad(Pad *pad) {
	gst_element_remove_ghost_pad(r(), GST_PAD(pad->realObject()));
}

Pad* Element::getPad(const QString &name) {
	return Pad::wrap(gst_element_get_pad(r(), name.latin1()));
}

Pad* Element::getStaticPad(const QString &name) {
	return Pad::wrap(gst_element_get_static_pad(r(), name.latin1()));
}

Pad* Element::getRequestPad(const QString &name) {
	return Pad::wrap(gst_element_get_pad(r(), name.latin1()));
}

void Element::releaseRequestPad(Pad *pad) {
	gst_element_release_request_pad(r(), GST_PAD(pad->realObject()));
}

QValueVector<Pad*> Element::getPadList() {
	return convertList<Pad>(gst_element_get_pad_list(r()));
}

PadTemplate* Element::getPadTemplate(const QString &name) {
	return PadTemplate::wrap(gst_element_get_pad_template(r(), 
							      name.latin1()));
}

QValueVector<PadTemplate*> Element::getPadTemplateList() {
	return convertList<PadTemplate>(gst_element_get_pad_template_list(r()));
}

void Element::classAddPadTemplate (PadTemplate *templ) {
	gst_element_class_add_pad_template(GST_ELEMENT_CLASS(G_OBJECT_CLASS(G_OBJECT(m_real))), 
					   GST_PAD_TEMPLATE(templ->realObject()));
}

bool Element::link(Element *elem1,
		   Element *elem2,
		   Element *elem3,
		   Element *elem4,
		   Element *elem5) {
	if (!elem2)
		return gst_element_link(r(), GST_ELEMENT(elem1->m_real));
	else if (!elem3)
		return gst_element_link_many(r(), 
					     GST_ELEMENT(elem1->m_real),
					     GST_ELEMENT(elem2->m_real),
					     0);
	else if (!elem4)
		return gst_element_link_many(r(), 
					     GST_ELEMENT(elem1->m_real),
					     GST_ELEMENT(elem2->m_real),
					     GST_ELEMENT(elem3->m_real),
					     0);
	else if (!elem5)
		return gst_element_link_many(r(), 
					     GST_ELEMENT(elem1->m_real),
					     GST_ELEMENT(elem2->m_real),
					     GST_ELEMENT(elem3->m_real),
					     GST_ELEMENT(elem4->m_real),
					     0);
	else
		return gst_element_link_many(r(), 
					     GST_ELEMENT(elem1->m_real),
					     GST_ELEMENT(elem2->m_real),
					     GST_ELEMENT(elem3->m_real),
					     GST_ELEMENT(elem4->m_real),
					     GST_ELEMENT(elem5->m_real),
					     0);
}

bool Element::linkFiltered(Element *dest, Caps *filtercaps) {
	return gst_element_link_filtered(r(), GST_ELEMENT(dest->m_real),
					 GST_CAPS(filtercaps->realObject()));
}

bool Element::linkPads(const QString &srcpadname,
		       Element *dest, 
		       const QString &destpadname,
		       Caps *filtercaps) {
	if (filtercaps)
		return gst_element_link_pads_filtered(r(), srcpadname.latin1(), 
						      GST_ELEMENT(dest->m_real), destpadname.latin1(),
							 GST_CAPS(filtercaps->realObject()));
	else
		return gst_element_link_pads(r(), srcpadname.latin1(),
					     GST_ELEMENT(dest->m_real), 
					     destpadname.latin1());
}

void Element::unlink(Element *elem1,
		     Element *elem2,
		     Element *elem3,
		     Element *elem4,
		     Element *elem5) {
	if (!elem2)
		gst_element_unlink(r(), GST_ELEMENT(elem1->m_real));
	else if (!elem3)
		gst_element_unlink_many(r(), 
					GST_ELEMENT(elem1->m_real),
					GST_ELEMENT(elem2->m_real),
					0);
	else if (!elem4)
		gst_element_unlink_many(r(), 
					GST_ELEMENT(elem1->m_real),
					GST_ELEMENT(elem2->m_real),
					GST_ELEMENT(elem3->m_real),
					0);
	else if (!elem5)
		gst_element_unlink_many(r(), 
					GST_ELEMENT(elem1->m_real),
					GST_ELEMENT(elem2->m_real),
					GST_ELEMENT(elem3->m_real),
					GST_ELEMENT(elem4->m_real),
					0);
	else
		gst_element_unlink_many(r(), 
					GST_ELEMENT(elem1->m_real),
					GST_ELEMENT(elem2->m_real),
					GST_ELEMENT(elem3->m_real),
					GST_ELEMENT(elem4->m_real),
					GST_ELEMENT(elem5->m_real),
					0);
}


void Element::unlinkPads(const QString &srcpadname,
			     Element *dest, const QString &destpadname) {
	return gst_element_unlink_pads(r(), srcpadname.latin1(),
				       GST_ELEMENT(dest->m_real),
				       destpadname.latin1());
}

Pad* Element::getCompatiblePad(Pad *pad) {
	return Pad::wrap(gst_element_get_compatible_pad(r(), GST_PAD(pad->realObject())));
}

Pad* Element::getCompatiblePadFiltered(Pad *pad, Caps *filtercaps) {
	return Pad::wrap(gst_element_get_compatible_pad_filtered(r(), 
								 GST_PAD(pad->realObject()),
								 GST_CAPS(filtercaps->realObject())));
}

PadTemplate* Element::getCompatiblePadTemplate(PadTemplate *compattempl) {
	return PadTemplate::wrap(gst_element_get_compatible_pad_template(r(), 
									 GST_PAD_TEMPLATE(compattempl->realObject())));
}

int Element::setState(State state) {
	return gst_element_set_state(r(), (GstElementState)state);
}

Element::State Element::getState() {
	return (Element::State) gst_element_get_state(r());
}

QString Element::stateGetName(State state) {
	return QString(gst_element_state_get_name((GstElementState)state));
}

void Element::waitStateChange() {
	gst_element_wait_state_change(r());
}

void Element::setEos() {
	gst_element_set_eos(r());
}

bool Element::interrupt() {
	return gst_element_interrupt(r());
}

void Element::yield() {
	return gst_element_yield(r());
}

bool Element::releaseLocks() {
	return gst_element_release_locks(r());
}

Clock* Element::getClock() {
	return Clock::wrap(gst_element_get_clock(r()));
}

void Element::setClock(Clock *clock) {
	gst_element_set_clock(r(), GST_CLOCK(clock->realObject()));
}

Clock::ClockReturn Element::clockWait(Clock::ID id, 
				      long long *jitter) {
	return (Clock::ClockReturn) gst_element_clock_wait(r(), 
							   id, 
							   jitter);
}

Bin* Element::getManagingBin() {
	return Bin::wrap(gst_element_get_managing_bin(r()));
}

bool Element::query(QueryType type,
		    Format *format, long long *value) {
	return gst_element_query(r(), 
				 (GstQueryType) type, 
				 (GstFormat*)format, 
				 value);
}

bool Element::sendEvent(Event *event) {
	return gst_element_send_event(r(), 
				      GST_EVENT(event->realObject()));
}

namespace {
	void eosCallback(GstElement *e, CallbackData *cd) {
		void (*callback)(Element*,void *) = 
			(void (*)(Element*,void*)) cd->origCallback;
		(*callback)(Element::wrap(e), cd->cookie);
	}
	void errorCallback(GstElement *e, 
			   GObject *o,
			   const char *msg,
			   CallbackData* cd) {
		void (*callback)(Element*,Object*,const char*,void *) = 
			(void (*)(Element*,Object*,const char*,void*)) cd->origCallback;
		(*callback)(Element::wrap(e), 
			    Object::wrap(o), 
			    msg, 
			    cd->cookie);
	}
	void stateChangeCallback(GstElement *e,
				 gint arg1,
				 gint arg2,
				 CallbackData *cd) {
		void (*callback)(Element*,int,int,void *) = 
			(void (*)(Element*,int,int,void*)) cd->origCallback;
		(*callback)(Element::wrap(e), arg1, arg2, cd->cookie);
	}
}

void Element::registerEosCallback(void (*callback)(Element *e, 
						   void *cookie),
				  void *cookie) {
	CallbackData *cd = 
		getCBManager()->registerWrappedCallback("eos",
							(void*)eosCallback, 
							(void*)callback, 
							cookie);
	registerRawSignal("eos", cd->wrapCallback, cd);
}

void Element::unregisterEosCallback(void (*callback)(Element *e, 
						     void *cookie),
				    void *cookie) {
	CallbackData *cd = 
		getCBManager()->findWrappedCallback("eos",
						    (void*)eosCallback,
						    (void*)callback, 
						    cookie);
	unregisterRawSignal(cd->name, cd->wrapCallback, cd);
	getCBManager()->removeWrappedCallback(cd);
}

void Element::registerStateChangeCallback(void (*callback)(Element *e, 
							   int newState,
							   int arg,
							   void *cookie),
					  void *cookie) {
	CallbackData *cd = 
		getCBManager()->registerWrappedCallback("state-change",
							(void*)stateChangeCallback, 
							(void*)callback, 
							cookie);
	registerRawSignal("state-change", cd->wrapCallback, cd);
}

void Element::unregisterStateChangeCallback(void (*callback)(Element *e, 
							     int newState,
							     int arg,
							     void *cookie),
					    void *cookie) {
	CallbackData *cd = 
		getCBManager()->findWrappedCallback("state-change",
						    (void*)stateChangeCallback,
						    (void*)callback, 
						    cookie);
	unregisterRawSignal(cd->name, cd->wrapCallback, cd);
	getCBManager()->removeWrappedCallback(cd);
}

void Element::registerErrorCallback(void (*callback)(Element *e, 
						     const char *error,
						     void *cookie),
				    void *cookie) {
	CallbackData *cd = 
		getCBManager()->registerWrappedCallback("error",
							(void*)errorCallback, 
							(void*)callback, 
							cookie);
	registerRawSignal("error", cd->wrapCallback, cd);
}

void Element::unregisterErrorCallback(void (*callback)(Element *e, 
						       const char *error,
						       void *cookie),
				      void *cookie) {
	CallbackData *cd = 
		getCBManager()->findWrappedCallback("error",
						    (void*)errorCallback,
						    (void*)callback, 
						    cookie);
	unregisterRawSignal(cd->name, cd->wrapCallback, cd);
	getCBManager()->removeWrappedCallback(cd);
}

namespace {
	void asyncEosCallback(Element *, void *self) {
		QApplication::postEvent((Element*)self, new EosEvent());
	}
	void asyncErrorCallback(Element*, 
				const char *msg,
			        void *self) {
		QApplication::postEvent((Element*)self, new ErrorEvent(msg));
	}
	void asyncStateChangeCallback(Element *,
				      int oldS,
				      int newS,
				      void *self) {
		QApplication::postEvent((Element*)self, 
				   new StateChangeEvent(
					       (Element::State) oldS,
					       (Element::State) newS));
	}
}

// only register callbacks on first connect
void Element::connectNotify(const char *signal) {
	QString s(signal);

	if (s == SIGNAL(eos()))
		registerEosCallback(asyncEosCallback, this);
	else if (s == SIGNAL(stateChange(KDE::GST::Element::State,KDE::GST::Element::State)))
		registerStateChangeCallback(asyncStateChangeCallback, 
					     this);
	else if (s == SIGNAL(error(QString)))
		registerErrorCallback(asyncErrorCallback, 
				      this);
	else 
		qDebug("Unknown connectNotify %s", signal);
}

void Element::customEvent(QCustomEvent *e) {
	if (e->type() == EosEventType)
		emit eos();
	else if (e->type() == StateChangeEventType) {
		StateChangeEvent *sce = (StateChangeEvent*) e;
		emit stateChange(sce->oldState(), sce->newState());
	}
	else if (e->type() == ErrorEventType) {
		ErrorEvent *ee = (ErrorEvent*) e;
		emit error(ee->msg());
	}
}

#include "element.moc"

