/*
 *  Interface/wrapper for GStreamer GstPad
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "pad.h"
#include "padtemplate.h"
#include "caps.h"
#include "helper.h"
#include "element.h"
#include "event.h"

extern "C" {
#include <gst/gst.h>
}

using namespace KDE::GST;

#define r() GST_PAD(m_real)

// helper factory function for classMap. See used in gstreamer.cpp
Wrapper* Pad::wrapperFactory(void *real) {
	return new Pad(real);
}

Pad::Pad(void *real) :
	Object(real) {
}
Pad::~Pad() {
}

Pad* Pad::wrap(void *real) {
	return (Pad*) Object::wrap(real);
}

Pad::Pad(const QString &name, Direction direction) :
	Object(0) {
	m_real = gst_pad_new(name.latin1(), (GstPadDirection)direction);
	registerReal();
}

Pad::Pad(PadTemplate *templ, const QString &name) :
	Object(0) {
	m_real = gst_pad_new_from_template(GST_PAD_TEMPLATE(templ->realObject()),
					   name.latin1());
	registerReal();
}

Pad::Direction Pad::getDirection() {
	return (Direction) gst_pad_get_direction(r());
}

bool Pad::trySetCaps(Caps *caps) {
	return gst_pad_try_set_caps(r(), GST_CAPS(caps->realObject()));
}

Caps* Pad::getCaps() {
	return Caps::wrap(gst_pad_get_caps(r()));
}

bool Pad::checkCompatibility(Pad *sinkpad) {
	return gst_pad_check_compatibility(r(), 
					   GST_PAD(sinkpad->realObject()));
}

Element* Pad::getRealParent() {
	return Element::wrap(gst_pad_get_real_parent(r()));
}

void Pad::addGhostPad(Pad *ghostpad) {
	gst_pad_add_ghost_pad(r(), GST_PAD(ghostpad->m_real));
}

void Pad::removeGhostPad(Pad *ghostpad) {
	gst_pad_remove_ghost_pad(r(), GST_PAD(ghostpad->m_real));
}

QValueVector<Pad*> Pad::getGhostPadList() {
	return convertList<Pad>(gst_pad_get_ghost_pad_list(r()));
}

Pad* Pad::getPeer() {
	return wrap(gst_pad_get_peer(r()));
}

bool Pad::link(Pad *sinkpad) {
	return gst_pad_link(r(), GST_PAD(sinkpad->m_real));
}

bool Pad::linkFiltered(Pad *sinkpad, Caps *filtercaps) {
	return gst_pad_link_filtered(r(), 
				     GST_PAD(sinkpad->m_real),
				     GST_CAPS(filtercaps->realObject()));
}

bool Pad::relinkFiltered(Pad *sinkpad, Caps *filtercaps) {
	return gst_pad_relink_filtered(r(), 
				       GST_PAD(sinkpad->m_real),
				       GST_CAPS(filtercaps->realObject()));
}

bool Pad::tryRelinkFiltered(Pad *sinkpad, Caps *filtercaps) {
	return gst_pad_try_relink_filtered(r(), 
					   GST_PAD(sinkpad->m_real),
					   GST_CAPS(filtercaps->realObject()));
}

void Pad::unlink(Pad *sinkpad) {
	gst_pad_unlink(r(), GST_PAD(sinkpad->m_real));
}

bool Pad::canLink(Pad *sinkpad) {
	return gst_pad_can_link(r(), 
				GST_PAD(sinkpad->m_real));
}

PadTemplate* Pad::getPadTemplate() {
	return PadTemplate::wrap(gst_pad_get_pad_template(r()));
}

Caps* Pad::getPadTemplateCaps() {
	return Caps::wrap(gst_pad_get_pad_template_caps(r()));
}

Caps* Pad::getAllowedCaps() {
	return Caps::wrap(gst_pad_get_allowed_caps(r()));
}

bool Pad::recalcAllowedCaps() {
	return gst_pad_recalc_allowed_caps(r());
}

bool Pad::performNegotiate(Pad *sinkpad) {
	return gst_pad_perform_negotiate(r(), 
					 GST_PAD(sinkpad->m_real));
}

bool Pad::sendEvent(Event *event) {
	return gst_pad_send_event(r(),
				  GST_EVENT(event->realObject()));
}

bool Pad::queryDefault(QueryType type,
		       Format *format,  long long *value){
	return gst_pad_query_default(r(),
				     (GstQueryType)type,
				     (GstFormat*)format,
				     value);
}

bool Pad::query(QueryType type,
		Format *format, long long *value) {
	return gst_pad_query(r(),
			     (GstQueryType)type,
			     (GstFormat*)format,
			     value);
}


#include "pad.moc"

